"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCategorizationRoutes = registerCategorizationRoutes;
var _server = require("@kbn/data-plugin/server");
var _language_models = require("@kbn/langchain/server/language_models");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("../../common");
var _constants = require("../constants");
var _categorization = require("../graphs/categorization");
var _route_validation = require("../util/route_validation");
var _with_availability = require("./with_availability");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerCategorizationRoutes(router) {
  router.versioned.post({
    path: _common.CATEGORIZATION_GRAPH_PATH,
    access: 'internal',
    options: {
      timeout: {
        idleSocket: _constants.ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_common.CategorizationRequestBody)
      }
    }
  }, (0, _with_availability.withAvailability)(async (context, req, res) => {
    const {
      packageName,
      dataStreamName,
      rawSamples,
      currentPipeline,
      langSmithOptions
    } = req.body;
    const services = await context.resolve(['core']);
    const {
      client
    } = services.core.elasticsearch;
    const {
      getStartServices,
      logger
    } = await context.integrationAssistant;
    const [, {
      actions: actionsPlugin
    }] = await getStartServices();
    try {
      var _connector$config, _langSmithOptions$pro;
      const actionsClient = await actionsPlugin.getActionsClientWithRequest(req);
      const connector = req.body.connectorId ? await actionsClient.get({
        id: req.body.connectorId
      }) : (await actionsClient.getAll()).filter(connectorItem => connectorItem.actionTypeId === '.bedrock')[0];
      const abortSignal = (0, _server.getRequestAbortedSignal)(req.events.aborted$);
      const isOpenAI = connector.actionTypeId === '.gen-ai';
      const llmClass = isOpenAI ? _language_models.ActionsClientChatOpenAI : _language_models.ActionsClientSimpleChatModel;
      const model = new llmClass({
        actionsClient,
        connectorId: connector.id,
        logger,
        llmType: isOpenAI ? 'openai' : 'bedrock',
        model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel,
        temperature: 0.05,
        maxTokens: 4096,
        signal: abortSignal,
        streaming: false
      });
      const parameters = {
        packageName,
        dataStreamName,
        rawSamples,
        currentPipeline
      };
      const options = {
        callbacks: [new _apm.APMTracer({
          projectName: (_langSmithOptions$pro = langSmithOptions === null || langSmithOptions === void 0 ? void 0 : langSmithOptions.projectName) !== null && _langSmithOptions$pro !== void 0 ? _langSmithOptions$pro : 'default'
        }, logger), ...(0, _langsmith.getLangSmithTracer)({
          ...langSmithOptions,
          logger
        })]
      };
      const graph = await (0, _categorization.getCategorizationGraph)(client, model);
      const results = await graph.invoke(parameters, options);
      return res.ok({
        body: _common.CategorizationResponse.parse(results)
      });
    } catch (e) {
      return res.badRequest({
        body: e
      });
    }
  }));
}