"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildPackage = buildPackage;
var _admZip = _interopRequireDefault(require("adm-zip"));
var _nunjucks = _interopRequireDefault(require("nunjucks"));
var _utils = require("@kbn/utils");
var _path = require("path");
var _util = require("../util");
var _agent = require("./agent");
var _data_stream = require("./data_stream");
var _fields = require("./fields");
var _pipeline = require("./pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const initialVersion = '1.0.0';
async function buildPackage(integration) {
  const templateDir = (0, _path.join)(__dirname, '../templates');
  const agentTemplates = (0, _path.join)(templateDir, 'agent');
  const manifestTemplates = (0, _path.join)(templateDir, 'manifest');
  const systemTestTemplates = (0, _path.join)(templateDir, 'system_tests');
  _nunjucks.default.configure([templateDir, agentTemplates, manifestTemplates, systemTestTemplates], {
    autoescape: false
  });
  const workingDir = (0, _path.join)((0, _utils.getDataPath)(), `integration-assistant-${(0, _util.generateUniqueId)()}`);
  const packageDirectoryName = `${integration.name}-${initialVersion}`;
  const packageDir = createDirectories(workingDir, integration, packageDirectoryName);
  const dataStreamsDir = (0, _path.join)(packageDir, 'data_stream');
  for (const dataStream of integration.dataStreams) {
    const dataStreamName = dataStream.name;
    const specificDataStreamDir = (0, _path.join)(dataStreamsDir, dataStreamName);
    (0, _data_stream.createDataStream)(integration.name, specificDataStreamDir, dataStream);
    (0, _agent.createAgentInput)(specificDataStreamDir, dataStream.inputTypes);
    (0, _pipeline.createPipeline)(specificDataStreamDir, dataStream.pipeline);
    (0, _fields.createFieldMapping)(integration.name, dataStreamName, specificDataStreamDir, dataStream.docs);
  }
  const zipBuffer = await createZipArchive(workingDir, packageDirectoryName);
  (0, _util.removeDirSync)(workingDir);
  return zipBuffer;
}
function createDirectories(workingDir, integration, packageDirectoryName) {
  const packageDir = (0, _path.join)(workingDir, packageDirectoryName);
  (0, _util.ensureDirSync)(workingDir);
  (0, _util.ensureDirSync)(packageDir);
  createPackage(packageDir, integration);
  return packageDir;
}
function createPackage(packageDir, integration) {
  createReadme(packageDir, integration);
  createChangelog(packageDir);
  createBuildFile(packageDir);
  createPackageManifest(packageDir, integration);
  //  Skipping creation of system tests temporarily for custom package generation
  //  createPackageSystemTests(packageDir, integration);
  if ((integration === null || integration === void 0 ? void 0 : integration.logo) !== undefined) {
    createLogo(packageDir, integration.logo);
  }
}
function createLogo(packageDir, logo) {
  const logoDir = (0, _path.join)(packageDir, 'img');
  (0, _util.ensureDirSync)(logoDir);
  const buffer = Buffer.from(logo, 'base64');
  (0, _util.createSync)((0, _path.join)(logoDir, 'logo.svg'), buffer);
}
function createBuildFile(packageDir) {
  const buildFile = _nunjucks.default.render('build.yml.njk', {
    ecs_version: '8.11.0'
  });
  const buildDir = (0, _path.join)(packageDir, '_dev/build');
  (0, _util.ensureDirSync)(buildDir);
  (0, _util.createSync)((0, _path.join)(buildDir, 'build.yml'), buildFile);
}
function createChangelog(packageDir) {
  const changelogTemplate = _nunjucks.default.render('changelog.yml.njk', {
    initial_version: initialVersion
  });
  (0, _util.createSync)((0, _path.join)(packageDir, 'changelog.yml'), changelogTemplate);
}
function createReadme(packageDir, integration) {
  const readmeDirPath = (0, _path.join)(packageDir, '_dev/build/docs/');
  (0, _util.ensureDirSync)(readmeDirPath);
  const readmeTemplate = _nunjucks.default.render('package_readme.md.njk', {
    package_name: integration.name,
    data_streams: integration.dataStreams
  });
  (0, _util.createSync)((0, _path.join)(readmeDirPath, 'README.md'), readmeTemplate);
}
async function createZipArchive(workingDir, packageDirectoryName) {
  const tmpPackageDir = (0, _path.join)(workingDir, packageDirectoryName);
  const zip = new _admZip.default();
  zip.addLocalFolder(tmpPackageDir, packageDirectoryName);
  const buffer = zip.toBuffer();
  return buffer;
}
function createPackageManifest(packageDir, integration) {
  const uniqueInputs = {};
  integration.dataStreams.forEach(dataStream => {
    dataStream.inputTypes.forEach(inputType => {
      if (!uniqueInputs[inputType]) {
        uniqueInputs[inputType] = {
          type: inputType,
          title: dataStream.title,
          description: dataStream.description
        };
      }
    });
  });
  const uniqueInputsList = Object.values(uniqueInputs);
  const packageManifest = _nunjucks.default.render('package_manifest.yml.njk', {
    format_version: '3.1.4',
    package_title: integration.title,
    package_name: integration.name,
    package_version: initialVersion,
    package_description: integration.description,
    package_logo: integration.logo,
    package_owner: '@elastic/custom-integrations',
    min_version: '^8.13.0',
    inputs: uniqueInputsList
  });
  (0, _util.createSync)((0, _path.join)(packageDir, 'manifest.yml'), packageManifest);
}