"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findInvalidEcsFields = findInvalidEcsFields;
exports.handleValidateMappings = handleValidateMappings;
exports.processMapping = processMapping;
var _ecs = require("../../../common/ecs");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/* eslint-disable @typescript-eslint/no-explicit-any */

const valueFieldKeys = new Set(['target', 'confidence', 'date_formats', 'type']);
function extractKeys(data, prefix = '') {
  const keys = new Set();
  for (const [key, value] of Object.entries(data)) {
    const fullKey = prefix ? `${prefix}.${key}` : key;
    if (Array.isArray(value)) {
      // Directly add the key for arrays without iterating over elements
      keys.add(fullKey);
    } else if (typeof value === 'object' && value !== null) {
      const valueKeys = new Set(Object.keys(value));
      if ([...valueFieldKeys].every(k => valueKeys.has(k))) {
        keys.add(fullKey);
      } else {
        // Recursively extract keys if the current value is a nested object
        for (const nestedKey of extractKeys(value, fullKey)) {
          keys.add(nestedKey);
        }
      }
    } else {
      // Add the key if the value is not an object or is null
      keys.add(fullKey);
    }
  }
  return keys;
}
function findMissingFields(formattedSamples, ecsMapping) {
  const combinedSamples = JSON.parse(formattedSamples);
  const uniqueKeysFromSamples = extractKeys(combinedSamples);
  const ecsResponseKeys = extractKeys(ecsMapping);
  const missingKeys = [...uniqueKeysFromSamples].filter(key => !ecsResponseKeys.has(key));
  return missingKeys;
}
function processMapping(path, value, output) {
  if (typeof value === 'object' && value !== null) {
    if (!Array.isArray(value)) {
      // If the value is a dict with all the keys returned for each source field, this is the full path of the field.
      const valueKeys = new Set(Object.keys(value));
      if ([...valueFieldKeys].every(k => valueKeys.has(k))) {
        if ((value === null || value === void 0 ? void 0 : value.target) !== null) {
          if (!output[value === null || value === void 0 ? void 0 : value.target]) {
            output[value.target] = [];
          }
          output[value.target].push(path);
        }
      } else {
        // Regular dictionary, continue traversing
        for (const [k, v] of Object.entries(value)) {
          processMapping([...path, k], v, output);
        }
      }
    } else {
      // If the value is an array, iterate through items and process them
      for (const item of value) {
        if (typeof item === 'object' && item !== null) {
          processMapping(path, item, output);
        }
      }
    }
  } else if (value !== null) {
    // Direct value, accumulate path
    if (!output[value]) {
      output[value] = [];
    }
    output[value].push(path);
  }
}
function getValueFromPath(obj, path) {
  return path.reduce((acc, key) => acc && acc[key] !== undefined ? acc[key] : null, obj);
}
function findDuplicateFields(samples, ecsMapping) {
  const parsedSamples = samples.map(sample => JSON.parse(sample));
  const results = [];
  const output = {};

  // Get all keys for each target ECS mapping field
  processMapping([], ecsMapping, output);

  // Filter out any ECS field that does not have multiple source fields mapped to it
  const filteredOutput = Object.fromEntries(Object.entries(output).filter(([_, paths]) => paths.length > 1 && _ !== null));

  // For each ECS field where value is the ECS field and paths is the array of source field names
  for (const [value, paths] of Object.entries(filteredOutput)) {
    // For each log sample, checking if more than 1 source field exists in the same sample
    for (const sample of parsedSamples) {
      const foundPaths = paths.filter(path => getValueFromPath(sample, path) !== null);
      if (foundPaths.length > 1) {
        const matchingFields = foundPaths.map(p => p.join('.'));
        results.push(`One or more samples have matching fields for ECS field '${value}': ${matchingFields.join(', ')}`);
        break;
      }
    }
  }
  return results;
}

// Function to find invalid ECS fields
function findInvalidEcsFields(ecsMapping) {
  const results = [];
  const output = {};
  const ecsDict = _ecs.ECS_FULL;
  const ecsReserved = _constants.ECS_RESERVED;
  processMapping([], ecsMapping, output);
  const filteredOutput = Object.fromEntries(Object.entries(output).filter(([key, _]) => key !== null));
  for (const [ecsValue, paths] of Object.entries(filteredOutput)) {
    if (!Object.prototype.hasOwnProperty.call(ecsDict, ecsValue)) {
      const field = paths.map(p => p.join('.'));
      results.push(`Invalid ECS field mapping identified for ${ecsValue} : ${field.join(', ')}`);
    }
    if (ecsReserved.includes(ecsValue)) {
      const field = paths.map(p => p.join('.'));
      results.push(`Reserved ECS field mapping identified for ${ecsValue} : ${field.join(', ')}`);
    }
  }
  return results;
}
function handleValidateMappings(state) {
  const missingKeys = findMissingFields(state === null || state === void 0 ? void 0 : state.formattedSamples, state === null || state === void 0 ? void 0 : state.currentMapping);
  const duplicateFields = findDuplicateFields(state === null || state === void 0 ? void 0 : state.samples, state === null || state === void 0 ? void 0 : state.currentMapping);
  const invalidEcsFields = findInvalidEcsFields(state === null || state === void 0 ? void 0 : state.currentMapping);
  return {
    missingKeys,
    duplicateFields,
    invalidEcsFields,
    lastExecutedChain: 'validateMappings'
  };
}