"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleCategorizationValidation = handleCategorizationValidation;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function handleCategorizationValidation(state) {
  const errors = [];
  const pipelineResults = state.pipelineResults;

  // Loops through the pipeline results to find invalid categories and types
  for (const doc of pipelineResults) {
    var _doc$event, _doc$event2;
    let types = [];
    let categories = [];
    if (doc !== null && doc !== void 0 && (_doc$event = doc.event) !== null && _doc$event !== void 0 && _doc$event.type) {
      types = doc.event.type;
    }
    if (doc !== null && doc !== void 0 && (_doc$event2 = doc.event) !== null && _doc$event2 !== void 0 && _doc$event2.category) {
      categories = doc.event.category;
    }
    const invalidCategories = findInvalidCategories(categories);
    const invalidTypes = findInvalidTypes(types);
    if (invalidCategories.length > 0) {
      errors.push(createErrorMessage('event.category', invalidCategories, _constants.EVENT_CATEGORIES));
    }
    if (invalidTypes.length > 0) {
      errors.push(createErrorMessage('event.type', invalidTypes, _constants.EVENT_TYPES));
    }

    // Compatibility check is done only on valid categories and types
    const validCategories = categories.filter(x => !invalidCategories.includes(x));
    const validTypes = types.filter(x => !invalidTypes.includes(x));
    const compatibleErrors = getTypeCategoryIncompatibleError(validCategories, validTypes);
    for (const ce of compatibleErrors) {
      errors.push(ce);
    }
  }
  return {
    invalidCategorization: errors,
    lastExecutedChain: 'handleCategorizationValidation'
  };
}
function createErrorMessage(field, errorList, allowedValues) {
  return {
    error: `field ${field}'s values (${errorList.join(', ')}) is not one of the allowed values (${allowedValues.join(', ')})`
  };
}
function findInvalidCategories(categories) {
  const invalidCategories = [];
  for (const c of categories) {
    if (!_constants.EVENT_CATEGORIES.includes(c)) {
      invalidCategories.push(c);
    }
  }
  return invalidCategories;
}
function findInvalidTypes(types) {
  const invalidTypes = [];
  for (const t of types) {
    if (!_constants.EVENT_TYPES.includes(t)) {
      invalidTypes.push(t);
    }
  }
  return invalidTypes;
}
function getTypeCategoryIncompatibleError(categories, types) {
  const errors = [];
  let unmatchedTypes = new Set(types);
  const matchCategories = new Set(categories);
  let categoryExists = false;
  for (const c of matchCategories) {
    if (c in _constants.ECS_EVENT_TYPES_PER_CATEGORY) {
      categoryExists = true;
      const matchTypes = new Set(_constants.ECS_EVENT_TYPES_PER_CATEGORY[c]);
      unmatchedTypes = new Set([...unmatchedTypes].filter(x => !matchTypes.has(x)));
    }
  }
  if (categoryExists && unmatchedTypes.size > 0) {
    errors.push({
      error: `event.type (${[...unmatchedTypes].join(', ')}) not compatible with any of the event.category (${[...matchCategories].join(', ')})`
    });
  }
  return errors;
}