"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGeneration = exports.GenerationModal = void 0;
var _eui = require("@elastic/eui");
var _fp = require("lodash/fp");
var _react = _interopRequireWildcard(require("react"));
var _react2 = require("@emotion/react");
var _lang_smith = require("../../../../../common/lib/lang_smith");
var _api = require("../../../../../common/lib/api");
var _use_kibana = require("../../../../../common/hooks/use_kibana");
var i18n = _interopRequireWildcard(require("./translations"));
var _telemetry = require("../../../telemetry");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ProgressOrder = ['ecs', 'categorization', 'related'];
const progressText = {
  ecs: i18n.PROGRESS_ECS_MAPPING,
  categorization: i18n.PROGRESS_CATEGORIZATION,
  related: i18n.PROGRESS_RELATED_GRAPH
};
const useGeneration = ({
  integrationSettings,
  connector,
  onComplete
}) => {
  const {
    reportGenerationComplete
  } = (0, _telemetry.useTelemetry)();
  const {
    http,
    notifications
  } = (0, _use_kibana.useKibana)().services;
  const [progress, setProgress] = (0, _react.useState)();
  const [error, setError] = (0, _react.useState)(null);
  const [isRequesting, setIsRequesting] = (0, _react.useState)(true);
  (0, _react.useEffect)(() => {
    if (!isRequesting || http == null || connector == null || integrationSettings == null || (notifications === null || notifications === void 0 ? void 0 : notifications.toasts) == null) {
      return;
    }
    const generationStartedAt = Date.now();
    const abortController = new AbortController();
    const deps = {
      http,
      abortSignal: abortController.signal
    };
    (async () => {
      try {
        var _integrationSettings$, _integrationSettings$2, _integrationSettings$3;
        const ecsRequest = {
          packageName: (_integrationSettings$ = integrationSettings.name) !== null && _integrationSettings$ !== void 0 ? _integrationSettings$ : '',
          dataStreamName: (_integrationSettings$2 = integrationSettings.dataStreamName) !== null && _integrationSettings$2 !== void 0 ? _integrationSettings$2 : '',
          rawSamples: (_integrationSettings$3 = integrationSettings.logsSampleParsed) !== null && _integrationSettings$3 !== void 0 ? _integrationSettings$3 : [],
          connectorId: connector.id,
          langSmithOptions: (0, _lang_smith.getLangSmithOptions)()
        };
        setProgress('ecs');
        const ecsGraphResult = await (0, _api.runEcsGraph)(ecsRequest, deps);
        if (abortController.signal.aborted) return;
        if ((0, _fp.isEmpty)(ecsGraphResult === null || ecsGraphResult === void 0 ? void 0 : ecsGraphResult.results)) {
          setError('No results from ECS graph');
          return;
        }
        const categorizationRequest = {
          ...ecsRequest,
          currentPipeline: ecsGraphResult.results.pipeline
        };
        setProgress('categorization');
        const categorizationResult = await (0, _api.runCategorizationGraph)(categorizationRequest, deps);
        if (abortController.signal.aborted) return;
        const relatedRequest = {
          ...categorizationRequest,
          currentPipeline: categorizationResult.results.pipeline
        };
        setProgress('related');
        const relatedGraphResult = await (0, _api.runRelatedGraph)(relatedRequest, deps);
        if (abortController.signal.aborted) return;
        if ((0, _fp.isEmpty)(relatedGraphResult === null || relatedGraphResult === void 0 ? void 0 : relatedGraphResult.results)) {
          throw new Error('Results not found in response');
        }
        reportGenerationComplete({
          connector,
          integrationSettings,
          durationMs: Date.now() - generationStartedAt
        });
        onComplete(relatedGraphResult.results);
      } catch (e) {
        if (abortController.signal.aborted) return;
        const errorMessage = `${e.message}${e.body ? ` (${e.body.statusCode}): ${e.body.message}` : ''}`;
        reportGenerationComplete({
          connector,
          integrationSettings,
          durationMs: Date.now() - generationStartedAt,
          error: errorMessage
        });
        setError(errorMessage);
      } finally {
        setIsRequesting(false);
      }
    })();
    return () => {
      abortController.abort();
    };
  }, [isRequesting, onComplete, setProgress, connector, http, integrationSettings, reportGenerationComplete, notifications === null || notifications === void 0 ? void 0 : notifications.toasts]);
  const retry = (0, _react.useCallback)(() => {
    setError(null);
    setIsRequesting(true);
  }, []);
  return {
    progress,
    error,
    retry
  };
};
exports.useGeneration = useGeneration;
const useModalCss = () => {
  const {
    euiTheme
  } = (0, _eui.useEuiTheme)();
  return {
    headerCss: (0, _react2.css)`
      justify-content: center;
      margin-top: ${euiTheme.size.m};
    `,
    bodyCss: (0, _react2.css)`
      padding: ${euiTheme.size.xxxxl};
      min-width: 600px;
    `
  };
};
const GenerationModal = exports.GenerationModal = /*#__PURE__*/_react.default.memo(({
  integrationSettings,
  connector,
  onComplete,
  onClose
}) => {
  const {
    headerCss,
    bodyCss
  } = useModalCss();
  const {
    progress,
    error,
    retry
  } = useGeneration({
    integrationSettings,
    connector,
    onComplete
  });
  const progressValue = (0, _react.useMemo)(() => progress ? ProgressOrder.indexOf(progress) + 1 : 0, [progress]);
  return /*#__PURE__*/_react.default.createElement(_eui.EuiModal, {
    onClose: onClose,
    "data-test-subj": "generationModal"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiModalHeader, {
    css: headerCss
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiModalHeaderTitle, null, i18n.ANALYZING)), /*#__PURE__*/_react.default.createElement(_eui.EuiModalBody, {
    css: bodyCss
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    gutterSize: "l",
    justifyContent: "center"
  }, progress && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, error ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    title: i18n.GENERATION_ERROR(progressText[progress]),
    color: "danger",
    iconType: "alert",
    "data-test-subj": "generationErrorCallout"
  }, error)) : /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "row",
    gutterSize: "s",
    alignItems: "center",
    justifyContent: "center"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, {
    size: "s"
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
    size: "xs",
    color: "subdued"
  }, progressText[progress])))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiProgress, {
    value: progressValue,
    max: 4,
    color: "primary",
    size: "m"
  })))))), /*#__PURE__*/_react.default.createElement(_eui.EuiModalFooter, null, error ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    justifyContent: "center"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
    iconType: "refresh",
    onClick: retry,
    "data-test-subj": "retryButton"
  }, i18n.RETRY))) : /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "xl"
  })));
});
GenerationModal.displayName = 'GenerationModal';