"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RawSamples = exports.Pipeline = exports.PackageName = exports.Mapping = exports.LangSmithOptions = exports.Integration = exports.InputTypeEnum = exports.InputType = exports.Docs = exports.DataStreamName = exports.DataStream = exports.Connector = void 0;
var _zod = require("zod");
var _processor_attributes = require("./processor_attributes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Package name for the integration to be built.
 */

const PackageName = exports.PackageName = _zod.z.string().min(1);

/**
 * DataStream name for the integration to be built.
 */

const DataStreamName = exports.DataStreamName = _zod.z.string().min(1);

/**
 * String array containing the json raw samples that are used for ecs mapping.
 */

const RawSamples = exports.RawSamples = _zod.z.array(_zod.z.string());

/**
 * mapping object to ECS Mapping Request.
 */

const Mapping = exports.Mapping = _zod.z.object({}).passthrough();

/**
 * LLM Connector to be used in each API request.
 */

const Connector = exports.Connector = _zod.z.string();

/**
 * An array of processed documents.
 */

const Docs = exports.Docs = _zod.z.array(_zod.z.object({}).passthrough());

/**
 * The pipeline object.
 */

const Pipeline = exports.Pipeline = _zod.z.object({
  /**
   * The name of the pipeline.
   */
  name: _zod.z.string().optional(),
  /**
   * The description of the pipeline.
   */
  description: _zod.z.string().optional(),
  /**
   * The version of the pipeline.
   */
  version: _zod.z.number().int().optional(),
  /**
   * The processors to execute.
   */
  processors: _zod.z.array(_processor_attributes.ESProcessorItem),
  /**
   * The processors to execute if the pipeline fails.
   */
  on_failure: _zod.z.array(_processor_attributes.ESProcessorItem).optional()
});

/**
 * The input type for the dataStream to pull logs from.
 */

const InputType = exports.InputType = _zod.z.enum(['aws-cloudwatch', 'aws-s3', 'azure-blob-storage', 'azure-eventhub', 'cel', 'cloudfoundry', 'filestream', 'gcp-pubsub', 'gcs', 'http-endpoint', 'journald', 'kafka', 'tcp', 'udp']);
const InputTypeEnum = exports.InputTypeEnum = InputType.enum;

/**
 * The dataStream object.
 */

const DataStream = exports.DataStream = _zod.z.object({
  /**
   * The name of the dataStream.
   */
  name: _zod.z.string(),
  /**
   * The title of the dataStream.
   */
  title: _zod.z.string(),
  /**
   * The description of the dataStream.
   */
  description: _zod.z.string(),
  /**
   * The input types of the dataStream.
   */
  inputTypes: _zod.z.array(InputType),
  /**
   * The raw samples of the dataStream.
   */
  rawSamples: RawSamples,
  /**
   * The pipeline of the dataStream.
   */
  pipeline: Pipeline,
  /**
   * The documents of the dataStream.
   */
  docs: Docs
});

/**
 * The integration object.
 */

const Integration = exports.Integration = _zod.z.object({
  /**
   * The name of the integration.
   */
  name: _zod.z.string(),
  /**
   * The title of the integration.
   */
  title: _zod.z.string(),
  /**
   * The description of the integration.
   */
  description: _zod.z.string(),
  /**
   * The dataStreams of the integration.
   */
  dataStreams: _zod.z.array(DataStream),
  /**
   * The logo of the integration.
   */
  logo: _zod.z.string().optional()
});

/**
 * The LangSmith options object.
 */

const LangSmithOptions = exports.LangSmithOptions = _zod.z.object({
  /**
   * The project name to use with tracing.
   */
  projectName: _zod.z.string(),
  /**
   * The api key for the project
   */
  apiKey: _zod.z.string()
});