"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.config = exports.InfraServerPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/es-ui-shared-plugin/server");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/observability-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _constants = require("../common/constants");
var _plugin_config_types = require("../common/plugin_config_types");
var _features = require("./features");
var _infra_server = require("./infra_server");
var _framework_fields_adapter = require("./lib/adapters/fields/framework_fields_adapter");
var _kibana_framework_adapter = require("./lib/adapters/framework/kibana_framework_adapter");
var _kibana_metrics_adapter = require("./lib/adapters/metrics/kibana_metrics_adapter");
var _source_status = require("./lib/adapters/source_status");
var _alerting = require("./lib/alerting");
var _register_rule_types = require("./lib/alerting/register_rule_types");
var _fields_domain = require("./lib/domains/fields_domain");
var _metrics_domain = require("./lib/domains/metrics_domain");
var _sources = require("./lib/sources");
var _source_status2 = require("./lib/source_status");
var _saved_objects = require("./saved_objects");
var _inventory_views = require("./services/inventory_views");
var _metrics_explorer_views = require("./services/metrics_explorer_views");
var _rules = require("./services/rules");
var _usage_collector = require("./usage/usage_collector");
var _map_source_to_log_view = require("./utils/map_source_to_log_view");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const config = exports.config = {
  schema: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    }),
    alerting: _configSchema.schema.object({
      inventory_threshold: _configSchema.schema.object({
        group_by_page_size: _configSchema.schema.number({
          defaultValue: 5_000
        })
      }),
      metric_threshold: _configSchema.schema.object({
        group_by_page_size: _configSchema.schema.number({
          defaultValue: 10_000
        })
      })
    }),
    inventory: _configSchema.schema.object({
      compositeSize: _configSchema.schema.number({
        defaultValue: 2000
      })
    }),
    sources: _configSchema.schema.maybe(_configSchema.schema.object({
      default: _configSchema.schema.maybe(_configSchema.schema.object({
        fields: _configSchema.schema.maybe(_configSchema.schema.object({
          message: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
        }))
      }))
    })),
    featureFlags: _configSchema.schema.object({
      customThresholdAlertsEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: false
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      logsUIEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      metricsExplorerEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      osqueryEnabled: _configSchema.schema.boolean({
        defaultValue: true
      }),
      inventoryThresholdAlertRuleEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: true
        })
      }),
      metricThresholdAlertRuleEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      logThresholdAlertRuleEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: false
        })
      }),
      alertsAndRulesDropdownEnabled: (0, _configSchema.offeringBasedSchema)({
        traditional: _configSchema.schema.boolean({
          defaultValue: true
        }),
        serverless: _configSchema.schema.boolean({
          defaultValue: true
        })
      }),
      /**
       * Depends on optional "profilingDataAccess" and "profiling"
       * plugins. Enable both with `xpack.profiling.enabled: true` before
       * enabling this feature flag.
       */
      profilingEnabled: _configSchema.schema.boolean({
        defaultValue: false
      }),
      ruleFormV2Enabled: _configSchema.schema.boolean({
        defaultValue: false
      })
    })
  }),
  exposeToBrowser: _plugin_config_types.publicConfigKeys
};
const logsSampleDataLinkLabel = _i18n.i18n.translate('xpack.infra.sampleDataLinkLabel', {
  defaultMessage: 'Logs'
});
class InfraServerPlugin {
  constructor(context) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "libs", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "logsRules", void 0);
    (0, _defineProperty2.default)(this, "metricsRules", void 0);
    (0, _defineProperty2.default)(this, "inventoryViews", void 0);
    (0, _defineProperty2.default)(this, "metricsExplorerViews", void 0);
    this.config = context.config.get();
    this.logger = context.logger.get();
    this.logsRules = new _rules.RulesService(_constants.LOGS_FEATURE_ID, _register_rule_types.LOGS_RULES_ALERT_CONTEXT, this.logger.get('logsRules'));
    this.metricsRules = new _rules.RulesService(_constants.METRICS_FEATURE_ID, _register_rule_types.METRICS_RULES_ALERT_CONTEXT, this.logger.get('metricsRules'));
    this.inventoryViews = new _inventory_views.InventoryViewsService(this.logger.get('inventoryViews'));
    this.metricsExplorerViews = this.config.featureFlags.metricsExplorerEnabled ? new _metrics_explorer_views.MetricsExplorerViewsService(this.logger.get('metricsExplorerViews')) : undefined;
  }
  setup(core, plugins) {
    var _this$metricsExplorer, _this$config$sources, _this$config$sources$, _this$config$sources$2;
    const framework = new _kibana_framework_adapter.KibanaFramework(core, this.config, plugins);
    const metricsClient = plugins.metricsDataAccess.client;
    const getApmIndices = plugins.apmDataAccess.getApmIndices;
    metricsClient.setDefaultMetricIndicesHandler(async options => {
      const sourceConfiguration = await sources.getInfraSourceConfiguration(options.savedObjectsClient, 'default');
      return sourceConfiguration.configuration.metricAlias;
    });
    const sources = new _sources.InfraSources({
      config: this.config,
      metricsClient
    });
    const sourceStatus = new _source_status2.InfraSourceStatus(new _source_status.InfraElasticsearchSourceStatusAdapter(framework), {
      sources
    });

    // Setup infra services
    const inventoryViews = this.inventoryViews.setup();
    const metricsExplorerViews = (_this$metricsExplorer = this.metricsExplorerViews) === null || _this$metricsExplorer === void 0 ? void 0 : _this$metricsExplorer.setup();

    // Register saved object types
    core.savedObjects.registerType(_sources.infraSourceConfigurationSavedObjectType);
    core.savedObjects.registerType(_saved_objects.inventoryViewSavedObjectType);
    core.savedObjects.registerType(_saved_objects.infraCustomDashboardsSavedObjectType);
    if (this.config.featureFlags.metricsExplorerEnabled) {
      core.savedObjects.registerType(_saved_objects.metricsExplorerViewSavedObjectType);
    }

    // TODO: separate these out individually and do away with "domains" as a temporary group
    // and make them available via the request context so we can do away with
    // the wrapper classes
    const domainLibs = {
      fields: new _fields_domain.InfraFieldsDomain(new _framework_fields_adapter.FrameworkFieldsAdapter(framework), {
        sources
      }),
      logEntries: plugins.logsShared.logEntries,
      metrics: new _metrics_domain.InfraMetricsDomain(new _kibana_metrics_adapter.KibanaMetricsAdapter(framework))
    };
    this.libs = {
      configuration: this.config,
      framework,
      sources,
      sourceStatus,
      metricsClient,
      getApmIndices,
      ...domainLibs,
      handleEsError: _server.handleEsError,
      logsRules: this.logsRules.setup(core, plugins),
      metricsRules: this.metricsRules.setup(core, plugins),
      getStartServices: () => core.getStartServices(),
      getAlertDetailsConfig: () => plugins.observability.getAlertDetailsConfig(),
      logger: this.logger,
      basePath: core.http.basePath,
      alertsLocator: plugins.share.url.locators.get(_common.alertsLocatorID)
    };
    plugins.features.registerKibanaFeature(_features.METRICS_FEATURE);
    plugins.features.registerKibanaFeature(_features.LOGS_FEATURE);

    // Register an handler to retrieve the fallback logView starting from a source configuration
    plugins.logsShared.logViews.registerLogViewFallbackHandler(async (sourceId, {
      soClient
    }) => {
      const sourceConfiguration = await sources.getInfraSourceConfiguration(soClient, sourceId);
      return (0, _map_source_to_log_view.mapSourceToLogView)(sourceConfiguration);
    });
    plugins.logsShared.logViews.setLogViewsStaticConfig({
      messageFields: (_this$config$sources = this.config.sources) === null || _this$config$sources === void 0 ? void 0 : (_this$config$sources$ = _this$config$sources.default) === null || _this$config$sources$ === void 0 ? void 0 : (_this$config$sources$2 = _this$config$sources$.fields) === null || _this$config$sources$2 === void 0 ? void 0 : _this$config$sources$2.message
    });
    plugins.logsShared.registerUsageCollectorActions({
      countLogs: () => _usage_collector.UsageCollector.countLogs()
    });
    if (this.config.featureFlags.logsUIEnabled) {
      plugins.home.sampleData.addAppLinksToSampleDataset('logs', [{
        sampleObject: null,
        // indicates that there is no sample object associated with this app link's path
        getPath: () => `/app/logs`,
        label: logsSampleDataLinkLabel,
        icon: 'logsApp'
      }]);
    }
    (0, _alerting.registerRuleTypes)(plugins.alerting, this.libs, this.config);
    core.http.registerRouteHandlerContext('infra', async (context, request) => {
      var _plugins$ml, _plugins$ml2, _plugins$spaces$space, _plugins$spaces;
      const coreContext = await context.core;
      const savedObjectsClient = coreContext.savedObjects.client;
      const uiSettingsClient = coreContext.uiSettings.client;
      const mlSystem = (_plugins$ml = plugins.ml) === null || _plugins$ml === void 0 ? void 0 : _plugins$ml.mlSystemProvider(request, savedObjectsClient);
      const mlAnomalyDetectors = (_plugins$ml2 = plugins.ml) === null || _plugins$ml2 === void 0 ? void 0 : _plugins$ml2.anomalyDetectorsProvider(request, savedObjectsClient);
      const spaceId = (_plugins$spaces$space = (_plugins$spaces = plugins.spaces) === null || _plugins$spaces === void 0 ? void 0 : _plugins$spaces.spacesService.getSpaceId(request)) !== null && _plugins$spaces$space !== void 0 ? _plugins$spaces$space : _common2.DEFAULT_SPACE_ID;
      return {
        mlAnomalyDetectors,
        mlSystem,
        spaceId,
        savedObjectsClient,
        uiSettingsClient
      };
    });

    // Telemetry
    _usage_collector.UsageCollector.registerUsageCollector(plugins.usageCollection);
    return {
      inventoryViews,
      metricsExplorerViews
    };
  }
  start(core, pluginsStart) {
    var _this$metricsExplorer2;
    const inventoryViews = this.inventoryViews.start({
      infraSources: this.libs.sources,
      savedObjects: core.savedObjects
    });
    const metricsExplorerViews = (_this$metricsExplorer2 = this.metricsExplorerViews) === null || _this$metricsExplorer2 === void 0 ? void 0 : _this$metricsExplorer2.start({
      infraSources: this.libs.sources,
      savedObjects: core.savedObjects
    });
    (0, _infra_server.initInfraServer)(this.libs, core, pluginsStart);
    return {
      inventoryViews,
      metricsExplorerViews
    };
  }
  stop() {}
}
exports.InfraServerPlugin = InfraServerPlugin;