"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServices = void 0;
var _server = require("@kbn/observability-plugin/server");
var _elasticsearch = require("@kbn/observability-shared-plugin/common/field_names/elasticsearch");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getServices = async (client, apmIndices, options) => {
  var _resultMetrics$aggreg, _resultMetrics$aggreg2, _resultLogs$aggregati, _resultLogs$aggregati2;
  const {
    error,
    metric
  } = apmIndices;
  const {
    filters,
    size = 10,
    from,
    to
  } = options;
  const commonFiltersList = [{
    range: {
      '@timestamp': {
        gte: from,
        lte: to
      }
    }
  }, {
    exists: {
      field: 'service.name'
    }
  }];
  if (filters['host.name']) {
    // also query for host.hostname field along with host.name, as some services may use this field
    const HOST_HOSTNAME_FIELD = 'host.hostname';
    commonFiltersList.push({
      bool: {
        should: [...(0, _server.termQuery)(_constants.HOST_NAME_FIELD, filters[_constants.HOST_NAME_FIELD]), ...(0, _server.termQuery)(HOST_HOSTNAME_FIELD, filters[_constants.HOST_NAME_FIELD])],
        minimum_should_match: 1
      }
    });
  }
  const aggs = {
    services: {
      terms: {
        field: 'service.name',
        size
      },
      aggs: {
        latestAgent: {
          top_metrics: {
            metrics: [{
              field: 'agent.name'
            }],
            sort: {
              '@timestamp': 'desc'
            },
            size: 1
          }
        }
      }
    }
  };
  // get services from transaction metrics
  const metricsQuery = {
    size: 0,
    _source: false,
    query: {
      bool: {
        filter: [{
          term: {
            [_elasticsearch.PROCESSOR_EVENT]: 'metric'
          }
        }, {
          bool: {
            should: [{
              term: {
                'metricset.name': 'app'
              }
            }, {
              bool: {
                must: [{
                  term: {
                    'metricset.name': 'transaction'
                  }
                }, {
                  term: {
                    'metricset.interval': '1m' // make this dynamic if we start returning time series data
                  }
                }]
              }
            }],
            minimum_should_match: 1
          }
        }, ...commonFiltersList]
      }
    },
    aggs
  };
  // get services from logs
  const logsQuery = {
    size: 0,
    _source: false,
    query: {
      bool: {
        filter: commonFiltersList
      }
    },
    aggs
  };
  const resultMetrics = await client({
    body: metricsQuery,
    index: [metric]
  });
  const resultLogs = await client({
    body: logsQuery,
    index: [error]
  });
  const servicesListBucketsFromMetrics = ((_resultMetrics$aggreg = resultMetrics.aggregations) === null || _resultMetrics$aggreg === void 0 ? void 0 : (_resultMetrics$aggreg2 = _resultMetrics$aggreg.services) === null || _resultMetrics$aggreg2 === void 0 ? void 0 : _resultMetrics$aggreg2.buckets) || [];
  const servicesListBucketsFromLogs = ((_resultLogs$aggregati = resultLogs.aggregations) === null || _resultLogs$aggregati === void 0 ? void 0 : (_resultLogs$aggregati2 = _resultLogs$aggregati.services) === null || _resultLogs$aggregati2 === void 0 ? void 0 : _resultLogs$aggregati2.buckets) || [];
  const serviceMap = [...servicesListBucketsFromMetrics, ...servicesListBucketsFromLogs].reduce((acc, bucket) => {
    const serviceName = bucket.key;
    const latestAgentEntry = bucket.latestAgent.top[0];
    const latestTimestamp = latestAgentEntry.sort[0];
    const agentName = latestAgentEntry.metrics['agent.name'];
    // dedup and get the latest timestamp
    const existingService = acc.get(serviceName);
    if (!existingService || existingService.latestTimestamp < latestTimestamp) {
      acc.set(serviceName, {
        latestTimestamp,
        agentName
      });
    }
    return acc;
  }, new Map());
  const services = Array.from(serviceMap).slice(0, size).map(([serviceName, {
    agentName
  }]) => ({
    serviceName,
    agentName
  }));
  return {
    services
  };
};
exports.getServices = getServices;