"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMetricsExplorerData = useMetricsExplorerData;
var _reactQuery = require("@tanstack/react-query");
var _public = require("@kbn/kibana-react-plugin/public");
var _metrics_source = require("../../../../containers/metrics_source");
var _metrics_explorer = require("../../../../../common/http_api/metrics_explorer");
var _kuery = require("../../../../utils/kuery");
var _runtime_types = require("../../../../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useMetricsExplorerData({
  options,
  timestamps: {
    fromTimestamp,
    toTimestamp,
    interval
  },
  enabled = true
}) {
  const {
    http
  } = (0, _public.useKibana)().services;
  const {
    metricsView
  } = (0, _metrics_source.useMetricsDataViewContext)();
  const {
    isLoading,
    data,
    error,
    refetch,
    fetchNextPage
  } = (0, _reactQuery.useInfiniteQuery)({
    queryKey: ['metricExplorer', options, fromTimestamp, toTimestamp],
    queryFn: async ({
      signal,
      pageParam = {
        afterKey: null
      }
    }) => {
      if (!fromTimestamp || !toTimestamp) {
        throw new Error('Unable to parse timerange');
      }
      if (!http) {
        throw new Error('HTTP service is unavailable');
      }
      if (!(metricsView !== null && metricsView !== void 0 && metricsView.dataViewReference)) {
        throw new Error('DataView is unavailable');
      }
      const {
        afterKey
      } = pageParam;
      const response = await http.post('/api/infra/metrics_explorer', {
        method: 'POST',
        body: JSON.stringify({
          forceInterval: options.forceInterval,
          dropLastBucket: options.dropLastBucket != null ? options.dropLastBucket : true,
          metrics: options.aggregation === 'count' ? [{
            aggregation: 'count'
          }] : options.metrics,
          groupBy: options.groupBy,
          groupInstance: options.groupInstance,
          afterKey,
          limit: options.limit,
          indexPattern: metricsView.indices,
          filterQuery: options.filterQuery && (0, _kuery.convertKueryToElasticSearchQuery)(options.filterQuery, metricsView.dataViewReference) || void 0,
          timerange: {
            interval,
            from: fromTimestamp,
            to: toTimestamp
          }
        }),
        signal
      });
      return (0, _runtime_types.decodeOrThrow)(_metrics_explorer.metricsExplorerResponseRT)(response);
    },
    getNextPageParam: lastPage => lastPage.pageInfo,
    enabled: enabled && !!fromTimestamp && !!toTimestamp && !!http && !!metricsView,
    refetchOnWindowFocus: false,
    retry: false
  });
  return {
    data,
    error: (error === null || error === void 0 ? void 0 : error.body) || error,
    fetchNextPage,
    isLoading,
    refetch
  };
}