"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEsWarningText = void 0;
exports.registerPutDataRetention = registerPutDataRetention;
var _configSchema = require("@kbn/config-schema");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** HTTP Warning headers have the following syntax:
 * <warn-code> <warn-agent> <warn-text> (where warn-code is a three-digit number)
 * This function only returns the warn-text if it exists.
 * */
const getEsWarningText = warning => {
  const match = warning.match(/\d{3} Elasticsearch-\w+ "(.*)"/);
  return match ? match[1] : null;
};
exports.getEsWarningText = getEsWarningText;
function registerPutDataRetention({
  router,
  lib: {
    handleEsError
  }
}) {
  const paramsSchema = _configSchema.schema.object({
    name: _configSchema.schema.string()
  });
  const bodySchema = _configSchema.schema.object({
    dataRetention: _configSchema.schema.maybe(_configSchema.schema.string()),
    enabled: _configSchema.schema.maybe(_configSchema.schema.boolean())
  });
  router.put({
    path: (0, _.addBasePath)('/data_streams/{name}/data_retention'),
    validate: {
      params: paramsSchema,
      body: bodySchema
    }
  }, async (context, request, response) => {
    const {
      name
    } = request.params;
    const {
      dataRetention,
      enabled
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      // Only when enabled is explicitly set to false, we delete the data retention policy.
      if (enabled === false) {
        await client.asCurrentUser.indices.deleteDataLifecycle({
          name
        });
      } else {
        var _getEsWarningText;
        // Otherwise, we create or update the data retention policy.
        //
        // Be aware that in serverless it could happen that the user defined
        // data retention wont be the effective retention as there might be a
        // global data retention limit set.
        const {
          headers
        } = await client.asCurrentUser.indices.putDataLifecycle({
          name,
          data_retention: dataRetention
        }, {
          meta: true
        });
        return response.ok({
          body: {
            success: true,
            ...(headers !== null && headers !== void 0 && headers.warning ? {
              warning: (_getEsWarningText = getEsWarningText(headers.warning)) !== null && _getEsWarningText !== void 0 ? _getEsWarningText : headers.warning
            } : {})
          }
        });
      }
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}