"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EditDataRetentionModal = void 0;
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var _react = _interopRequireDefault(require("react"));
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _shared_imports = require("../../../../../shared_imports");
var _routing = require("../../../../services/routing");
var _documentation = require("../../../../services/documentation");
var _common = require("../../../../../../common");
var _time_units = require("../../../../constants/time_units");
var _data_streams = require("../../../../lib/data_streams");
var _app_context = require("../../../../app_context");
var _shared = require("../../../../components/shared");
var _api = require("../../../../services/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const configurationFormSchema = {
  dataRetention: {
    type: _shared_imports.FIELD_TYPES.TEXT,
    label: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.dataRetentionField', {
      defaultMessage: 'Data retention'
    }),
    formatters: [_shared_imports.fieldFormatters.toInt],
    validations: [{
      validator: ({
        value,
        formData
      }) => {
        // If infiniteRetentionPeriod is set, we dont need to validate the data retention field
        if (formData.infiniteRetentionPeriod) {
          return undefined;
        }
        if (!value) {
          return {
            message: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.dataRetentionFieldRequiredError', {
              defaultMessage: 'A data retention value is required.'
            })
          };
        }
        if (value <= 0) {
          return {
            message: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.dataRetentionFieldNonNegativeError', {
              defaultMessage: `A positive value is required.`
            })
          };
        }
      }
    }]
  },
  timeUnit: {
    type: _shared_imports.FIELD_TYPES.TEXT,
    label: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.timeUnitField', {
      defaultMessage: 'Time unit'
    })
  },
  infiniteRetentionPeriod: {
    type: _shared_imports.FIELD_TYPES.TOGGLE,
    defaultValue: false,
    label: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.infiniteRetentionPeriodField', {
      defaultMessage: 'Keep data indefinitely'
    })
  },
  dataRetentionEnabled: {
    type: _shared_imports.FIELD_TYPES.TOGGLE,
    defaultValue: false,
    label: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.dataRetentionEnabledField', {
      defaultMessage: 'Enable data retention'
    })
  }
};
const MixedIndicesCallout = ({
  ilmPolicyLink,
  ilmPolicyName,
  dataStreamName,
  history
}) => {
  return /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    title: _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.someManagedByILMTitle', {
      defaultMessage: 'Some indices are managed by ILM'
    }),
    color: "warning",
    iconType: "warning",
    "data-test-subj": "someIndicesAreManagedByILMCallout"
  }, /*#__PURE__*/_react.default.createElement("p", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.someManagedByILMBody",
    defaultMessage: "One or more indices are managed by an ILM policy ({viewAllIndicesLink}). Updating data retention for this data stream won't affect these indices. Instead you will have to update the {ilmPolicyLink} policy.",
    values: {
      ilmPolicyLink: /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
        "data-test-subj": "viewIlmPolicyLink",
        href: ilmPolicyLink
      }, ilmPolicyName),
      viewAllIndicesLink: /*#__PURE__*/_react.default.createElement(_eui.EuiLink, (0, _extends2.default)({}, (0, _shared_imports.reactRouterNavigate)(history, (0, _routing.getIndexListUri)(`data_stream="${dataStreamName}"`, true)), {
        "data-test-subj": "viewAllIndicesLink"
      }), /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.viewAllIndices",
        defaultMessage: "view indices"
      }))
    }
  })));
};
const EditDataRetentionModal = ({
  dataStream,
  ilmPolicyName,
  ilmPolicyLink,
  onClose
}) => {
  const lifecycle = dataStream === null || dataStream === void 0 ? void 0 : dataStream.lifecycle;
  const dataStreamName = dataStream === null || dataStream === void 0 ? void 0 : dataStream.name;
  const {
    history
  } = (0, _app_context.useAppContext)();
  const dslWithIlmIndices = (0, _data_streams.isDSLWithILMIndices)(dataStream);
  const {
    size,
    unit
  } = (0, _common.splitSizeAndUnits)(lifecycle === null || lifecycle === void 0 ? void 0 : lifecycle.data_retention);
  const {
    services: {
      notificationService
    },
    config: {
      enableTogglingDataRetention
    }
  } = (0, _app_context.useAppContext)();
  const {
    form
  } = (0, _shared_imports.useForm)({
    defaultValue: {
      dataRetention: size,
      timeUnit: unit || 'd',
      dataRetentionEnabled: lifecycle === null || lifecycle === void 0 ? void 0 : lifecycle.enabled,
      // When data retention is not set and lifecycle is enabled, is the only scenario in
      // which data retention will be infinite. If lifecycle isnt set or is not enabled, we
      // dont have inifinite data retention.
      infiniteRetentionPeriod: (lifecycle === null || lifecycle === void 0 ? void 0 : lifecycle.enabled) && !(lifecycle !== null && lifecycle !== void 0 && lifecycle.data_retention)
    },
    schema: configurationFormSchema,
    id: 'editDataRetentionForm'
  });
  const [formData] = (0, _shared_imports.useFormData)({
    form
  });
  const isDirty = (0, _shared_imports.useFormIsModified)({
    form
  });
  const onSubmitForm = async () => {
    const {
      isValid,
      data
    } = await form.submit();
    if (!isValid) {
      return;
    }

    // When enableTogglingDataRetention is disabled (ie: serverless) we don't mount
    // the dataRetentionEnabled field in the UI, which means that the form state for
    // this field regardless if it has defaultValue or if its set with form.setValue.
    // This seems to be a design decision from the formlib and there doesnt seem to
    // be a way around it AFAICT.
    // So when that happens we want to make sure that the dataRetention is always enabled.
    if (!(0, _lodash.has)(data, 'dataRetentionEnabled')) {
      data.dataRetentionEnabled = true;
    }
    return (0, _api.updateDataRetention)(dataStreamName, data).then(({
      data: responseData,
      error
    }) => {
      if (responseData) {
        // If the response came back with a warning from ES, rely on that for the
        // toast message.
        if (responseData.warning) {
          notificationService.showWarningToast(responseData.warning);
          return onClose({
            hasUpdatedDataRetention: true
          });
        }
        const successMessage = _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.successDataRetentionNotification', {
          defaultMessage: 'Data retention {disabledDataRetention, plural, one { disabled } other { updated } }',
          values: {
            disabledDataRetention: !data.dataRetentionEnabled ? 1 : 0
          }
        });
        notificationService.showSuccessToast(successMessage);
        return onClose({
          hasUpdatedDataRetention: true
        });
      }
      if (error) {
        const errorMessage = _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.errorDataRetentionNotification', {
          defaultMessage: "Error updating data retention: ''{error}''",
          values: {
            error: error.message
          }
        });
        notificationService.showDangerToast(errorMessage);
      }
      onClose();
    });
  };
  return /*#__PURE__*/_react.default.createElement(_eui.EuiModal, {
    onClose: () => onClose(),
    "data-test-subj": "editDataRetentionModal"
  }, /*#__PURE__*/_react.default.createElement(_shared_imports.Form, {
    form: form,
    "data-test-subj": "editDataRetentionForm"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiModalHeader, null, /*#__PURE__*/_react.default.createElement(_eui.EuiModalHeaderTitle, null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.modalTitleText",
    defaultMessage: "Edit data retention"
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiModalBody, null, dslWithIlmIndices && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(MixedIndicesCallout, {
    history: history,
    ilmPolicyLink: ilmPolicyLink,
    ilmPolicyName: ilmPolicyName,
    dataStreamName: dataStreamName
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null)), enableTogglingDataRetention && /*#__PURE__*/_react.default.createElement(_shared_imports.UseField, {
    path: "dataRetentionEnabled",
    component: _shared_imports.ToggleField,
    "data-test-subj": "dataRetentionEnabledField"
  }), /*#__PURE__*/_react.default.createElement(_shared_imports.UseField, {
    path: "dataRetention",
    component: _shared_imports.NumericField,
    labelAppend: /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      size: "xs"
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
      href: _documentation.documentationService.getUpdateExistingDS(),
      target: "_blank",
      external: true
    }, _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.learnMoreLinkText', {
      defaultMessage: 'How does it work?'
    }))),
    componentProps: {
      fullWidth: false,
      euiFieldProps: {
        disabled: formData.infiniteRetentionPeriod || !formData.dataRetentionEnabled && enableTogglingDataRetention,
        'data-test-subj': `dataRetentionValue`,
        min: 1,
        append: /*#__PURE__*/_react.default.createElement(_shared.UnitField, {
          path: "timeUnit",
          options: _time_units.timeUnits,
          disabled: formData.infiniteRetentionPeriod || !formData.dataRetentionEnabled && enableTogglingDataRetention,
          euiFieldProps: {
            'data-test-subj': 'timeUnit',
            'aria-label': _i18n.i18n.translate('xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.unitsAriaLabel', {
              defaultMessage: 'Time unit'
            })
          }
        })
      }
    }
  }), /*#__PURE__*/_react.default.createElement(_shared_imports.UseField, {
    path: "infiniteRetentionPeriod",
    component: _shared_imports.ToggleField,
    "data-test-subj": "infiniteRetentionPeriod",
    componentProps: {
      euiFieldProps: {
        disabled: !formData.dataRetentionEnabled && enableTogglingDataRetention
      }
    }
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null)), /*#__PURE__*/_react.default.createElement(_eui.EuiModalFooter, null, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
    "data-test-subj": "cancelButton",
    onClick: () => onClose()
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.cancelButtonLabel",
    defaultMessage: "Cancel"
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
    fill: true,
    type: "submit",
    isLoading: false,
    disabled: form.isSubmitted && form.isValid === false || !isDirty,
    "data-test-subj": "saveButton",
    onClick: onSubmitForm
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.idxMgmt.dataStreamsDetailsPanel.editDataRetentionModal.saveButtonLabel",
    defaultMessage: "Save"
  })))));
};
exports.EditDataRetentionModal = EditDataRetentionModal;