"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataTierAllocationField = void 0;
var _lodash = require("lodash");
var _react = _interopRequireDefault(require("react"));
var _i18n = require("@kbn/i18n");
var _eui = require("@elastic/eui");
var _shared_imports = require("../../../../../../../shared_imports");
var _lib = require("../../../../../../lib");
var _api = require("../../../../../../services/api");
var _components = require("./components");
require("./_data_tier_allocation.scss");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const i18nTexts = {
  title: _i18n.i18n.translate('xpack.indexLifecycleMgmt.common.dataTier.title', {
    defaultMessage: 'Data allocation'
  })
};
/**
 * Top-level layout control for the data tier allocation field.
 */
const DataTierAllocationField = ({
  phase,
  description
}) => {
  var _cloud$isCloudEnabled;
  const {
    services: {
      cloud
    }
  } = (0, _shared_imports.useKibana)();
  const dataTierAllocationTypePath = `_meta.${phase}.dataTierAllocationType`;
  const [formData] = (0, _shared_imports.useFormData)({
    watch: dataTierAllocationTypePath
  });
  const allocationType = (0, _lodash.get)(formData, dataTierAllocationTypePath);
  const {
    data,
    resendRequest,
    error,
    isLoading
  } = (0, _api.useLoadNodes)();
  const {
    nodesByRoles,
    nodesByAttributes,
    isUsingDeprecatedDataRoleConfig
  } = data;
  const hasNodeAttributes = Boolean(Object.keys(nodesByAttributes !== null && nodesByAttributes !== void 0 ? nodesByAttributes : {}).length);
  const isCloudEnabled = (_cloud$isCloudEnabled = cloud === null || cloud === void 0 ? void 0 : cloud.isCloudEnabled) !== null && _cloud$isCloudEnabled !== void 0 ? _cloud$isCloudEnabled : false;
  const cloudDeploymentUrl = cloud === null || cloud === void 0 ? void 0 : cloud.deploymentUrl;
  const allocationNodeRoleForPhase = (0, _lib.getAvailableNodeRoleForPhase)(phase, nodesByRoles);
  const noTiersAvailable = allocationNodeRoleForPhase === undefined;
  const willUseFallbackTier = allocationNodeRoleForPhase !== undefined && !(0, _lib.isNodeRoleFirstPreference)(phase, allocationNodeRoleForPhase);
  const renderNotice = () => {
    switch (allocationType) {
      case 'node_roles':
        /**
         * On Cloud most users should be using autoscaling which will provision tiers as they are needed. We do not surface any
         * of the notices below.
         */
        if (isCloudEnabled) {
          return null;
        }

        /**
         * Node role allocation moves data in a phase to a corresponding tier of the same name. To prevent policy execution from getting
         * stuck ILM allocation will fall back to a previous tier if possible. We show the WARNING below to inform a user when even
         * this fallback will not succeed.
         */
        if (noTiersAvailable) {
          return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
            size: "s"
          }), /*#__PURE__*/_react.default.createElement(_components.NoTiersAvailableNotice, {
            phase: phase
          }));
        }

        /**
         * If we are able to fallback to a data tier that does not map to this phase, we show a notice informing the user that their
         * data will not be assigned to a corresponding tier.
         */
        if (willUseFallbackTier) {
          return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
            size: "s"
          }), /*#__PURE__*/_react.default.createElement(_components.WillUseFallbackTierNotice, {
            phase: phase,
            targetNodeRole: allocationNodeRoleForPhase
          }));
        }
        break;
      case 'node_attrs':
        /**
         * If there are no node attributes, advise the user on the default allocation behavior.
         */
        if (!hasNodeAttributes) {
          /**
           * If data nodes are available, default allocation behavior will be to those nodes.
           */
          if (isUsingDeprecatedDataRoleConfig) {
            return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
              size: "s"
            }), /*#__PURE__*/_react.default.createElement(_components.DefaultToDataNodesNotice, {
              phase: phase
            }));
          }

          /**
           * Node role allocation moves data in a phase to a corresponding tier of the same name. To prevent policy execution from getting
           * stuck ILM allocation will fall back to a previous tier if possible. We show the WARNING below to inform a user when even
           * this fallback will not succeed, for example if the user only has 'data' node roles, and no `data_<tier>` node roles.
           */
          if (noTiersAvailable) {
            return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
              size: "s"
            }), /*#__PURE__*/_react.default.createElement(_components.NoTiersAvailableUsingNodeAttributesNotice, null));
          }

          /**
           * If we are able to fallback to a data tier that does not map to this phase, we show a notice informing the user that their
           * data will not be assigned to a corresponding tier.
           */
          if (willUseFallbackTier) {
            return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
              size: "s"
            }), /*#__PURE__*/_react.default.createElement(_components.WillUseFallbackTierUsingNodeAttributesNotice, {
              phase: phase,
              targetNodeRole: allocationNodeRoleForPhase
            }));
          }

          /**
           * If using node roles, default allocation behavior will be to the preferred nodes, depending on the phase.
           */
          return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
            size: "s"
          }), /*#__PURE__*/_react.default.createElement(_components.DefaultToDataTiersNotice, {
            phase: phase
          }));
        }

        /**
         * Special Cloud case: when deprecated data role configuration is in use, it means that this deployment is not using
         * the new node role based allocation. We drive users to the cloud console to migrate to node role based allocation
         * in that case.
         */
        if (isCloudEnabled && isUsingDeprecatedDataRoleConfig) {
          return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
            size: "s"
          }), /*#__PURE__*/_react.default.createElement(_components.CloudDataTierCallout, {
            linkToCloudDeployment: cloudDeploymentUrl
          }));
        }
        break;
      default:
        return null;
    }
  };
  return /*#__PURE__*/_react.default.createElement(_eui.EuiDescribedFormGroup, {
    title: /*#__PURE__*/_react.default.createElement("h3", null, i18nTexts.title),
    description: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, description, isLoading ? /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "m"
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, {
      "data-test-subj": "allocationLoadingSpinner",
      size: "xl"
    })) : error && /*#__PURE__*/_react.default.createElement(_components.LoadingError, {
      onResendRequest: resendRequest,
      message: error.message,
      statusCode: error.statusCode
    })),
    fullWidth: true
  }, /*#__PURE__*/_react.default.createElement("div", {
    className: "ilmDataTierAllocationField"
  }, /*#__PURE__*/_react.default.createElement(_components.DataTierAllocation, {
    hasNodeAttributes: hasNodeAttributes,
    phase: phase,
    nodes: nodesByAttributes,
    isCloudEnabled: isCloudEnabled,
    isUsingDeprecatedDataRoleConfig: isUsingDeprecatedDataRoleConfig,
    isLoading: isLoading
  }), !isLoading && renderNotice()));
};
exports.DataTierAllocationField = DataTierAllocationField;