"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchCursorPit = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _search_cursor = require("./search_cursor");
var _i18n_texts = require("./i18n_texts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class SearchCursorPit extends _search_cursor.SearchCursor {
  constructor(indexPatternTitle, settings, clients, abortController, logger) {
    super(indexPatternTitle, settings, clients, abortController, logger);
    (0, _defineProperty2.default)(this, "searchAfter", void 0);
  }

  /**
   * When point-in-time strategy is used, the first step is to open a PIT ID for search context.
   */
  async initialize() {
    this.cursorId = await this.openPointInTime();
  }
  async openPointInTime() {
    const {
      includeFrozen,
      maxConcurrentShardRequests,
      scroll,
      taskInstanceFields
    } = this.settings;
    let pitId;
    this.logger.debug(`Requesting PIT for: [${this.indexPatternTitle}]...`);
    try {
      // NOTE: if ES is overloaded, this request could time out
      const response = await this.clients.es.asCurrentUser.openPointInTime({
        index: this.indexPatternTitle,
        keep_alive: scroll.duration(taskInstanceFields),
        ignore_unavailable: true,
        // @ts-expect-error ignore_throttled is not in the type definition, but it is accepted by es
        ignore_throttled: includeFrozen ? false : undefined // "true" will cause deprecation warnings logged in ES
      }, {
        signal: this.abortController.signal,
        requestTimeout: scroll.duration(taskInstanceFields),
        maxRetries: 0,
        maxConcurrentShardRequests
      });
      pitId = response.id;
    } catch (err) {
      this.logger.error(err);
    }
    if (!pitId) {
      throw new Error(`Could not receive a PIT ID!`);
    }
    this.logger.debug(`Opened PIT ID: ${this.formatCursorId(pitId)}`);
    return pitId;
  }
  async searchWithPit(searchBody) {
    const {
      maxConcurrentShardRequests,
      scroll,
      taskInstanceFields
    } = this.settings;

    // maxConcurrentShardRequests=0 is not supported
    const effectiveMaxConcurrentShardRequests = maxConcurrentShardRequests > 0 ? maxConcurrentShardRequests : undefined;
    const searchParamsPit = {
      params: {
        body: searchBody,
        max_concurrent_shard_requests: effectiveMaxConcurrentShardRequests
      }
    };
    return await (0, _rxjs.lastValueFrom)(this.clients.data.search(searchParamsPit, {
      strategy: _common.ES_SEARCH_STRATEGY,
      abortSignal: this.abortController.signal,
      transport: {
        maxRetries: 0,
        // retrying reporting jobs is handled in the task manager scheduling logic
        requestTimeout: scroll.duration(taskInstanceFields)
      }
    }));
  }
  async getPage(searchSource) {
    const {
      scroll,
      taskInstanceFields
    } = this.settings;
    if (!this.cursorId) {
      throw new Error(`No access to valid PIT ID!`);
    }
    searchSource.setField('pit', {
      id: this.cursorId,
      keep_alive: scroll.duration(taskInstanceFields)
    });
    const searchAfter = this.getSearchAfter();
    if (searchAfter) {
      searchSource.setField('searchAfter', searchAfter);
    }
    this.logger.debug(`Executing search request with PIT ID: [${this.formatCursorId(this.cursorId)}]` + (searchAfter ? ` search_after: [${searchAfter}]` : ''));
    const searchBody = searchSource.getSearchRequestBody();
    if (searchBody == null) {
      throw new Error('Could not retrieve the search body!');
    }
    const response = await this.searchWithPit(searchBody);
    if (!response) {
      throw new Error(`Response could not be retrieved!`);
    }
    const {
      rawResponse,
      ...rawDetails
    } = response;
    this.logSearchResults(rawDetails, rawResponse);
    this.logger.debug(`Received PIT ID: [${this.formatCursorId(rawResponse.pit_id)}]`);
    return rawResponse;
  }
  updateIdFromResults(results) {
    const cursorId = results.pit_id;
    this.cursorId = cursorId !== null && cursorId !== void 0 ? cursorId : this.cursorId;

    // track the beginning of the next page of search results
    const {
      hits
    } = results.hits;
    this.setSearchAfter(hits); // for pit only
  }
  getSearchAfter() {
    return this.searchAfter;
  }

  /**
   * For managing the search_after parameter, needed for paging using point-in-time
   */
  setSearchAfter(hits) {
    var _hits;
    // Update last sort results for next query. PIT is used, so the sort results
    // automatically include _shard_doc as a tiebreaker
    this.searchAfter = (_hits = hits[hits.length - 1]) === null || _hits === void 0 ? void 0 : _hits.sort;
    this.logger.debug(`Received search_after: [${this.searchAfter}]`);
  }
  async closeCursor() {
    if (this.cursorId) {
      this.logger.debug(`Executing close PIT on ${this.formatCursorId(this.cursorId)}`);
      await this.clients.es.asCurrentUser.closePointInTime({
        body: {
          id: this.cursorId
        }
      });
    } else {
      this.logger.warn(`No PIT Id to clear!`);
    }
  }
  getUnableToCloseCursorMessage() {
    return _i18n_texts.i18nTexts.csvUnableToClosePit();
  }
}
exports.SearchCursorPit = SearchCursorPit;