"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureFleetDirectories = ensureFleetDirectories;
exports.ensureFleetGlobalEsAssets = ensureFleetGlobalEsAssets;
exports.formatNonFatalErrors = formatNonFatalErrors;
exports.setupFleet = setupFleet;
var _promises = _interopRequireDefault(require("fs/promises"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _uuid = require("uuid");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _errors = require("../../common/errors");
var _constants2 = require("../../common/constants");
var _app_context = require("./app_context");
var _preconfiguration = require("./preconfiguration");
var _outputs = require("./preconfiguration/outputs");
var _fleet_proxies = require("./preconfiguration/fleet_proxies");
var _output = require("./output");
var _download_source = require("./download_source");
var _ = require(".");
var _setup_utils = require("./setup_utils");
var _install = require("./epm/elasticsearch/ingest_pipeline/install");
var _install2 = require("./epm/elasticsearch/template/install");
var _packages = require("./epm/packages");
var _install3 = require("./epm/packages/install");
var _managed_package_policies = require("./managed_package_policies");
var _upgrade_package_install_version = require("./setup/upgrade_package_install_version");
var _upgrade_agent_policy_schema_version = require("./setup/upgrade_agent_policy_schema_version");
var _fleet_server_host = require("./fleet_server_host");
var _fleet_server_host2 = require("./preconfiguration/fleet_server_host");
var _clean_old_fleet_indices = require("./setup/clean_old_fleet_indices");
var _fleet_server_policies_enrollment_keys = require("./setup/fleet_server_policies_enrollment_keys");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function setupFleet(soClient, esClient, options = {
  useLock: false
}) {
  const t = _elasticApmNode.default.startTransaction('fleet-setup', 'fleet');
  let created = false;
  try {
    if (options.useLock) {
      const {
        created: isCreated,
        toReturn
      } = await createLock(soClient);
      created = isCreated;
      if (toReturn) return toReturn;
    }
    return await (0, _setup_utils.awaitIfPending)(async () => createSetupSideEffects(soClient, esClient));
  } catch (error) {
    _elasticApmNode.default.captureError(error);
    t.setOutcome('failure');
    throw error;
  } finally {
    t.end();
    // only delete lock if it was created by this instance
    if (options.useLock && created) {
      await deleteLock(soClient);
    }
  }
}
async function createLock(soClient) {
  const logger = _app_context.appContextService.getLogger();
  let created;
  try {
    // check if fleet setup is already started
    const fleetSetupLock = await soClient.get(_constants2.FLEET_SETUP_LOCK_TYPE, _constants2.FLEET_SETUP_LOCK_TYPE);
    const LOCK_TIMEOUT = 60 * 60 * 1000; // 1 hour

    // started more than 1 hour ago, delete previous lock
    if (fleetSetupLock.attributes.started_at && new Date(fleetSetupLock.attributes.started_at).getTime() < Date.now() - LOCK_TIMEOUT) {
      await deleteLock(soClient);
    } else {
      logger.info('Fleet setup already in progress, abort setup');
      return {
        created: false,
        toReturn: {
          isInitialized: false,
          nonFatalErrors: []
        }
      };
    }
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      logger.debug('Fleet setup lock does not exist, continue setup');
    }
  }
  try {
    created = await soClient.create(_constants2.FLEET_SETUP_LOCK_TYPE, {
      status: 'in_progress',
      uuid: (0, _uuid.v4)(),
      started_at: new Date().toISOString()
    }, {
      id: _constants2.FLEET_SETUP_LOCK_TYPE
    });
    logger.debug(`Fleet setup lock created: ${JSON.stringify(created)}`);
  } catch (error) {
    logger.info(`Could not create fleet setup lock, abort setup: ${error}`);
    return {
      created: false,
      toReturn: {
        isInitialized: false,
        nonFatalErrors: []
      }
    };
  }
  return {
    created: !!created
  };
}
async function deleteLock(soClient) {
  const logger = _app_context.appContextService.getLogger();
  try {
    await soClient.delete(_constants2.FLEET_SETUP_LOCK_TYPE, _constants2.FLEET_SETUP_LOCK_TYPE, {
      refresh: true
    });
    logger.debug(`Fleet setup lock deleted`);
  } catch (error) {
    // ignore 404 errors
    if (error.statusCode !== 404) {
      logger.error('Could not delete fleet setup lock', error);
    }
  }
}
async function createSetupSideEffects(soClient, esClient) {
  var _appContextService$ge, _stepSpan, _stepSpan2, _stepSpan3, _stepSpan4, _appContextService$ge2, _stepSpan5, _stepSpan6, _stepSpan7;
  const logger = _app_context.appContextService.getLogger();
  logger.info('Beginning fleet setup');
  await (0, _clean_old_fleet_indices.cleanUpOldFileIndices)(esClient, logger);
  await ensureFleetDirectories();
  const {
    agentPolicies: policiesOrUndefined,
    packages: packagesOrUndefined
  } = (_appContextService$ge = _app_context.appContextService.getConfig()) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : {};
  const policies = policiesOrUndefined !== null && policiesOrUndefined !== void 0 ? policiesOrUndefined : [];
  let packages = packagesOrUndefined !== null && packagesOrUndefined !== void 0 ? packagesOrUndefined : [];
  logger.debug('Setting Fleet server config');
  await (0, _fleet_server_host.migrateSettingsToFleetServerHost)(soClient);
  logger.debug('Setting up Fleet download source');
  const defaultDownloadSource = await _download_source.downloadSourceService.ensureDefault(soClient);
  // Need to be done before outputs and fleet server hosts as these object can reference a proxy
  logger.debug('Setting up Proxy');
  await (0, _fleet_proxies.ensurePreconfiguredFleetProxies)(soClient, esClient, (0, _fleet_proxies.getPreconfiguredFleetProxiesFromConfig)(_app_context.appContextService.getConfig()));
  logger.debug('Setting up Fleet Sever Hosts');
  await (0, _fleet_server_host2.ensurePreconfiguredFleetServerHosts)(soClient, esClient, (0, _fleet_server_host2.getPreconfiguredFleetServerHostFromConfig)(_app_context.appContextService.getConfig()));
  logger.debug('Setting up Fleet outputs');
  await Promise.all([(0, _outputs.ensurePreconfiguredOutputs)(soClient, esClient, (0, _outputs.getPreconfiguredOutputFromConfig)(_app_context.appContextService.getConfig())), _.settingsService.settingsSetup(soClient)]);
  const defaultOutput = await _output.outputService.ensureDefaultOutput(soClient, esClient);
  logger.debug('Backfilling output performance presets');
  await _output.outputService.backfillAllOutputPresets(soClient, esClient);
  logger.debug('Setting up Fleet Elasticsearch assets');
  let stepSpan = _elasticApmNode.default.startSpan('Install Fleet global assets', 'preconfiguration');
  await ensureFleetGlobalEsAssets(soClient, esClient);
  (_stepSpan = stepSpan) === null || _stepSpan === void 0 ? void 0 : _stepSpan.end();

  // Ensure that required packages are always installed even if they're left out of the config
  const preconfiguredPackageNames = new Set(packages.map(pkg => pkg.name));
  const autoUpdateablePackages = (0, _lodash.compact)(await Promise.all(_constants2.AUTO_UPDATE_PACKAGES.map(pkg => (0, _install3.isPackageInstalled)({
    savedObjectsClient: soClient,
    pkgName: pkg.name
  }).then(installed => installed ? pkg : undefined))));
  packages = [...packages, ...autoUpdateablePackages.filter(pkg => !preconfiguredPackageNames.has(pkg.name))];
  logger.debug('Setting up initial Fleet packages');
  stepSpan = _elasticApmNode.default.startSpan('Install preconfigured packages and policies', 'preconfiguration');
  const {
    nonFatalErrors: preconfiguredPackagesNonFatalErrors
  } = await (0, _preconfiguration.ensurePreconfiguredPackagesAndPolicies)(soClient, esClient, policies, packages, defaultOutput, defaultDownloadSource, _constants.DEFAULT_SPACE_ID);
  (_stepSpan2 = stepSpan) === null || _stepSpan2 === void 0 ? void 0 : _stepSpan2.end();
  stepSpan = _elasticApmNode.default.startSpan('Upgrade managed package policies', 'preconfiguration');
  const packagePolicyUpgradeErrors = (await (0, _managed_package_policies.upgradeManagedPackagePolicies)(soClient, esClient)).filter(result => {
    var _result$errors;
    return ((_result$errors = result.errors) !== null && _result$errors !== void 0 ? _result$errors : []).length > 0;
  });
  (_stepSpan3 = stepSpan) === null || _stepSpan3 === void 0 ? void 0 : _stepSpan3.end();
  logger.debug('Upgrade Fleet package install versions');
  stepSpan = _elasticApmNode.default.startSpan('Upgrade package install format version', 'preconfiguration');
  await (0, _upgrade_package_install_version.upgradePackageInstallVersion)({
    soClient,
    esClient,
    logger
  });
  (_stepSpan4 = stepSpan) === null || _stepSpan4 === void 0 ? void 0 : _stepSpan4.end();
  logger.debug('Generating key pair for message signing');
  stepSpan = _elasticApmNode.default.startSpan('Configure message signing', 'preconfiguration');
  if (!((_appContextService$ge2 = _app_context.appContextService.getMessageSigningService()) !== null && _appContextService$ge2 !== void 0 && _appContextService$ge2.isEncryptionAvailable)) {
    logger.warn('xpack.encryptedSavedObjects.encryptionKey is not configured, private key passphrase is being stored in plain text');
  }
  let messageSigningServiceNonFatalError;
  try {
    var _appContextService$ge3;
    await ((_appContextService$ge3 = _app_context.appContextService.getMessageSigningService()) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.generateKeyPair());
  } catch (error) {
    if (error instanceof _errors.MessageSigningError) {
      messageSigningServiceNonFatalError = {
        error
      };
    } else {
      throw error;
    }
  }
  (_stepSpan5 = stepSpan) === null || _stepSpan5 === void 0 ? void 0 : _stepSpan5.end();
  stepSpan = _elasticApmNode.default.startSpan('Upgrade agent policy schema', 'preconfiguration');
  logger.debug('Upgrade Agent policy schema version');
  await (0, _upgrade_agent_policy_schema_version.upgradeAgentPolicySchemaVersion)(soClient);
  (_stepSpan6 = stepSpan) === null || _stepSpan6 === void 0 ? void 0 : _stepSpan6.end();
  stepSpan = _elasticApmNode.default.startSpan('Set up enrollment keys for preconfigured policies', 'preconfiguration');
  logger.debug('Setting up Fleet enrollment keys and verifying fleet server policies are not out-of-sync');
  await (0, _fleet_server_policies_enrollment_keys.ensureAgentPoliciesFleetServerKeysAndPolicies)({
    soClient,
    esClient,
    logger
  });
  (_stepSpan7 = stepSpan) === null || _stepSpan7 === void 0 ? void 0 : _stepSpan7.end();
  const nonFatalErrors = [...preconfiguredPackagesNonFatalErrors, ...packagePolicyUpgradeErrors, ...(messageSigningServiceNonFatalError ? [messageSigningServiceNonFatalError] : [])];
  if (nonFatalErrors.length > 0) {
    logger.info('Encountered non fatal errors during Fleet setup');
    formatNonFatalErrors(nonFatalErrors).map(e => JSON.stringify(e)).forEach(error => {
      logger.info(error);
      _elasticApmNode.default.captureError(error);
    });
  }
  logger.info('Fleet setup completed');
  return {
    isInitialized: true,
    nonFatalErrors
  };
}

/**
 * Ensure ES assets shared by all Fleet index template are installed
 */
async function ensureFleetGlobalEsAssets(soClient, esClient) {
  const logger = _app_context.appContextService.getLogger();
  // Ensure Global Fleet ES assets are installed
  logger.debug('Creating Fleet component template and ingest pipeline');
  const globalAssetsRes = await Promise.all([(0, _install2.ensureDefaultComponentTemplates)(esClient, logger),
  // returns an array
  (0, _install.ensureFleetFinalPipelineIsInstalled)(esClient, logger)]);
  const assetResults = globalAssetsRes.flat();
  if (assetResults.some(asset => asset.isCreated)) {
    // Update existing index template
    const installedPackages = await (0, _packages.getInstallations)(soClient);
    await (0, _pMap.default)(installedPackages.saved_objects, async ({
      attributes: installation
    }) => {
      await (0, _packages.reinstallPackageForInstallation)({
        soClient,
        esClient,
        installation
      }).catch(err => {
        _elasticApmNode.default.captureError(err);
        logger.error(`Package needs to be manually reinstalled ${installation.name} after installing Fleet global assets: ${err.message}`);
      });
    }, {
      concurrency: 10
    });
  }
}

/**
 * Maps the `nonFatalErrors` object returned by the setup process to a more readable
 * and predictable format suitable for logging output or UI presentation.
 */
function formatNonFatalErrors(nonFatalErrors) {
  return nonFatalErrors.flatMap(e => {
    if ('error' in e) {
      return {
        name: e.error.name,
        message: e.error.message
      };
    } else if ('errors' in e) {
      return e.errors.map(upgradePackagePolicyError => {
        if (typeof upgradePackagePolicyError === 'string') {
          return {
            name: 'SetupNonFatalError',
            message: upgradePackagePolicyError
          };
        }
        return {
          name: upgradePackagePolicyError.key,
          message: upgradePackagePolicyError.message
        };
      });
    }
  });
}

/**
 * Confirm existence of various directories used by Fleet and warn if they don't exist
 */
async function ensureFleetDirectories() {
  var _config$developer;
  const logger = _app_context.appContextService.getLogger();
  const config = _app_context.appContextService.getConfig();
  const bundledPackageLocation = config === null || config === void 0 ? void 0 : (_config$developer = config.developer) === null || _config$developer === void 0 ? void 0 : _config$developer.bundledPackageLocation;
  const registryUrl = (0, _.getRegistryUrl)();
  if (!bundledPackageLocation) {
    logger.warn('xpack.fleet.developer.bundledPackageLocation is not configured');
    return;
  }
  try {
    await _promises.default.stat(bundledPackageLocation);
    logger.debug(`Bundled package directory ${bundledPackageLocation} exists`);
  } catch (error) {
    logger.warn(`Bundled package directory ${bundledPackageLocation} does not exist. All packages will be sourced from ${registryUrl}.`);
  }
}