"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._stateMachineInstallPackage = _stateMachineInstallPackage;
var _server = require("@kbn/core/server");
var _errors = require("../../../../errors");
var _types = require("../../../../../common/types");
var _steps = require("./steps");
var _state_machine = require("./state_machine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * _stateMachineInstallPackage installs packages using the generic state machine in ./state_machine
 * installStates is the data structure providing the state machine definition
 * Usually the install process starts with `create_restart_installation` and continues based on nextState parameter in the definition
 * The `onTransition` functions are the steps executed to go from one state to another, and all accept an `InstallContext` object as input parameter
 * After each transition `updateLatestExecutedState` is executed, it updates the executed state in the SO
 */
async function _stateMachineInstallPackage(context) {
  const installStates = {
    context,
    states: {
      create_restart_installation: {
        nextState: 'install_kibana_assets',
        onTransition: _steps.stepCreateRestartInstallation,
        onPostTransition: _steps.updateLatestExecutedState
      },
      install_kibana_assets: {
        onTransition: _steps.stepInstallKibanaAssets,
        nextState: 'install_ilm_policies',
        onPostTransition: _steps.updateLatestExecutedState
      },
      install_ilm_policies: {
        onTransition: _steps.stepInstallILMPolicies,
        nextState: 'install_ml_model',
        onPostTransition: _steps.updateLatestExecutedState
      },
      install_ml_model: {
        onTransition: _steps.stepInstallMlModel,
        nextState: 'install_index_template_pipelines',
        onPostTransition: _steps.updateLatestExecutedState
      },
      install_index_template_pipelines: {
        onTransition: _steps.stepInstallIndexTemplatePipelines,
        nextState: 'remove_legacy_templates',
        onPostTransition: _steps.updateLatestExecutedState
      },
      remove_legacy_templates: {
        onTransition: _steps.stepRemoveLegacyTemplates,
        nextState: 'update_current_write_indices',
        onPostTransition: _steps.updateLatestExecutedState
      },
      update_current_write_indices: {
        onTransition: _steps.stepUpdateCurrentWriteIndices,
        nextState: 'install_transforms',
        onPostTransition: _steps.updateLatestExecutedState
      },
      install_transforms: {
        onTransition: _steps.stepInstallTransforms,
        nextState: 'delete_previous_pipelines',
        onPostTransition: _steps.updateLatestExecutedState
      },
      delete_previous_pipelines: {
        onTransition: _steps.stepDeletePreviousPipelines,
        nextState: 'save_archive_entries_from_assets_map',
        onPostTransition: _steps.updateLatestExecutedState
      },
      save_archive_entries_from_assets_map: {
        onTransition: _steps.stepSaveArchiveEntries,
        nextState: 'resolve_kibana_promise',
        onPostTransition: _steps.updateLatestExecutedState
      },
      resolve_kibana_promise: {
        onTransition: _steps.stepResolveKibanaPromise,
        nextState: 'update_so',
        onPostTransition: _steps.updateLatestExecutedState
      },
      update_so: {
        onTransition: _steps.stepSaveSystemObject,
        nextState: 'end',
        onPostTransition: _steps.updateLatestExecutedState
      }
    }
  };
  try {
    const {
      installedKibanaAssetsRefs,
      esReferences
    } = await (0, _state_machine.handleState)(_types.INSTALL_STATES.CREATE_RESTART_INSTALLATION, installStates, installStates.context);
    return [...installedKibanaAssetsRefs, ...esReferences];
  } catch (err) {
    const {
      packageInfo
    } = installStates.context.packageInstallContext;
    const {
      name: pkgName,
      version: pkgVersion
    } = packageInfo;
    if (_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      throw new _errors.PackageSavedObjectConflictError(`Saved Object conflict encountered while installing ${pkgName || 'unknown'}-${pkgVersion || 'unknown'}. There may be a conflicting Saved Object saved to another Space. Original error: ${err.message}`);
    } else {
      throw err;
    }
  }
}