"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UPLOAD_RETRY_AFTER_MS = void 0;
exports.createInstallation = createInstallation;
exports.ensureInstalledPackage = ensureInstalledPackage;
exports.ensurePackagesCompletedInstall = ensurePackagesCompletedInstall;
exports.getInstallType = getInstallType;
exports.handleInstallPackageFailure = handleInstallPackageFailure;
exports.installAssetsForInputPackagePolicy = installAssetsForInputPackagePolicy;
exports.installCustomPackage = installCustomPackage;
exports.installPackage = installPackage;
exports.isPackageInstalled = isPackageInstalled;
exports.isPackageVersionOrLaterInstalled = isPackageVersionOrLaterInstalled;
exports.kibanaAssetsToAssetsRef = void 0;
exports.restartInstallation = restartInstallation;
exports.updateVersion = exports.updateInstallStatusToFailed = exports.saveKibanaAssetsRefs = void 0;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _i18n = require("@kbn/i18n");
var _lt = _interopRequireDefault(require("semver/functions/lt"));
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _services = require("../../../../common/services");
var _fleet_es_assets = require("../../../constants/fleet_es_assets");
var _template = require("../elasticsearch/template/template");
var _constants2 = require("../../../../common/constants");
var _errors = require("../../../errors");
var _constants3 = require("../../../constants");
var _ = require("../..");
var _app_context = require("../../app_context");
var Registry = _interopRequireWildcard(require("../registry"));
var _archive = require("../archive");
var _install = require("../kibana/assets/install");
var _upgrade_sender = require("../../upgrade_sender");
var _audit_logging = require("../../audit_logging");
var _filtered_packages = require("../filtered_packages");
var _state_machine_package_install = require("./install_state_machine/_state_machine_package_install");
var _package_verification = require("./package_verification");
var _get = require("./get");
var _remove = require("./remove");
var _install_package = require("./_install_package");
var _cleanup = require("./cleanup");
var _bundled_packages = require("./bundled_packages");
var _utils = require("./custom_integrations/utils");
var _constants4 = require("./custom_integrations/constants");
var _custom_integrations = require("./custom_integrations");
var _utils2 = require("./custom_integrations/assets/dataset/utils");
var _check_naming_collision = require("./custom_integrations/validation/check_naming_collision");
var _check_dataset_name_format = require("./custom_integrations/validation/check_dataset_name_format");
var _install_errors_helpers = require("./install_errors_helpers");
var _install_index_template_pipeline = require("./install_index_template_pipeline");
var _es_assets_reference = require("./es_assets_reference");
var _utils3 = require("./utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UPLOAD_RETRY_AFTER_MS = exports.UPLOAD_RETRY_AFTER_MS = 10000; // 10s
const MAX_ENSURE_INSTALL_TIME = 60 * 1000;
async function isPackageInstalled(options) {
  const installedPackage = await (0, _get.getInstallation)(options);
  return installedPackage !== undefined;
}

// Error used to retry in isPackageVersionOrLaterInstalled
class CurrentlyInstallingError extends Error {}

/**
 * Check if a package is currently installed,
 * if the package is currently installing it will retry until MAX_ENSURE_INSTALL_TIME is reached
 */
async function isPackageVersionOrLaterInstalled(options) {
  return (0, _pRetry.default)(async () => {
    const {
      savedObjectsClient,
      pkgName,
      pkgVersion
    } = options;
    const installedPackageObject = await (0, _get.getInstallationObject)({
      savedObjectsClient,
      pkgName
    });
    const installedPackage = installedPackageObject === null || installedPackageObject === void 0 ? void 0 : installedPackageObject.attributes;
    if (installedPackage && (installedPackage.version === pkgVersion || (0, _lt.default)(pkgVersion, installedPackage.version))) {
      if (installedPackage.install_status === 'installing') {
        throw new CurrentlyInstallingError(`Package ${pkgName}-${pkgVersion} is currently installing`);
      } else if (installedPackage.install_status === 'install_failed') {
        return false;
      }
      return {
        package: installedPackage
      };
    }
    return false;
  }, {
    maxRetryTime: MAX_ENSURE_INSTALL_TIME,
    onFailedAttempt: error => {
      if (!(error instanceof CurrentlyInstallingError)) {
        throw error;
      }
    }
  }).catch(err => {
    if (err instanceof CurrentlyInstallingError) {
      return false;
    }
    throw err;
  });
}
async function ensureInstalledPackage(options) {
  const {
    savedObjectsClient,
    pkgName,
    esClient,
    pkgVersion,
    force = false,
    spaceId = _constants.DEFAULT_SPACE_ID,
    authorizationHeader
  } = options;

  // If pkgVersion isn't specified, find the latest package version
  const pkgKeyProps = pkgVersion ? {
    name: pkgName,
    version: pkgVersion
  } : await Registry.fetchFindLatestPackageOrThrow(pkgName, {
    prerelease: true
  });
  const installedPackageResult = await isPackageVersionOrLaterInstalled({
    savedObjectsClient,
    pkgName: pkgKeyProps.name,
    pkgVersion: pkgKeyProps.version
  });
  if (installedPackageResult) {
    return installedPackageResult.package;
  }
  const pkgkey = Registry.pkgToPkgKey(pkgKeyProps);
  const installResult = await installPackage({
    installSource: 'registry',
    savedObjectsClient,
    pkgkey,
    spaceId,
    esClient,
    neverIgnoreVerificationError: !force,
    force: true,
    // Always force outdated packages to be installed if a later version isn't installed
    authorizationHeader
  });
  if (installResult.error) {
    const errorPrefix = installResult.installType === 'update' || installResult.installType === 'reupdate' ? _i18n.i18n.translate('xpack.fleet.epm.install.packageUpdateError', {
      defaultMessage: 'Error updating {pkgName} to {pkgVersion}',
      values: {
        pkgName: pkgKeyProps.name,
        pkgVersion: pkgKeyProps.version
      }
    }) : _i18n.i18n.translate('xpack.fleet.epm.install.packageInstallError', {
      defaultMessage: 'Error installing {pkgName} {pkgVersion}',
      values: {
        pkgName: pkgKeyProps.name,
        pkgVersion: pkgKeyProps.version
      }
    });
    installResult.error.message = `${errorPrefix}: ${installResult.error.message}`;
    throw installResult.error;
  }
  const installation = await (0, _get.getInstallation)({
    savedObjectsClient,
    pkgName
  });
  if (!installation) throw new _errors.FleetError(`Could not get installation for ${pkgName}`);
  return installation;
}
async function handleInstallPackageFailure({
  savedObjectsClient,
  error,
  pkgName,
  pkgVersion,
  installedPkg,
  esClient,
  spaceId,
  authorizationHeader
}) {
  if (error instanceof _errors.ConcurrentInstallOperationError) {
    return;
  }
  const logger = _app_context.appContextService.getLogger();
  const pkgkey = Registry.pkgToPkgKey({
    name: pkgName,
    version: pkgVersion
  });
  const latestInstallFailedAttempts = (0, _install_errors_helpers.addErrorToLatestFailedAttempts)({
    error,
    targetVersion: pkgVersion,
    createdAt: new Date().toISOString(),
    latestAttempts: installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.latest_install_failed_attempts
  });

  // if there is an unknown server error, uninstall any package assets or reinstall the previous version if update
  try {
    const installType = getInstallType({
      pkgVersion,
      installedPkg
    });
    if (installType === 'install') {
      logger.error(`uninstalling ${pkgkey} after error installing: [${error.toString()}]`);
      await (0, _remove.removeInstallation)({
        savedObjectsClient,
        pkgName,
        pkgVersion,
        esClient
      });
      return;
    }
    await updateInstallStatusToFailed({
      logger,
      savedObjectsClient,
      pkgName,
      status: 'install_failed',
      latestInstallFailedAttempts
    });
    if (installType === 'reinstall') {
      logger.error(`Failed to reinstall ${pkgkey}: [${error.toString()}]`, {
        error
      });
    }
    if (installType === 'update') {
      if (!installedPkg) {
        logger.error(`failed to rollback package after installation error ${error} because saved object was undefined`);
        return;
      }
      const prevVersion = `${pkgName}-${installedPkg.attributes.version}`;
      logger.error(`rolling back to ${prevVersion} after error installing ${pkgkey}`);
      await installPackage({
        installSource: 'registry',
        savedObjectsClient,
        pkgkey: prevVersion,
        esClient,
        spaceId,
        force: true,
        authorizationHeader
      });
    }
  } catch (e) {
    // If an error happens while removing the integration or while doing a rollback update the status to failed
    await updateInstallStatusToFailed({
      logger,
      savedObjectsClient,
      pkgName,
      status: 'install_failed',
      latestInstallFailedAttempts: installedPkg ? (0, _install_errors_helpers.addErrorToLatestFailedAttempts)({
        error: e,
        targetVersion: installedPkg.attributes.version,
        createdAt: installedPkg.attributes.install_started_at,
        latestAttempts: latestInstallFailedAttempts
      }) : []
    });
    logger.error(`failed to uninstall or rollback package after installation error ${e}`);
  }
}
function getTelemetryEvent(pkgName, pkgVersion) {
  return {
    packageName: pkgName,
    currentVersion: 'unknown',
    newVersion: pkgVersion,
    status: 'failure',
    dryRun: false,
    eventType: _upgrade_sender.UpdateEventType.PACKAGE_INSTALL,
    installType: 'unknown'
  };
}
function sendEvent(telemetryEvent) {
  (0, _upgrade_sender.sendTelemetryEvents)(_app_context.appContextService.getLogger(), _app_context.appContextService.getTelemetryEventsSender(), telemetryEvent);
}
async function installPackageFromRegistry({
  savedObjectsClient,
  pkgkey,
  esClient,
  spaceId,
  authorizationHeader,
  force = false,
  ignoreConstraints = false,
  neverIgnoreVerificationError = false,
  prerelease = false,
  ignoreMappingUpdateErrors = false,
  skipDataStreamRollover = false
}) {
  const logger = _app_context.appContextService.getLogger();
  // TODO: change epm API to /packageName/version so we don't need to do this
  const {
    pkgName,
    pkgVersion: version
  } = Registry.splitPkgKey(pkgkey);
  let pkgVersion = version !== null && version !== void 0 ? version : '';

  // if an error happens during getInstallType, report that we don't know
  let installType = 'unknown';
  const installSource = 'registry';
  const telemetryEvent = getTelemetryEvent(pkgName, pkgVersion);
  try {
    // get the currently installed package

    const installedPkg = await (0, _get.getInstallationObject)({
      savedObjectsClient,
      pkgName
    });
    installType = getInstallType({
      pkgVersion,
      installedPkg
    });
    telemetryEvent.installType = installType;
    telemetryEvent.currentVersion = (installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.version) || 'not_installed';
    const queryLatest = () => Registry.fetchFindLatestPackageOrThrow(pkgName, {
      ignoreConstraints,
      prerelease: prerelease === true || (0, _services.isPackagePrerelease)(pkgVersion) // fetching latest GA version if the package to install is GA, so that it is allowed to install
    });
    let latestPkg;
    // fetching latest package first to set the version
    if (!pkgVersion) {
      latestPkg = await queryLatest();
      pkgVersion = latestPkg.version;
    }

    // get latest package version and requested version in parallel for performance
    const [latestPackage, {
      paths,
      packageInfo,
      assetsMap,
      verificationResult
    }] = await Promise.all([latestPkg ? Promise.resolve(latestPkg) : queryLatest(), Registry.getPackage(pkgName, pkgVersion, {
      ignoreUnverified: force && !neverIgnoreVerificationError
    })]);
    const packageInstallContext = {
      packageInfo,
      assetsMap,
      paths
    };

    // let the user install if using the force flag or needing to reinstall or install a previous version due to failed update
    const installOutOfDateVersionOk = force || ['reinstall', 'reupdate', 'rollback'].includes(installType);

    // if the requested version is out-of-date of the latest package version, check if we allow it
    // if we don't allow it, return an error
    if ((0, _lt.default)(pkgVersion, latestPackage.version)) {
      if (!installOutOfDateVersionOk) {
        throw new _errors.PackageOutdatedError(`${pkgkey} is out-of-date and cannot be installed or updated`);
      }
      logger.debug(`${pkgkey} is out-of-date, installing anyway due to ${force ? 'force flag' : `install type ${installType}`}`);
    }
    const {
      enablePackagesStateMachine
    } = _app_context.appContextService.getExperimentalFeatures();
    if (enablePackagesStateMachine) {
      return await installPackageWitStateMachine({
        pkgName,
        pkgVersion,
        installSource,
        installedPkg,
        installType,
        savedObjectsClient,
        esClient,
        spaceId,
        force,
        packageInstallContext,
        paths,
        verificationResult,
        authorizationHeader,
        ignoreMappingUpdateErrors,
        skipDataStreamRollover
      });
    } else {
      return await installPackageCommon({
        pkgName,
        pkgVersion,
        installSource,
        installedPkg,
        installType,
        savedObjectsClient,
        esClient,
        spaceId,
        force,
        packageInstallContext,
        paths,
        verificationResult,
        authorizationHeader,
        ignoreMappingUpdateErrors,
        skipDataStreamRollover
      });
    }
  } catch (e) {
    sendEvent({
      ...telemetryEvent,
      errorMessage: e.message
    });
    return {
      error: e,
      installType,
      installSource
    };
  }
}
function getElasticSubscription(packageInfo) {
  var _packageInfo$conditio, _packageInfo$conditio2;
  const subscription = (_packageInfo$conditio = packageInfo.conditions) === null || _packageInfo$conditio === void 0 ? void 0 : (_packageInfo$conditio2 = _packageInfo$conditio.elastic) === null || _packageInfo$conditio2 === void 0 ? void 0 : _packageInfo$conditio2.subscription;
  // Keep packageInfo.license for backward compatibility
  return subscription || packageInfo.license || 'basic';
}
async function installPackageCommon(options) {
  const packageInfo = options.packageInstallContext.packageInfo;
  const {
    pkgName,
    pkgVersion,
    installSource,
    installedPkg,
    installType,
    savedObjectsClient,
    force,
    esClient,
    spaceId,
    verificationResult,
    authorizationHeader,
    ignoreMappingUpdateErrors,
    skipDataStreamRollover,
    packageInstallContext
  } = options;
  let {
    telemetryEvent
  } = options;
  const logger = _app_context.appContextService.getLogger();
  logger.info(`Install - Starting installation of ${pkgName}@${pkgVersion} from ${installSource} `);

  // Workaround apm issue with async spans: https://github.com/elastic/apm-agent-nodejs/issues/2611
  await Promise.resolve();
  const span = _elasticApmNode.default.startSpan(`Install package from ${installSource} ${pkgName}@${pkgVersion}`, 'package');
  if (!telemetryEvent) {
    telemetryEvent = getTelemetryEvent(pkgName, pkgVersion);
    telemetryEvent.installType = installType;
    telemetryEvent.currentVersion = (installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.version) || 'not_installed';
  }
  try {
    span === null || span === void 0 ? void 0 : span.addLabels({
      packageName: pkgName,
      packageVersion: pkgVersion,
      installType
    });
    const filteredPackages = (0, _filtered_packages.getFilteredInstallPackages)();
    if (filteredPackages.includes(pkgName)) {
      throw new _errors.FleetUnauthorizedError(`${pkgName} installation is not authorized`);
    }

    // if the requested version is the same as installed version, check if we allow it based on
    // current installed package status and force flag, if we don't allow it,
    // just return the asset references from the existing installation
    if ((installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.version) === pkgVersion && (installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.install_status) === 'installed') {
      if (!force) {
        logger.debug(`${pkgName}-${pkgVersion} is already installed, skipping installation`);
        return {
          assets: [...installedPkg.attributes.installed_es, ...installedPkg.attributes.installed_kibana],
          status: 'already_installed',
          installType,
          installSource
        };
      }
    }
    const elasticSubscription = getElasticSubscription(packageInfo);
    if (!_.licenseService.hasAtLeast(elasticSubscription)) {
      logger.error(`Installation requires ${elasticSubscription} license`);
      const err = new _errors.FleetError(`Installation requires ${elasticSubscription} license`);
      sendEvent({
        ...telemetryEvent,
        errorMessage: err.message
      });
      return {
        error: err,
        installType,
        installSource
      };
    }

    // try installing the package, if there was an error, call error handler and rethrow
    return await (0, _install_package._installPackage)({
      savedObjectsClient,
      esClient,
      logger,
      installedPkg,
      packageInstallContext,
      installType,
      spaceId,
      verificationResult,
      installSource,
      authorizationHeader,
      force,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    }).then(async assets => {
      logger.debug(`Removing old assets from previous versions of ${pkgName}`);
      await (0, _cleanup.removeOldAssets)({
        soClient: savedObjectsClient,
        pkgName: packageInfo.name,
        currentVersion: packageInfo.version
      });
      sendEvent({
        ...telemetryEvent,
        status: 'success'
      });
      return {
        assets,
        status: 'installed',
        installType,
        installSource
      };
    }).catch(async err => {
      logger.warn(`Failure to install package [${pkgName}]: [${err.toString()}]`, {
        error: {
          stack_trace: err.stack
        }
      });
      await handleInstallPackageFailure({
        savedObjectsClient,
        error: err,
        pkgName,
        pkgVersion,
        installedPkg,
        spaceId,
        esClient,
        authorizationHeader
      });
      sendEvent({
        ...telemetryEvent,
        errorMessage: err.message
      });
      return {
        error: err,
        installType,
        installSource
      };
    });
  } catch (e) {
    sendEvent({
      ...telemetryEvent,
      errorMessage: e.message
    });
    return {
      error: e,
      installType,
      installSource
    };
  } finally {
    span === null || span === void 0 ? void 0 : span.end();
  }
}
async function installPackageWitStateMachine(options) {
  const packageInfo = options.packageInstallContext.packageInfo;
  const {
    pkgName,
    pkgVersion,
    installSource,
    installedPkg,
    installType,
    savedObjectsClient,
    force,
    esClient,
    spaceId,
    verificationResult,
    authorizationHeader,
    ignoreMappingUpdateErrors,
    skipDataStreamRollover,
    packageInstallContext
  } = options;
  let {
    telemetryEvent
  } = options;
  const logger = _app_context.appContextService.getLogger();
  logger.info(`Install with enablePackagesStateMachine - Starting installation of ${pkgName}@${pkgVersion} from ${installSource} `);

  // Workaround apm issue with async spans: https://github.com/elastic/apm-agent-nodejs/issues/2611
  await Promise.resolve();
  const span = _elasticApmNode.default.startSpan(`Install package from ${installSource} ${pkgName}@${pkgVersion}`, 'package');
  if (!telemetryEvent) {
    telemetryEvent = getTelemetryEvent(pkgName, pkgVersion);
    telemetryEvent.installType = installType;
    telemetryEvent.currentVersion = (installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.version) || 'not_installed';
  }
  try {
    span === null || span === void 0 ? void 0 : span.addLabels({
      packageName: pkgName,
      packageVersion: pkgVersion,
      installType
    });
    const filteredPackages = (0, _filtered_packages.getFilteredInstallPackages)();
    if (filteredPackages.includes(pkgName)) {
      throw new _errors.FleetUnauthorizedError(`${pkgName} installation is not authorized`);
    }

    // if the requested version is the same as installed version, check if we allow it based on
    // current installed package status and force flag, if we don't allow it,
    // just return the asset references from the existing installation
    if ((installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.version) === pkgVersion && (installedPkg === null || installedPkg === void 0 ? void 0 : installedPkg.attributes.install_status) === 'installed') {
      if (!force) {
        logger.debug(`${pkgName}-${pkgVersion} is already installed, skipping installation`);
        return {
          assets: [...installedPkg.attributes.installed_es, ...installedPkg.attributes.installed_kibana],
          status: 'already_installed',
          installType,
          installSource
        };
      }
    }
    const elasticSubscription = getElasticSubscription(packageInfo);
    if (!_.licenseService.hasAtLeast(elasticSubscription)) {
      logger.error(`Installation requires ${elasticSubscription} license`);
      const err = new _errors.FleetError(`Installation requires ${elasticSubscription} license`);
      sendEvent({
        ...telemetryEvent,
        errorMessage: err.message
      });
      return {
        error: err,
        installType,
        installSource
      };
    }

    // Saved object client need to be scopped with the package space for saved object tagging
    const savedObjectClientWithSpace = _app_context.appContextService.getInternalUserSOClientForSpaceId(spaceId);
    const savedObjectsImporter = _app_context.appContextService.getSavedObjects().createImporter(savedObjectClientWithSpace, {
      importSizeLimit: 15_000
    });
    const savedObjectTagAssignmentService = _app_context.appContextService.getSavedObjectsTagging().createInternalAssignmentService({
      client: savedObjectClientWithSpace
    });
    const savedObjectTagClient = _app_context.appContextService.getSavedObjectsTagging().createTagClient({
      client: savedObjectClientWithSpace
    });

    // try installing the package, if there was an error, call error handler and rethrow
    return await (0, _state_machine_package_install._stateMachineInstallPackage)({
      savedObjectsClient,
      savedObjectsImporter,
      savedObjectTagAssignmentService,
      savedObjectTagClient,
      esClient,
      logger,
      installedPkg,
      packageInstallContext,
      installType,
      spaceId,
      verificationResult,
      installSource,
      authorizationHeader,
      force,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    }).then(async assets => {
      logger.debug(`Removing old assets from previous versions of ${pkgName}`);
      await (0, _cleanup.removeOldAssets)({
        soClient: savedObjectsClient,
        pkgName: packageInfo.name,
        currentVersion: packageInfo.version
      });
      sendEvent({
        ...telemetryEvent,
        status: 'success'
      });
      return {
        assets,
        status: 'installed',
        installType,
        installSource
      };
    }).catch(async err => {
      logger.warn(`Failure to install package [${pkgName}]: [${err.toString()}]`, {
        error: {
          stack_trace: err.stack
        }
      });
      await handleInstallPackageFailure({
        savedObjectsClient,
        error: err,
        pkgName,
        pkgVersion,
        installedPkg,
        spaceId,
        esClient,
        authorizationHeader
      });
      sendEvent({
        ...telemetryEvent,
        errorMessage: err.message
      });
      return {
        error: err,
        installType,
        installSource
      };
    });
  } catch (e) {
    sendEvent({
      ...telemetryEvent,
      errorMessage: e.message
    });
    return {
      error: e,
      installType,
      installSource
    };
  } finally {
    span === null || span === void 0 ? void 0 : span.end();
  }
}
async function installPackageByUpload({
  savedObjectsClient,
  esClient,
  archiveBuffer,
  contentType,
  spaceId,
  version,
  authorizationHeader,
  ignoreMappingUpdateErrors,
  skipDataStreamRollover,
  isBundledPackage,
  skipRateLimitCheck
}) {
  const logger = _app_context.appContextService.getLogger();

  // if an error happens during getInstallType, report that we don't know
  let installType = 'unknown';
  const installSource = isBundledPackage ? 'bundled' : 'upload';
  const timeToWaitString = _moment.default.utc(_moment.default.duration(UPLOAD_RETRY_AFTER_MS).asMilliseconds()).format('s[s]');
  try {
    // Check cached timestamp for rate limiting
    const lastInstalledBy = (0, _utils3.getLastUploadInstallCache)();
    if (lastInstalledBy && !skipRateLimitCheck) {
      const msSinceLastFetched = Date.now() - (lastInstalledBy || 0);
      if (msSinceLastFetched < UPLOAD_RETRY_AFTER_MS) {
        logger.error(`Install by Upload - Too many requests. Wait ${timeToWaitString} before uploading again.`);
        throw new _errors.FleetTooManyRequestsError(`Too many requests. Please wait ${timeToWaitString} before uploading again.`);
      }
    }
    const {
      packageInfo
    } = await (0, _archive.generatePackageInfoFromArchiveBuffer)(archiveBuffer, contentType);
    const pkgName = packageInfo.name;

    // Allow for overriding the version in the manifest for cases where we install
    // stack-aligned bundled packages to support special cases around the
    // `forceAlignStackVersion` flag in `fleet_packages.json`.
    const pkgVersion = version || packageInfo.version;
    const installedPkg = await (0, _get.getInstallationObject)({
      savedObjectsClient,
      pkgName
    });
    installType = getInstallType({
      pkgVersion,
      installedPkg
    });

    // as we do not verify uploaded packages, we must invalidate the verification cache
    (0, _archive.deleteVerificationResult)(packageInfo);
    (0, _archive.setPackageInfo)({
      name: packageInfo.name,
      version: pkgVersion,
      packageInfo
    });
    const {
      assetsMap,
      paths
    } = await (0, _archive.unpackBufferToAssetsMap)({
      name: packageInfo.name,
      version: pkgVersion,
      archiveBuffer,
      contentType
    });
    const packageInstallContext = {
      packageInfo: {
        ...packageInfo,
        version: pkgVersion
      },
      assetsMap,
      paths
    };
    // update the timestamp of latest installation
    (0, _utils3.setLastUploadInstallCache)();
    return await installPackageCommon({
      packageInstallContext,
      pkgName,
      pkgVersion,
      installSource,
      installedPkg,
      installType,
      savedObjectsClient,
      esClient,
      spaceId,
      force: true,
      // upload has implicit force
      paths,
      authorizationHeader,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    });
  } catch (e) {
    return {
      error: e,
      installType,
      installSource
    };
  }
}
async function installPackage(args) {
  if (!('installSource' in args)) {
    throw new _errors.FleetError('installSource is required');
  }
  const logger = _app_context.appContextService.getLogger();
  const {
    savedObjectsClient,
    esClient
  } = args;
  const authorizationHeader = args.authorizationHeader;
  if (args.installSource === 'registry') {
    const {
      pkgkey,
      force,
      ignoreConstraints,
      spaceId,
      neverIgnoreVerificationError,
      prerelease,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    } = args;
    const matchingBundledPackage = await (0, _bundled_packages.getBundledPackageByPkgKey)(pkgkey);
    if (matchingBundledPackage) {
      logger.debug(`Found bundled package for requested install of ${pkgkey} - installing from bundled package archive`);
      const archiveBuffer = await matchingBundledPackage.getBuffer();
      const response = await installPackageByUpload({
        savedObjectsClient,
        esClient,
        archiveBuffer,
        contentType: 'application/zip',
        spaceId,
        version: matchingBundledPackage.version,
        authorizationHeader,
        ignoreMappingUpdateErrors,
        skipDataStreamRollover,
        isBundledPackage: true,
        skipRateLimitCheck: true
      });
      return {
        ...response,
        installSource: 'bundled'
      };
    }
    logger.debug(`Kicking off install of ${pkgkey} from registry`);
    const response = await installPackageFromRegistry({
      savedObjectsClient,
      pkgkey,
      esClient,
      spaceId,
      force,
      neverIgnoreVerificationError,
      ignoreConstraints,
      prerelease,
      authorizationHeader,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    });
    return response;
  } else if (args.installSource === 'upload') {
    const {
      archiveBuffer,
      contentType,
      spaceId,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    } = args;
    logger.debug(`Installing package by upload`);
    const response = await installPackageByUpload({
      savedObjectsClient,
      esClient,
      archiveBuffer,
      contentType,
      spaceId,
      authorizationHeader,
      ignoreMappingUpdateErrors,
      skipDataStreamRollover
    });
    return response;
  } else if (args.installSource === 'custom') {
    const {
      pkgName,
      force,
      datasets,
      spaceId,
      kibanaVersion
    } = args;
    logger.debug(`Kicking off install of custom package ${pkgName}`);
    const response = await installCustomPackage({
      savedObjectsClient,
      pkgName,
      datasets,
      esClient,
      spaceId,
      force,
      authorizationHeader,
      kibanaVersion
    });
    return response;
  }
  throw new _errors.FleetError(`Unknown installSource: ${args.installSource}`);
}
async function installCustomPackage(args) {
  var _authorizationHeader$;
  const {
    savedObjectsClient,
    esClient,
    spaceId,
    pkgName,
    force,
    authorizationHeader,
    datasets,
    kibanaVersion
  } = args;

  // Validate that we can create this package, validations will throw if they don't pass
  await (0, _check_naming_collision.checkForNamingCollision)(savedObjectsClient, pkgName);
  (0, _check_dataset_name_format.checkDatasetsNameFormat)(datasets, pkgName);

  // Compose a packageInfo
  const packageInfo = {
    format_version: _constants2.CUSTOM_INTEGRATION_PACKAGE_SPEC_VERSION,
    name: pkgName,
    title: (0, _utils.convertStringToTitle)(pkgName),
    description: (0, _utils.generateDescription)(datasets.map(dataset => dataset.name)),
    version: _constants4.INITIAL_VERSION,
    owner: {
      github: (_authorizationHeader$ = authorizationHeader === null || authorizationHeader === void 0 ? void 0 : authorizationHeader.username) !== null && _authorizationHeader$ !== void 0 ? _authorizationHeader$ : 'unknown'
    },
    type: 'integration',
    data_streams: (0, _utils2.generateDatastreamEntries)(datasets, pkgName)
  };
  const assets = (0, _custom_integrations.createAssets)({
    ...packageInfo,
    kibanaVersion,
    datasets
  });
  const assetsMap = assets.reduce((acc, asset) => {
    acc.set(asset.path, asset.content);
    return acc;
  }, new Map());
  const paths = [...assetsMap.keys()];
  const packageInstallContext = {
    assetsMap,
    paths,
    packageInfo
  };
  return await installPackageCommon({
    packageInstallContext,
    pkgName,
    pkgVersion: _constants4.INITIAL_VERSION,
    installSource: 'custom',
    installType: 'install',
    savedObjectsClient,
    esClient,
    spaceId,
    force,
    paths,
    authorizationHeader
  });
}
const updateVersion = async (savedObjectsClient, pkgName, pkgVersion) => {
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'update',
    id: pkgName,
    savedObjectType: _constants3.PACKAGES_SAVED_OBJECT_TYPE
  });
  return savedObjectsClient.update(_constants3.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
    version: pkgVersion
  });
};
exports.updateVersion = updateVersion;
const updateInstallStatusToFailed = async ({
  logger,
  savedObjectsClient,
  pkgName,
  status,
  latestInstallFailedAttempts
}) => {
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'update',
    id: pkgName,
    savedObjectType: _constants3.PACKAGES_SAVED_OBJECT_TYPE
  });
  try {
    return await savedObjectsClient.update(_constants3.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
      install_status: status,
      latest_install_failed_attempts: latestInstallFailedAttempts
    });
  } catch (err) {
    if (!_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      logger.error(`failed to update package status to: install_failed  ${err}`);
    }
  }
};
exports.updateInstallStatusToFailed = updateInstallStatusToFailed;
async function restartInstallation(options) {
  const {
    savedObjectsClient,
    pkgVersion,
    pkgName,
    installSource,
    verificationResult
  } = options;
  let savedObjectUpdate = {
    install_version: pkgVersion,
    install_status: 'installing',
    install_started_at: new Date().toISOString(),
    install_source: installSource
  };
  if (verificationResult) {
    savedObjectUpdate = {
      ...savedObjectUpdate,
      verification_key_id: null,
      // unset any previous verification key id
      ...(0, _package_verification.formatVerificationResultForSO)(verificationResult)
    };
  }
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'update',
    id: pkgName,
    savedObjectType: _constants3.PACKAGES_SAVED_OBJECT_TYPE
  });
  await savedObjectsClient.update(_constants3.PACKAGES_SAVED_OBJECT_TYPE, pkgName, savedObjectUpdate);
}
async function createInstallation(options) {
  const {
    savedObjectsClient,
    packageInfo,
    installSource,
    verificationResult
  } = options;
  const {
    name: pkgName,
    version: pkgVersion
  } = packageInfo;
  const toSaveESIndexPatterns = (0, _template.generateESIndexPatterns)((0, _services.getNormalizedDataStreams)(packageInfo, _constants2.GENERIC_DATASET_NAME));

  // For "stack-aligned" packages, default the `keep_policies_up_to_date` setting to true. For all other
  // packages, default it to undefined. Use undefined rather than false to allow us to differentiate
  // between "unset" and "user explicitly disabled".
  const defaultKeepPoliciesUpToDate = _constants2.AUTO_UPGRADE_POLICIES_PACKAGES.some(({
    name
  }) => name === packageInfo.name) ? true : undefined;
  let savedObject = {
    installed_kibana: [],
    installed_kibana_space_id: options.spaceId,
    installed_es: [],
    package_assets: [],
    es_index_patterns: toSaveESIndexPatterns,
    name: pkgName,
    version: pkgVersion,
    install_version: pkgVersion,
    install_status: 'installing',
    install_started_at: new Date().toISOString(),
    install_source: installSource,
    install_format_schema_version: _fleet_es_assets.FLEET_INSTALL_FORMAT_VERSION,
    keep_policies_up_to_date: defaultKeepPoliciesUpToDate,
    verification_status: 'unknown'
  };
  if (verificationResult) {
    savedObject = {
      ...savedObject,
      ...(0, _package_verification.formatVerificationResultForSO)(verificationResult)
    };
  }
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'create',
    id: pkgName,
    savedObjectType: _constants3.PACKAGES_SAVED_OBJECT_TYPE
  });
  const created = await savedObjectsClient.create(_constants3.PACKAGES_SAVED_OBJECT_TYPE, savedObject, {
    id: pkgName,
    overwrite: true
  });
  return created;
}
const kibanaAssetsToAssetsRef = kibanaAssets => {
  return Object.values(kibanaAssets).flat().map(_install.toAssetReference);
};
exports.kibanaAssetsToAssetsRef = kibanaAssetsToAssetsRef;
const saveKibanaAssetsRefs = async (savedObjectsClient, pkgName, assetRefs, saveAsAdditionnalSpace = false) => {
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'update',
    id: pkgName,
    savedObjectType: _constants3.PACKAGES_SAVED_OBJECT_TYPE
  });
  const spaceId = savedObjectsClient.getCurrentNamespace() || _constants.DEFAULT_SPACE_ID;

  // Because Kibana assets are installed in parallel with ES assets with refresh: false, we almost always run into an
  // issue that causes a conflict error due to this issue: https://github.com/elastic/kibana/issues/126240. This is safe
  // to retry constantly until it succeeds to optimize this critical user journey path as much as possible.
  await (0, _pRetry.default)(async () => {
    var _installation$attribu, _installation$attribu2;
    const installation = saveAsAdditionnalSpace ? await savedObjectsClient.get(_constants3.PACKAGES_SAVED_OBJECT_TYPE, pkgName).catch(e => {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        return undefined;
      }
      throw e;
    }) : undefined;
    return savedObjectsClient.update(_constants3.PACKAGES_SAVED_OBJECT_TYPE, pkgName, saveAsAdditionnalSpace ? {
      additional_spaces_installed_kibana: {
        ...(0, _lodash.omit)((_installation$attribu = installation === null || installation === void 0 ? void 0 : (_installation$attribu2 = installation.attributes) === null || _installation$attribu2 === void 0 ? void 0 : _installation$attribu2.additional_spaces_installed_kibana) !== null && _installation$attribu !== void 0 ? _installation$attribu : {}, spaceId),
        ...(assetRefs.length > 0 ? {
          [spaceId]: assetRefs
        } : {})
      }
    } : {
      installed_kibana: assetRefs
    }, {
      refresh: false
    });
  }, {
    retries: 20
  } // Use a number of retries higher than the number of es asset update operations
  );
  return assetRefs;
};
exports.saveKibanaAssetsRefs = saveKibanaAssetsRefs;
async function ensurePackagesCompletedInstall(savedObjectsClient, esClient) {
  const installingPackages = await (0, _get.getPackageSavedObjects)(savedObjectsClient, {
    searchFields: ['install_status'],
    search: 'installing'
  });
  const installingPromises = installingPackages.saved_objects.reduce((acc, pkg) => {
    const startDate = pkg.attributes.install_started_at;
    const nowDate = new Date().toISOString();
    const elapsedTime = Date.parse(nowDate) - Date.parse(startDate);
    const pkgkey = `${pkg.attributes.name}-${pkg.attributes.install_version}`;
    // reinstall package
    if (elapsedTime > _constants3.MAX_TIME_COMPLETE_INSTALL) {
      acc.push(installPackage({
        installSource: 'registry',
        savedObjectsClient,
        pkgkey,
        esClient,
        spaceId: pkg.attributes.installed_kibana_space_id || _constants.DEFAULT_SPACE_ID
      }));
    }
    return acc;
  }, []);
  await Promise.all(installingPromises);
  return installingPackages;
}
async function installAssetsForInputPackagePolicy(opts) {
  var _packagePolicy$inputs, _packagePolicy$inputs2;
  const {
    pkgInfo,
    logger,
    packagePolicy,
    esClient,
    soClient,
    force
  } = opts;
  if (pkgInfo.type !== 'input') return;
  const datasetName = (_packagePolicy$inputs = packagePolicy.inputs[0].streams[0].vars) === null || _packagePolicy$inputs === void 0 ? void 0 : (_packagePolicy$inputs2 = _packagePolicy$inputs[_constants2.DATASET_VAR_NAME]) === null || _packagePolicy$inputs2 === void 0 ? void 0 : _packagePolicy$inputs2.value;
  const [dataStream] = (0, _services.getNormalizedDataStreams)(pkgInfo, datasetName);
  const existingDataStreams = await _.dataStreamService.getMatchingDataStreams(esClient, {
    type: dataStream.type,
    dataset: datasetName
  });
  if (existingDataStreams.length) {
    const existingDataStreamsAreFromDifferentPackage = existingDataStreams.some(ds => {
      var _ds$_meta, _ds$_meta$package;
      return ((_ds$_meta = ds._meta) === null || _ds$_meta === void 0 ? void 0 : (_ds$_meta$package = _ds$_meta.package) === null || _ds$_meta$package === void 0 ? void 0 : _ds$_meta$package.name) !== pkgInfo.name;
    });
    if (existingDataStreamsAreFromDifferentPackage && !force) {
      // user has opted to send data to an existing data stream which is managed by another
      // package. This means certain custom setting such as elasticsearch settings
      // defined by the package will not have been applied which could lead
      // to unforeseen circumstances, so force flag must be used.
      const streamIndexPattern = _.dataStreamService.streamPartsToIndexPattern({
        type: dataStream.type,
        dataset: datasetName
      });
      throw new _errors.PackagePolicyValidationError(`Datastreams matching "${streamIndexPattern}" already exist and are not managed by this package, force flag is required`);
    } else {
      logger.info(`Data stream for dataset ${datasetName} already exists, skipping index template creation for ${packagePolicy.id}`);
      return;
    }
  }
  const existingIndexTemplate = await _.dataStreamService.getMatchingIndexTemplate(esClient, {
    type: dataStream.type,
    dataset: datasetName
  });
  if (existingIndexTemplate) {
    var _existingIndexTemplat, _existingIndexTemplat2;
    const indexTemplateOwnnedByDifferentPackage = ((_existingIndexTemplat = existingIndexTemplate._meta) === null || _existingIndexTemplat === void 0 ? void 0 : (_existingIndexTemplat2 = _existingIndexTemplat.package) === null || _existingIndexTemplat2 === void 0 ? void 0 : _existingIndexTemplat2.name) !== pkgInfo.name;
    if (indexTemplateOwnnedByDifferentPackage && !force) {
      // index template already exists but there is no data stream yet
      // we do not want to override the index template

      throw new _errors.PackagePolicyValidationError(`Index template "${dataStream.type}-${datasetName}" already exist and is not managed by this package, force flag is required`);
    } else {
      logger.info(`Index template "${dataStream.type}-${datasetName}" already exists, skipping index template creation for ${packagePolicy.id}`);
      return;
    }
  }
  const installedPkgWithAssets = await (0, _get.getInstalledPackageWithAssets)({
    savedObjectsClient: soClient,
    pkgName: pkgInfo.name,
    logger
  });
  let packageInstallContext;
  if (!installedPkgWithAssets) {
    throw new _errors.PackageNotFoundError(`Error while creating index templates: unable to find installed package ${pkgInfo.name}`);
  }
  if (installedPkgWithAssets.installation.version !== pkgInfo.version) {
    const pkg = await Registry.getPackage(pkgInfo.name, pkgInfo.version, {
      ignoreUnverified: force
    });
    packageInstallContext = {
      assetsMap: pkg.assetsMap,
      packageInfo: pkg.packageInfo,
      paths: pkg.paths
    };
  } else {
    packageInstallContext = {
      assetsMap: installedPkgWithAssets.assetsMap,
      packageInfo: installedPkgWithAssets.packageInfo,
      paths: installedPkgWithAssets.paths
    };
  }
  await (0, _install_index_template_pipeline.installIndexTemplatesAndPipelines)({
    installedPkg: installedPkgWithAssets.installation,
    packageInstallContext,
    esReferences: installedPkgWithAssets.installation.installed_es || [],
    savedObjectsClient: soClient,
    esClient,
    logger,
    onlyForDataStreams: [dataStream]
  });
  // Upate ES index patterns
  await (0, _es_assets_reference.optimisticallyAddEsAssetReferences)(soClient, installedPkgWithAssets.installation.name, [], (0, _template.generateESIndexPatterns)([dataStream]));
}

// overloads

// implementation
function getInstallType(args) {
  const {
    pkgVersion,
    installedPkg
  } = args;
  if (!installedPkg) return 'install';
  const currentPkgVersion = installedPkg.attributes.version;
  const lastStartedInstallVersion = installedPkg.attributes.install_version;
  if (pkgVersion === currentPkgVersion && pkgVersion !== lastStartedInstallVersion) return 'rollback';
  if (pkgVersion === currentPkgVersion) return 'reinstall';
  if (pkgVersion === lastStartedInstallVersion && pkgVersion !== currentPkgVersion) return 'reupdate';
  if (pkgVersion !== lastStartedInstallVersion && pkgVersion !== currentPkgVersion) return 'update';
  throw new _errors.FleetError('Unknown install type');
}