"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexPatternSavedObjects = getIndexPatternSavedObjects;
exports.indexPatternTypes = void 0;
exports.makeManagedIndexPatternsGlobal = makeManagedIndexPatternsGlobal;
exports.removeUnusedIndexPatterns = removeUnusedIndexPatterns;
var _constants = require("../../../../../common/constants");
var _ = require("../../..");
var _get = require("../../packages/get");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INDEX_PATTERN_SAVED_OBJECT_TYPE = 'index-pattern';
const indexPatternTypes = exports.indexPatternTypes = Object.values(_constants.dataTypes);
function getIndexPatternSavedObjects() {
  return indexPatternTypes.map(indexPatternType => ({
    id: `${indexPatternType}-*`,
    type: INDEX_PATTERN_SAVED_OBJECT_TYPE,
    // workaround until https://github.com/elastic/kibana/issues/164454 is fixed
    typeMigrationVersion: '8.0.0',
    attributes: {
      title: `${indexPatternType}-*`,
      timeFieldName: '@timestamp',
      allowNoIndex: true
    }
  }));
}
async function makeManagedIndexPatternsGlobal(savedObjectsClient) {
  const logger = _.appContextService.getLogger();
  const results = [];
  for (const indexPatternType of indexPatternTypes) {
    try {
      const result = await savedObjectsClient.updateObjectsSpaces([{
        id: `${indexPatternType}-*`,
        type: INDEX_PATTERN_SAVED_OBJECT_TYPE
      }], ['*'], []);
      results.push(result);
    } catch (error) {
      logger.error(`Error making managed index patterns global: ${error.message}`);
    }
  }
  return results;
}
async function removeUnusedIndexPatterns(savedObjectsClient) {
  const logger = _.appContextService.getLogger();
  // get all user installed packages
  const installedPackagesRes = await (0, _get.getPackageSavedObjects)(savedObjectsClient);
  const installedPackagesSavedObjects = installedPackagesRes.saved_objects.filter(so => so.attributes.install_status === _constants.installationStatuses.Installed);
  if (installedPackagesSavedObjects.length > 0) {
    return [];
  }
  const patternsToDelete = indexPatternTypes.map(indexPatternType => `${indexPatternType}-*`);
  const {
    resolved_objects: resolvedObjects
  } = await savedObjectsClient.bulkResolve(patternsToDelete.map(pattern => ({
    id: pattern,
    type: INDEX_PATTERN_SAVED_OBJECT_TYPE
  })));

  // eslint-disable-next-line @typescript-eslint/naming-convention
  const idsToDelete = resolvedObjects.map(({
    saved_object
  }) => saved_object.id);
  return Promise.all(idsToDelete.map(async id => {
    try {
      logger.debug(`deleting index pattern ${id}`);
      await savedObjectsClient.delete(INDEX_PATTERN_SAVED_OBJECT_TYPE, id);
    } catch (err) {
      // index pattern was probably deleted by the user already
      logger.debug(`Non fatal error encountered deleting index pattern ${id} : ${err}`);
    }
    return;
  }));
}