"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateTransformSecondaryAuthHeaders = generateTransformSecondaryAuthHeaders;
exports.isTransformApiKey = isTransformApiKey;
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isTransformApiKey(arg) {
  return arg && arg.hasOwnProperty('api_key') && arg.hasOwnProperty('encoded') && typeof arg.encoded === 'string';
}
function createKibanaRequestFromAuth(authorizationHeader) {
  const requestHeaders = {
    authorization: authorizationHeader.toString()
  };
  const fakeRawRequest = {
    headers: requestHeaders,
    path: '/'
  };

  // Since we're using API keys and accessing elasticsearch can only be done
  // via a request, we're faking one with the proper authorization headers.
  const fakeRequest = _coreHttpRouterServerInternal.CoreKibanaRequest.from(fakeRawRequest);
  return fakeRequest;
}

/** This function generates a new API based on current Kibana's user request.headers.authorization
 * then formats it into a es-secondary-authorization header object
 * @param authorizationHeader:
 * @param createParams
 */
async function generateTransformSecondaryAuthHeaders({
  authorizationHeader,
  createParams,
  logger,
  username,
  pkgName,
  pkgVersion
}) {
  if (!authorizationHeader) {
    return;
  }
  const fakeKibanaRequest = createKibanaRequestFromAuth(authorizationHeader);
  const user = username !== null && username !== void 0 ? username : authorizationHeader.getUsername();
  const name = pkgName ? `${pkgName}${pkgVersion ? '-' + pkgVersion : ''}-transform${user ? '-by-' + user : ''}` : `fleet-transform-api-key`;
  const security = _.appContextService.getSecurity();

  // If security is not enabled or available, we can't generate api key
  // but that's ok, cause all the index and transform commands should work
  if (!security) return;
  try {
    const apiKeyWithCurrentUserPermission = await (security === null || security === void 0 ? void 0 : security.authc.apiKeys.grantAsInternalUser(fakeKibanaRequest, createParams !== null && createParams !== void 0 ? createParams : {
      name,
      metadata: {
        managed_by: 'fleet',
        managed: true,
        type: 'transform'
      },
      role_descriptors: {}
    }));
    logger.debug(`Created api_key name: ${name}`);
    let encodedApiKey = null;

    // Property 'encoded' does exist in the resp coming back from request
    // and is required to use in authentication headers
    // It's just not defined in returned GrantAPIKeyResult type
    if (isTransformApiKey(apiKeyWithCurrentUserPermission)) {
      encodedApiKey = apiKeyWithCurrentUserPermission.encoded;
    }
    const secondaryAuth = encodedApiKey !== null ? {
      headers: {
        'es-secondary-authorization': `ApiKey ${encodedApiKey}`
      }
    } : undefined;
    return secondaryAuth;
  } catch (e) {
    logger.debug(`Failed to create api_key: ${name} because ${e}`);
    return undefined;
  }
}