"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addPackageToAgentPolicy = addPackageToAgentPolicy;
exports.agentPolicyService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _uuid = require("uuid");
var _jsYaml = require("js-yaml");
var _pMap = _interopRequireDefault(require("p-map"));
var _semver = require("semver");
var _server = require("@kbn/core/server");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _std = require("@kbn/std");
var _services = require("../../common/services");
var _constants2 = require("../constants");
var _constants3 = require("../../common/constants");
var _errors = require("../errors");
var _agent_cm_to_yaml = require("../../common/services/agent_cm_to_yaml");
var _ = require(".");
var _utils = require("./agent_policies/utils");
var _elastic_agent_manifest = require("./elastic_agent_manifest");
var _packages = require("./epm/packages");
var _agents = require("./agents");
var _package_policy = require("./package_policy");
var _package_policies = require("./package_policies");
var _output = require("./output");
var _agent_policy_update = require("./agent_policy_update");
var _saved_object = require("./saved_object");
var _agent_policies = require("./agent_policies");
var _audit_logging = require("./audit_logging");
var _license = require("./license");
var _create_so_find_iterable = require("./utils/create_so_find_iterable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SAVED_OBJECT_TYPE = _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE;
const KEY_EDITABLE_FOR_MANAGED_POLICIES = ['namespace'];
class AgentPolicyService {
  constructor() {
    (0, _defineProperty2.default)(this, "triggerAgentPolicyUpdatedEvent", async (esClient, action, agentPolicyId, options) => {
      return (0, _agent_policy_update.agentPolicyUpdateEventHandler)(esClient, action, agentPolicyId, options);
    });
  }
  async _update(soClient, esClient, id, agentPolicy, user, options = {
    bumpRevision: true,
    removeProtection: false,
    skipValidation: false
  }) {
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id,
      savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
    });
    const logger = _.appContextService.getLogger();
    logger.debug(`Starting update of agent policy ${id}`);
    const existingAgentPolicy = await this.get(soClient, id, true);
    if (!existingAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    if (existingAgentPolicy.status === _constants3.agentPolicyStatuses.Inactive && agentPolicy.status !== _constants3.agentPolicyStatuses.Active) {
      throw new _errors.FleetError(`Agent policy ${id} cannot be updated because it is ${existingAgentPolicy.status}`);
    }
    if (options.removeProtection) {
      logger.warn(`Setting tamper protection for Agent Policy ${id} to false`);
    }
    if (!options.skipValidation) {
      await (0, _agent_policies.validateOutputForPolicy)(soClient, agentPolicy, existingAgentPolicy, (0, _services.getAllowedOutputTypeForPolicy)(existingAgentPolicy));
    }
    await soClient.update(SAVED_OBJECT_TYPE, id, {
      ...agentPolicy,
      ...(options.bumpRevision ? {
        revision: existingAgentPolicy.revision + 1
      } : {}),
      ...(options.removeProtection ? {
        is_protected: false
      } : {
        is_protected: agentPolicy.is_protected
      }),
      updated_at: new Date().toISOString(),
      updated_by: user ? user.username : 'system'
    });
    if (options.bumpRevision || options.removeProtection) {
      await this.triggerAgentPolicyUpdatedEvent(esClient, 'updated', id, {
        spaceId: soClient.getCurrentNamespace()
      });
    }
    logger.debug(`Agent policy ${id} update completed, revision: ${options.bumpRevision ? existingAgentPolicy.revision + 1 : existingAgentPolicy.revision}`);
    return await this.get(soClient, id);
  }
  async ensurePreconfiguredAgentPolicy(soClient, esClient, config) {
    const {
      id,
      space_id: kibanaSpaceId,
      ...preconfiguredAgentPolicy
    } = (0, _lodash.omit)(config, 'package_policies');
    const newAgentPolicyDefaults = {
      namespace: 'default',
      monitoring_enabled: ['logs', 'metrics']
    };
    const newAgentPolicy = {
      ...newAgentPolicyDefaults,
      ...preconfiguredAgentPolicy,
      is_preconfigured: true
    };
    if (!id) throw new _errors.AgentPolicyNotFoundError('Missing ID');
    return await this.ensureAgentPolicy(soClient, esClient, newAgentPolicy, id);
  }
  async ensureAgentPolicy(soClient, esClient, newAgentPolicy, id) {
    // For preconfigured policies with a specified ID
    const agentPolicy = await this.get(soClient, id, false).catch(() => null);
    if (!agentPolicy) {
      return {
        created: true,
        policy: await this.create(soClient, esClient, newAgentPolicy, {
          id
        })
      };
    }
    return {
      created: false,
      policy: agentPolicy
    };
  }
  hasAPMIntegration(agentPolicy) {
    return (0, _services.policyHasAPMIntegration)(agentPolicy);
  }
  hasFleetServerIntegration(agentPolicy) {
    return (0, _services.policyHasFleetServer)(agentPolicy);
  }
  hasSyntheticsIntegration(agentPolicy) {
    return (0, _services.policyHasSyntheticsIntegration)(agentPolicy);
  }
  async runExternalCallbacks(externalCallbackType, agentPolicy) {
    const logger = _.appContextService.getLogger();
    logger.debug(`Running external callbacks for ${externalCallbackType}`);
    try {
      const externalCallbacks = _.appContextService.getExternalCallbacks(externalCallbackType);
      let newAgentPolicy = agentPolicy;
      if (externalCallbacks && externalCallbacks.size > 0) {
        let updatedNewAgentPolicy = newAgentPolicy;
        for (const callback of externalCallbacks) {
          let result;
          if (externalCallbackType === 'agentPolicyCreate') {
            result = await callback(newAgentPolicy);
            updatedNewAgentPolicy = result;
          }
          if (externalCallbackType === 'agentPolicyUpdate') {
            result = await callback(newAgentPolicy);
            updatedNewAgentPolicy = result;
          }
        }
        newAgentPolicy = updatedNewAgentPolicy;
      }
      return newAgentPolicy;
    } catch (error) {
      logger.error(`Error running external callbacks for ${externalCallbackType}`);
      logger.error(error);
      throw error;
    }
  }
  async create(soClient, esClient, agentPolicy, options = {}) {
    var _ref, _options$user, _appContextService$ge, _appContextService$ge2;
    // Ensure an ID is provided, so we can include it in the audit logs below
    if (!options.id) {
      options.id = _server.SavedObjectsUtils.generateId();
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id: options.id,
      savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
    });
    await this.runExternalCallbacks('agentPolicyCreate', agentPolicy);
    this.checkTamperProtectionLicense(agentPolicy);
    const logger = _.appContextService.getLogger();
    logger.debug(`Creating new agent policy`);
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected) {
      logger.warn('Agent policy requires Elastic Defend integration to set tamper protection to true');
    }
    this.checkAgentless(agentPolicy);
    await this.requireUniqueName(soClient, agentPolicy);
    await (0, _agent_policies.validateOutputForPolicy)(soClient, agentPolicy);
    const newSo = await soClient.create(SAVED_OBJECT_TYPE, {
      ...agentPolicy,
      status: 'active',
      is_managed: (_ref = agentPolicy.is_managed || (agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.supports_agentless)) !== null && _ref !== void 0 ? _ref : false,
      revision: 1,
      updated_at: new Date().toISOString(),
      updated_by: (options === null || options === void 0 ? void 0 : (_options$user = options.user) === null || _options$user === void 0 ? void 0 : _options$user.username) || 'system',
      schema_version: _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION,
      is_protected: false
    }, options);
    await ((_appContextService$ge = _.appContextService.getUninstallTokenService()) === null || _appContextService$ge === void 0 ? void 0 : (_appContextService$ge2 = _appContextService$ge.scoped(soClient.getCurrentNamespace())) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.generateTokenForPolicyId(newSo.id));
    await this.triggerAgentPolicyUpdatedEvent(esClient, 'created', newSo.id, {
      skipDeploy: options.skipDeploy,
      spaceId: soClient.getCurrentNamespace()
    });
    logger.debug(`Created new agent policy with id ${newSo.id}`);
    return {
      id: newSo.id,
      ...newSo.attributes
    };
  }
  async requireUniqueName(soClient, givenPolicy) {
    const results = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      searchFields: ['name'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(givenPolicy.name)
    });
    const idsWithName = results.total && results.saved_objects.map(({
      id
    }) => id);
    if (Array.isArray(idsWithName)) {
      const isEditingSelf = givenPolicy.id && idsWithName.includes(givenPolicy.id);
      if (!givenPolicy.id || !isEditingSelf) {
        const isSinglePolicy = idsWithName.length === 1;
        const existClause = isSinglePolicy ? `Agent Policy '${idsWithName[0]}' already exists` : `Agent Policies '${idsWithName.join(',')}' already exist`;
        throw new _errors.AgentPolicyNameExistsError(`${existClause} with name '${givenPolicy.name}'`);
      }
    }
  }
  async get(soClient, id, withPackagePolicies = true) {
    const agentPolicySO = await soClient.get(SAVED_OBJECT_TYPE, id);
    if (!agentPolicySO) {
      return null;
    }
    if (agentPolicySO.error) {
      throw new _errors.FleetError(agentPolicySO.error.message);
    }
    const agentPolicy = (0, _utils.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
    if (withPackagePolicies) {
      agentPolicy.package_policies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, id)) || [];
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'get',
      id,
      savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
    });
    return agentPolicy;
  }
  async getByIDs(soClient, ids, options = {}) {
    const objects = ids.map(id => {
      if (typeof id === 'string') {
        return {
          ...options,
          id,
          type: SAVED_OBJECT_TYPE
        };
      }
      return {
        ...options,
        id: id.id,
        namespaces: id.spaceId ? [id.spaceId] : undefined,
        type: SAVED_OBJECT_TYPE
      };
    });
    const bulkGetResponse = await soClient.bulkGet(objects);
    const agentPolicies = await (0, _pMap.default)(bulkGetResponse.saved_objects, async agentPolicySO => {
      if (agentPolicySO.error) {
        if (options.ignoreMissing && agentPolicySO.error.statusCode === 404) {
          return null;
        } else if (agentPolicySO.error.statusCode === 404) {
          throw new _errors.AgentPolicyNotFoundError(`Agent policy ${agentPolicySO.id} not found`);
        } else {
          throw new _errors.FleetError(agentPolicySO.error.message);
        }
      }
      const agentPolicy = (0, _utils.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
      if (options.withPackagePolicies) {
        const agentPolicyWithPackagePolicies = await this.get(soClient, agentPolicySO.id, options.withPackagePolicies);
        if (agentPolicyWithPackagePolicies) {
          agentPolicy.package_policies = agentPolicyWithPackagePolicies.package_policies;
        }
      }
      return agentPolicy;
    }, {
      concurrency: 50
    });
    const result = agentPolicies.filter(agentPolicy => agentPolicy !== null);
    for (const agentPolicy of result) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: agentPolicy.id,
        savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return result;
  }
  async list(soClient, options) {
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery,
      withPackagePolicies = false,
      fields
    } = options;
    const baseFindParams = {
      type: SAVED_OBJECT_TYPE,
      sortField,
      sortOrder,
      page,
      perPage,
      ...(fields ? {
        fields
      } : {})
    };
    const filter = kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined;
    let agentPoliciesSO;
    try {
      agentPoliciesSO = await soClient.find({
        ...baseFindParams,
        filter
      });
    } catch (e) {
      var _e$output, _e$message;
      const isBadRequest = ((_e$output = e.output) === null || _e$output === void 0 ? void 0 : _e$output.statusCode) === 400;
      const isKQLSyntaxError = (_e$message = e.message) === null || _e$message === void 0 ? void 0 : _e$message.startsWith('KQLSyntaxError');
      if (isBadRequest && !isKQLSyntaxError) {
        // fall back to simple search if the kuery is just a search term i.e not KQL
        agentPoliciesSO = await soClient.find({
          ...baseFindParams,
          search: kuery
        });
      } else {
        throw e;
      }
    }
    const agentPolicies = await (0, _pMap.default)(agentPoliciesSO.saved_objects, async agentPolicySO => {
      const agentPolicy = (0, _utils.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
      if (withPackagePolicies) {
        agentPolicy.package_policies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, agentPolicySO.id)) || [];
      }
      if (options.withAgentCount) {
        await (0, _agents.getAgentsByKuery)(_.appContextService.getInternalUserESClient(), _.appContextService.getInternalUserSOClientForSpaceId(agentPolicy.space_id), {
          showInactive: true,
          perPage: 0,
          page: 1,
          kuery: `${_constants2.AGENTS_PREFIX}.policy_id:${agentPolicy.id}`
        }).then(({
          total
        }) => agentPolicy.agents = total);
      } else {
        agentPolicy.agents = 0;
      }
      return agentPolicy;
    }, {
      concurrency: 50
    });
    for (const agentPolicy of agentPolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: agentPolicy.id,
        savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return {
      items: agentPolicies,
      total: agentPoliciesSO.total,
      page,
      perPage
    };
  }
  async update(soClient, esClient, id, agentPolicy, options) {
    var _options$skipValidati;
    const logger = _.appContextService.getLogger();
    logger.debug(`Starting update of agent policy ${id}`);
    if (agentPolicy.name) {
      await this.requireUniqueName(soClient, {
        id,
        name: agentPolicy.name
      });
    }
    const existingAgentPolicy = await this.get(soClient, id, true);
    if (!existingAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    try {
      await this.runExternalCallbacks('agentPolicyUpdate', agentPolicy);
    } catch (error) {
      logger.error(`Error running external callbacks for agentPolicyUpdate`);
      if (error.apiPassThrough) {
        throw error;
      }
    }
    this.checkTamperProtectionLicense(agentPolicy);
    this.checkAgentless(agentPolicy);
    await this.checkForValidUninstallToken(agentPolicy, id);
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected && !(0, _services.policyHasEndpointSecurity)(existingAgentPolicy)) {
      logger.warn('Agent policy requires Elastic Defend integration to set tamper protection to true');
      // force agent policy to be false if elastic defend is not present
      agentPolicy.is_protected = false;
    }
    if (existingAgentPolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
      Object.entries(agentPolicy).filter(([key]) => !KEY_EDITABLE_FOR_MANAGED_POLICIES.includes(key)).forEach(([key, val]) => {
        if (!(0, _lodash.isEqual)(existingAgentPolicy[key], val)) {
          throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot update ${key}`);
        }
      });
    }
    const {
      monitoring_enabled: monitoringEnabled
    } = agentPolicy;
    const packagesToInstall = [];
    if (!existingAgentPolicy.monitoring_enabled && monitoringEnabled !== null && monitoringEnabled !== void 0 && monitoringEnabled.length) {
      packagesToInstall.push(_constants3.FLEET_ELASTIC_AGENT_PACKAGE);
    }
    if (packagesToInstall.length > 0) {
      await (0, _packages.bulkInstallPackages)({
        savedObjectsClient: soClient,
        esClient,
        packagesToInstall,
        spaceId: (options === null || options === void 0 ? void 0 : options.spaceId) || _constants.DEFAULT_SPACE_ID,
        authorizationHeader: options === null || options === void 0 ? void 0 : options.authorizationHeader,
        force: options === null || options === void 0 ? void 0 : options.force
      });
    }
    return this._update(soClient, esClient, id, agentPolicy, options === null || options === void 0 ? void 0 : options.user, {
      bumpRevision: true,
      removeProtection: false,
      skipValidation: (_options$skipValidati = options === null || options === void 0 ? void 0 : options.skipValidation) !== null && _options$skipValidati !== void 0 ? _options$skipValidati : false
    });
  }
  async copy(soClient, esClient, id, newAgentPolicyProps, options) {
    const logger = _.appContextService.getLogger();
    logger.debug(`Starting copy of agent policy ${id}`);

    // Copy base agent policy
    const baseAgentPolicy = await this.get(soClient, id, true);
    if (!baseAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    const newAgentPolicy = await this.create(soClient, esClient, {
      ...(0, _lodash.pick)(baseAgentPolicy, ['namespace', 'monitoring_enabled', 'inactivity_timeout', 'unenroll_timeout', 'agent_features', 'overrides', 'data_output_id', 'monitoring_output_id', 'download_source_id', 'fleet_server_host_id', 'supports_agentless', 'global_data_tags']),
      ...newAgentPolicyProps
    }, options);

    // Copy all package policies and append (copy n) to their names
    if (baseAgentPolicy.package_policies) {
      const newPackagePolicies = await (0, _pMap.default)(baseAgentPolicy.package_policies, async packagePolicy => {
        const {
          id: packagePolicyId,
          version,
          ...newPackagePolicy
        } = packagePolicy;
        const updatedPackagePolicy = {
          ...newPackagePolicy,
          name: await (0, _package_policies.incrementPackagePolicyCopyName)(soClient, packagePolicy.name)
        };
        return updatedPackagePolicy;
      });
      await _package_policy.packagePolicyService.bulkCreate(soClient, esClient, newPackagePolicies.map(newPackagePolicy => ({
        ...newPackagePolicy,
        policy_ids: [newAgentPolicy.id]
      })), {
        ...options,
        bumpRevision: false
      });
    }

    // Tamper protection is dependent on endpoint package policy
    // Match tamper protection setting to the original policy
    if (baseAgentPolicy.is_protected) {
      await this._update(soClient, esClient, newAgentPolicy.id, {
        is_protected: true
      }, options === null || options === void 0 ? void 0 : options.user, {
        bumpRevision: false,
        removeProtection: false,
        skipValidation: false
      });
    }
    const policyNeedsBump = baseAgentPolicy.package_policies || baseAgentPolicy.is_protected;

    // bump revision if agent policy is updated after creation
    if (policyNeedsBump) {
      await this.bumpRevision(soClient, esClient, newAgentPolicy.id, {
        user: options === null || options === void 0 ? void 0 : options.user
      });
    } else {
      await this.deployPolicy(soClient, newAgentPolicy.id);
    }

    // Get updated agent policy with package policies and adjusted tamper protection
    const updatedAgentPolicy = await this.get(soClient, newAgentPolicy.id, true);
    if (!updatedAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Copied agent policy not found');
    }
    logger.debug(`Completed copy of agent policy ${id}`);
    return updatedAgentPolicy;
  }
  async bumpRevision(soClient, esClient, id, options) {
    var _options$removeProtec;
    const res = await this._update(soClient, esClient, id, {}, options === null || options === void 0 ? void 0 : options.user, {
      bumpRevision: true,
      removeProtection: (_options$removeProtec = options === null || options === void 0 ? void 0 : options.removeProtection) !== null && _options$removeProtec !== void 0 ? _options$removeProtec : false,
      skipValidation: false
    });
    return res;
  }

  /**
   * Remove an output from all agent policies that are using it, and replace the output by the default ones.
   * @param soClient
   * @param esClient
   * @param outputId
   */
  async removeOutputFromAll(soClient, esClient, outputId) {
    const agentPolicies = (await soClient.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'data_output_id', 'monitoring_output_id'],
      searchFields: ['data_output_id', 'monitoring_output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT
    })).saved_objects.map(_utils.mapAgentPolicySavedObjectToAgentPolicy);
    if (agentPolicies.length > 0) {
      const getAgentPolicy = agentPolicy => ({
        data_output_id: agentPolicy.data_output_id === outputId ? null : agentPolicy.data_output_id,
        monitoring_output_id: agentPolicy.monitoring_output_id === outputId ? null : agentPolicy.monitoring_output_id
      });
      // Validate that the output is not used by any agent policy before updating any policy
      await (0, _pMap.default)(agentPolicies, async agentPolicy => {
        const existingAgentPolicy = await this.get(soClient, agentPolicy.id, true);
        if (!existingAgentPolicy) {
          throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
        }
        await (0, _agent_policies.validateOutputForPolicy)(soClient, getAgentPolicy(agentPolicy), existingAgentPolicy, (0, _services.getAllowedOutputTypeForPolicy)(existingAgentPolicy));
      }, {
        concurrency: 50
      });
      await (0, _pMap.default)(agentPolicies, agentPolicy => this.update(soClient, esClient, agentPolicy.id, getAgentPolicy(agentPolicy), {
        skipValidation: true
      }), {
        concurrency: 50
      });
    }
  }

  /**
   * Remove a Fleet Server from all agent policies that are using it, to use the default one instead.
   */
  async removeFleetServerHostFromAll(soClient, esClient, fleetServerHostId) {
    const agentPolicies = (await soClient.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'fleet_server_host_id'],
      searchFields: ['fleet_server_host_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(fleetServerHostId),
      perPage: _constants2.SO_SEARCH_LIMIT
    })).saved_objects.map(_utils.mapAgentPolicySavedObjectToAgentPolicy);
    if (agentPolicies.length > 0) {
      await (0, _pMap.default)(agentPolicies, agentPolicy => this.update(soClient, esClient, agentPolicy.id, {
        fleet_server_host_id: null
      }), {
        concurrency: 50
      });
    }
  }
  async _bumpPolicies(internalSoClientWithoutSpaceExtension, esClient, savedObjectsResults, options) {
    const bumpedPolicies = savedObjectsResults.map(policy => {
      var _policy$namespaces;
      return {
        id: policy.id,
        type: policy.type,
        attributes: {
          ...policy.attributes,
          revision: policy.attributes.revision + 1,
          updated_at: new Date().toISOString(),
          updated_by: options !== null && options !== void 0 && options.user ? options.user.username : 'system'
        },
        version: policy.version,
        namespace: (_policy$namespaces = policy.namespaces) === null || _policy$namespaces === void 0 ? void 0 : _policy$namespaces[0]
      };
    });
    const bumpedPoliciesBySpaceId = (0, _lodash.groupBy)(bumpedPolicies, policy => policy.namespace || _constants.DEFAULT_SPACE_ID);
    const res = (await Promise.all(Object.entries(bumpedPoliciesBySpaceId).map(([spaceId, policies]) => internalSoClientWithoutSpaceExtension.bulkUpdate(policies, {
      namespace: spaceId
    })))).reduce((acc, r) => {
      if (r !== null && r !== void 0 && r.saved_objects) {
        acc.saved_objects.push(...r.saved_objects);
      }
      return acc;
    }, {
      saved_objects: []
    });
    await (0, _pMap.default)(savedObjectsResults, policy => {
      var _policy$namespaces2;
      return this.triggerAgentPolicyUpdatedEvent(esClient, 'updated', policy.id, {
        spaceId: (_policy$namespaces2 = policy.namespaces) === null || _policy$namespaces2 === void 0 ? void 0 : _policy$namespaces2[0]
      });
    }, {
      concurrency: 50
    });
    return res;
  }
  async bumpAllAgentPoliciesForOutput(esClient, outputId, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'data_output_id', 'monitoring_output_id', 'namespaces'],
      searchFields: ['data_output_id', 'monitoring_output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, esClient, currentPolicies.saved_objects, options);
  }
  async bumpAllAgentPolicies(esClient, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['name', 'revision', 'namespaces'],
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, esClient, currentPolicies.saved_objects, options);
  }
  async delete(soClient, esClient, id, options) {
    const logger = _.appContextService.getLogger();
    logger.debug(`Deleting agent policy ${id}`);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'delete',
      id,
      savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
    });
    const agentPolicy = await this.get(soClient, id, false);
    if (!agentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    if (agentPolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
      throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot delete hosted agent policy ${id}`);
    }
    // Prevent deleting policy when assigned agents are inactive
    const {
      total
    } = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
      showInactive: true,
      perPage: 0,
      page: 1,
      kuery: `${_constants2.AGENTS_PREFIX}.policy_id:${id}`
    });
    if (total > 0) {
      throw new _errors.FleetError('Cannot delete an agent policy that is assigned to any active or inactive agents');
    }
    const packagePolicies = await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, id);
    if (packagePolicies.length) {
      const hasManagedPackagePolicies = packagePolicies.some(packagePolicy => packagePolicy.is_managed);
      if (hasManagedPackagePolicies && !(options !== null && options !== void 0 && options.force)) {
        throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot delete agent policy ${id} that contains managed package policies`);
      }
      const {
        policiesWithSingleAP: packagePoliciesToDelete,
        policiesWithMultipleAP
      } = this.packagePoliciesWithSingleAndMultiplePolicies(packagePolicies);
      if (packagePoliciesToDelete.length > 0) {
        await _package_policy.packagePolicyService.delete(soClient, esClient, packagePoliciesToDelete.map(p => p.id), {
          force: options === null || options === void 0 ? void 0 : options.force,
          skipUnassignFromAgentPolicies: true
        });
        logger.debug(`Deleted package policies with single agent policy with ids ${packagePoliciesToDelete.map(policy => policy.id).join(', ')}`);
      }
      if (policiesWithMultipleAP.length > 0) {
        await _package_policy.packagePolicyService.bulkUpdate(soClient, esClient, policiesWithMultipleAP.map(policy => {
          const newPolicyIds = policy.policy_ids.filter(policyId => policyId !== id);
          return {
            ...policy,
            policy_id: newPolicyIds[0],
            policy_ids: newPolicyIds
          };
        }));
        logger.debug(`Updated package policies with multiple agent policies with ids ${policiesWithMultipleAP.map(policy => policy.id).join(', ')}`);
      }
    }
    if (agentPolicy.is_preconfigured && !(options !== null && options !== void 0 && options.force)) {
      await soClient.create(_constants2.PRECONFIGURATION_DELETION_RECORD_SAVED_OBJECT_TYPE, {
        id: String(id)
      });
    }
    await soClient.delete(SAVED_OBJECT_TYPE, id);
    await this.triggerAgentPolicyUpdatedEvent(esClient, 'deleted', id, {
      spaceId: soClient.getCurrentNamespace()
    });

    // cleanup .fleet-policies docs on delete
    await this.deleteFleetServerPoliciesForPolicyId(esClient, id);
    logger.debug(`Deleted agent policy ${id}`);
    return {
      id,
      name: agentPolicy.name
    };
  }
  async deployPolicy(soClient, agentPolicyId) {
    await this.deployPolicies(soClient, [agentPolicyId]);
  }
  async deployPolicies(soClient, agentPolicyIds) {
    // Use internal ES client so we have permissions to write to .fleet* indices
    const esClient = _.appContextService.getInternalUserESClient();
    const defaultOutputId = await _output.outputService.getDefaultDataOutputId(soClient);
    if (!defaultOutputId) {
      return;
    }
    for (const policyId of agentPolicyIds) {
      _audit_logging.auditLoggingService.writeCustomAuditLog({
        message: `User deploying policy [id=${policyId}]`
      });
    }
    const policies = await agentPolicyService.getByIDs(soClient, agentPolicyIds);
    const policiesMap = (0, _lodash.keyBy)(policies, 'id');
    const fullPolicies = await (0, _pMap.default)(agentPolicyIds,
    // There are some potential performance concerns around using `getFullAgentPolicy` in this context, e.g.
    // re-fetching outputs, settings, and upgrade download source URI data for each policy. This could potentially
    // be a bottleneck in environments with several thousand agent policies being deployed here.
    agentPolicyId => agentPolicyService.getFullAgentPolicy(soClient, agentPolicyId), {
      concurrency: 50
    });
    const fleetServerPolicies = fullPolicies.reduce((acc, fullPolicy) => {
      if (!fullPolicy || !fullPolicy.revision) {
        return acc;
      }
      const policy = policiesMap[fullPolicy.id];
      if (!policy) {
        return acc;
      }
      const fleetServerPolicy = {
        '@timestamp': new Date().toISOString(),
        revision_idx: fullPolicy.revision,
        coordinator_idx: 0,
        data: fullPolicy,
        policy_id: fullPolicy.id,
        default_fleet_server: policy.is_default_fleet_server === true
      };
      if (policy.unenroll_timeout) {
        fleetServerPolicy.unenroll_timeout = policy.unenroll_timeout;
      }
      acc.push(fleetServerPolicy);
      return acc;
    }, []);
    _.appContextService.getLogger().debug(`Deploying policies: ${fleetServerPolicies.map(pol => `${pol.policy_id}:${pol.revision_idx}`).join(', ')}`);
    const fleetServerPoliciesBulkBody = fleetServerPolicies.flatMap(fleetServerPolicy => [{
      index: {
        _id: (0, _uuid.v5)(`${fleetServerPolicy.policy_id}:${fleetServerPolicy.revision_idx}`, _uuid.v5.DNS)
      }
    }, fleetServerPolicy]);
    const bulkResponse = await esClient.bulk({
      index: _constants3.AGENT_POLICY_INDEX,
      operations: fleetServerPoliciesBulkBody,
      refresh: 'wait_for'
    });
    if (bulkResponse.errors) {
      const logger = _.appContextService.getLogger();
      const erroredDocuments = bulkResponse.items.reduce((acc, item) => {
        const value = item.index;
        if (!value || !value.error) {
          return acc;
        }
        acc.push(value);
        return acc;
      }, []);
      logger.warn(`Failed to index documents during policy deployment: ${JSON.stringify(erroredDocuments)}`);
    }
    await Promise.all(fleetServerPolicies.filter(fleetServerPolicy => {
      const policy = policiesMap[fleetServerPolicy.policy_id];
      return !policy.schema_version || (0, _semver.lt)(policy.schema_version, _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION);
    }).map(fleetServerPolicy =>
    // There are some potential performance concerns around using `agentPolicyService.update` in this context.
    // This could potentially be a bottleneck in environments with several thousand agent policies being deployed here.
    agentPolicyService.update(soClient, esClient, fleetServerPolicy.policy_id, {
      schema_version: _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION
    }, {
      force: true
    })));
  }
  async deleteFleetServerPoliciesForPolicyId(esClient, agentPolicyId) {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User deleting policy [id=${agentPolicyId}]`
    });
    let hasMore = true;
    while (hasMore) {
      var _res$deleted;
      const res = await esClient.deleteByQuery({
        index: _constants3.AGENT_POLICY_INDEX,
        ignore_unavailable: true,
        scroll_size: _constants2.SO_SEARCH_LIMIT,
        refresh: true,
        query: {
          term: {
            policy_id: agentPolicyId
          }
        }
      });
      hasMore = ((_res$deleted = res.deleted) !== null && _res$deleted !== void 0 ? _res$deleted : 0) === _constants2.SO_SEARCH_LIMIT;
    }
  }
  async getLatestFleetPolicy(esClient, agentPolicyId) {
    const res = await esClient.search({
      index: _constants3.AGENT_POLICY_INDEX,
      ignore_unavailable: true,
      rest_total_hits_as_int: true,
      body: {
        query: {
          term: {
            policy_id: agentPolicyId
          }
        },
        size: 1,
        sort: [{
          revision_idx: {
            order: 'desc'
          }
        }]
      }
    });
    if (res.hits.total === 0) {
      return null;
    }
    return res.hits.hits[0]._source;
  }
  async getFullAgentConfigMap(soClient, id, agentVersion, options) {
    const fullAgentPolicy = await (0, _agent_policies.getFullAgentPolicy)(soClient, id, options);
    if (fullAgentPolicy) {
      const fullAgentConfigMap = {
        apiVersion: 'v1',
        kind: 'ConfigMap',
        metadata: {
          name: 'agent-node-datastreams',
          namespace: 'kube-system',
          labels: {
            'k8s-app': 'elastic-agent'
          }
        },
        data: {
          'agent.yml': fullAgentPolicy
        }
      };
      const configMapYaml = (0, _agent_cm_to_yaml.fullAgentConfigMapToYaml)(fullAgentConfigMap, _jsYaml.safeDump);
      const updateManifestVersion = _elastic_agent_manifest.elasticAgentStandaloneManifest.replace('VERSION', agentVersion);
      const fixedAgentYML = configMapYaml.replace('agent.yml:', 'agent.yml: |-');
      return [fixedAgentYML, updateManifestVersion].join('\n');
    } else {
      return '';
    }
  }
  async getFullAgentManifest(fleetServer, enrolToken, agentVersion) {
    const updateManifestVersion = _elastic_agent_manifest.elasticAgentManagedManifest.replace('VERSION', agentVersion);
    let updateManifest = updateManifestVersion;
    if (fleetServer !== '') {
      updateManifest = updateManifest.replace('https://fleet-server:8220', fleetServer);
    }
    if (enrolToken !== '') {
      updateManifest = updateManifest.replace('token-id', enrolToken);
    }
    return updateManifest;
  }
  async getFullAgentPolicy(soClient, id, options) {
    return (0, _agent_policies.getFullAgentPolicy)(soClient, id, options);
  }

  /**
   * Remove a download source from all agent policies that are using it, and replace the output by the default ones.
   * @param soClient
   * @param esClient
   * @param downloadSourceId
   */
  async removeDefaultSourceFromAll(soClient, esClient, downloadSourceId) {
    const agentPolicies = (await soClient.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'download_source_id'],
      searchFields: ['download_source_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(downloadSourceId),
      perPage: _constants2.SO_SEARCH_LIMIT
    })).saved_objects.map(so => ({
      id: so.id,
      ...so.attributes
    }));
    if (agentPolicies.length > 0) {
      await (0, _pMap.default)(agentPolicies, agentPolicy => this.update(soClient, esClient, agentPolicy.id, {
        download_source_id: agentPolicy.download_source_id === downloadSourceId ? null : agentPolicy.download_source_id
      }), {
        concurrency: 50
      });
    }
  }
  async bumpAllAgentPoliciesForDownloadSource(esClient, downloadSourceId, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'download_source_id', 'namespaces'],
      searchFields: ['download_source_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(downloadSourceId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, esClient, currentPolicies.saved_objects, options);
  }
  async bumpAllAgentPoliciesForFleetServerHosts(esClient, fleetServerHostId, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: SAVED_OBJECT_TYPE,
      fields: ['revision', 'fleet_server_host_id', 'namespaces'],
      searchFields: ['fleet_server_host_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(fleetServerHostId),
      perPage: _constants2.SO_SEARCH_LIMIT
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, esClient, currentPolicies.saved_objects, options);
  }
  async getInactivityTimeouts(soClient) {
    const findRes = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      filter: `${SAVED_OBJECT_TYPE}.attributes.inactivity_timeout > 0`,
      fields: [`inactivity_timeout`]
    });
    const groupedResults = (0, _lodash.groupBy)(findRes.saved_objects, so => so.attributes.inactivity_timeout);
    return Object.entries(groupedResults).map(([inactivityTimeout, policies]) => ({
      inactivityTimeout: parseInt(inactivityTimeout, 10),
      policyIds: policies.map(policy => policy.id)
    }));
  }
  async turnOffAgentTamperProtections(soClient) {
    var _config$setup$agentPo, _config$setup;
    const agentPolicyFetcher = this.fetchAllAgentPolicies(soClient, {
      kuery: 'ingest-agent-policies.is_protected: true'
    });
    const updatedAgentPolicies = [];
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      const {
        saved_objects: bulkUpdateSavedObjects
      } = await soClient.bulkUpdate(agentPolicyPageResults.map(agentPolicy => {
        const {
          id,
          revision
        } = agentPolicy;
        return {
          id,
          type: SAVED_OBJECT_TYPE,
          attributes: {
            is_protected: false,
            revision: revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: 'system'
          }
        };
      }));
      updatedAgentPolicies.push(...bulkUpdateSavedObjects);
    }
    if (!updatedAgentPolicies.length) {
      return {
        updatedPolicies: null,
        failedPolicies: []
      };
    }
    const failedPolicies = [];
    updatedAgentPolicies.forEach(policy => {
      if (policy.error) {
        failedPolicies.push({
          id: policy.id,
          error: policy.error
        });
      }
    });
    const updatedPoliciesSuccess = updatedAgentPolicies.filter(policy => !policy.error);
    const config = _.appContextService.getConfig();
    const batchSize = (_config$setup$agentPo = config === null || config === void 0 ? void 0 : (_config$setup = config.setup) === null || _config$setup === void 0 ? void 0 : _config$setup.agentPolicySchemaUpgradeBatchSize) !== null && _config$setup$agentPo !== void 0 ? _config$setup$agentPo : 100;
    const policyIds = updatedPoliciesSuccess.map(policy => policy.id);
    await (0, _std.asyncForEach)((0, _lodash.chunk)(policyIds, batchSize), async policyIdsBatch => await this.deployPolicies(soClient, policyIdsBatch));
    return {
      updatedPolicies: updatedPoliciesSuccess,
      failedPolicies
    };
  }
  async getAllManagedAgentPolicies(soClient) {
    const {
      saved_objects: agentPolicies
    } = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      filter: (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, 'ingest-agent-policies.is_managed: true')
    });
    return agentPolicies;
  }
  fetchAllAgentPolicyIds(soClient, {
    perPage = 1000,
    kuery = undefined
  } = {}) {
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: SAVED_OBJECT_TYPE,
        perPage,
        sortField: 'created_at',
        sortOrder: 'asc',
        fields: ['id'],
        filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
      },
      resultsMapper: data => {
        return data.saved_objects.map(agentPolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: agentPolicySO.id,
            savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
          });
          return agentPolicySO.id;
        });
      }
    });
  }
  fetchAllAgentPolicies(soClient, {
    perPage = 1000,
    kuery,
    sortOrder = 'asc',
    sortField = 'created_at',
    fields = []
  } = {}) {
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: SAVED_OBJECT_TYPE,
        sortField,
        sortOrder,
        perPage,
        fields,
        filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
      },
      resultsMapper(data) {
        return data.saved_objects.map(agentPolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: agentPolicySO.id,
            savedObjectType: _constants2.AGENT_POLICY_SAVED_OBJECT_TYPE
          });
          return (0, _utils.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
        });
      }
    });
  }
  checkTamperProtectionLicense(agentPolicy) {
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected && !_license.licenseService.isPlatinum()) {
      throw new _errors.FleetUnauthorizedError('Tamper protection requires Platinum license');
    }
  }
  async checkForValidUninstallToken(agentPolicy, policyId) {
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected) {
      const uninstallTokenService = _.appContextService.getUninstallTokenService();
      const uninstallTokenError = await (uninstallTokenService === null || uninstallTokenService === void 0 ? void 0 : uninstallTokenService.checkTokenValidityForPolicy(policyId));
      if (uninstallTokenError) {
        throw new _errors.FleetError(`Cannot enable Agent Tamper Protection: ${uninstallTokenError.error.message}`);
      }
    }
  }
  checkAgentless(agentPolicy) {
    const cloudSetup = _.appContextService.getCloud();
    if ((!(cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled) || !_.appContextService.getExperimentalFeatures().agentless) && (agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.supports_agentless) !== undefined) {
      throw new _errors.AgentPolicyInvalidError('supports_agentless is only allowed in serverless environments that support the agentless feature');
    }
  }
  packagePoliciesWithSingleAndMultiplePolicies(packagePolicies) {
    // Find package policies that don't have multiple agent policies and mark them for deletion
    const policiesWithSingleAP = packagePolicies.filter(policy => !(policy !== null && policy !== void 0 && policy.policy_ids) || (policy === null || policy === void 0 ? void 0 : policy.policy_ids.length) <= 1);
    const policiesWithMultipleAP = packagePolicies.filter(policy => (policy === null || policy === void 0 ? void 0 : policy.policy_ids) && (policy === null || policy === void 0 ? void 0 : policy.policy_ids.length) > 1);
    return {
      policiesWithSingleAP,
      policiesWithMultipleAP
    };
  }
}
const agentPolicyService = exports.agentPolicyService = new AgentPolicyService();
async function addPackageToAgentPolicy(soClient, esClient, agentPolicy, packageInfo, packagePolicyName, packagePolicyId, packagePolicyDescription, transformPackagePolicy, bumpAgentPolicyRevison = false) {
  var _agentPolicy$namespac;
  const basePackagePolicy = (0, _services.packageToPackagePolicy)(packageInfo, agentPolicy.id, (_agentPolicy$namespac = agentPolicy.namespace) !== null && _agentPolicy$namespac !== void 0 ? _agentPolicy$namespac : 'default', packagePolicyName, packagePolicyDescription);
  const newPackagePolicy = transformPackagePolicy ? transformPackagePolicy(basePackagePolicy) : basePackagePolicy;

  // If an ID is provided via preconfiguration, use that value. Otherwise fall back to
  // a UUID v5 value seeded from the agent policy's ID and the provided package policy name.
  const id = packagePolicyId ? String(packagePolicyId) : (0, _uuid.v5)(`${agentPolicy.id}-${packagePolicyName}`, _constants3.UUID_V5_NAMESPACE);
  await _package_policy.packagePolicyService.create(soClient, esClient, newPackagePolicy, {
    id,
    bumpRevision: bumpAgentPolicyRevison,
    skipEnsureInstalled: true,
    skipUniqueNameVerification: true,
    overwrite: true,
    force: true,
    // To add package to managed policy we need the force flag
    packageInfo
  });
}