"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = exports.putSettingsHandler = exports.getSettingsHandler = void 0;
var _constants = require("../../../common/constants");
var _constants2 = require("../../constants");
var _types = require("../../types");
var _errors = require("../../errors");
var _services = require("../../services");
var _enrollment_settings_handler = require("./enrollment_settings_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSettingsHandler = async (context, request, response) => {
  const soClient = (await context.fleet).internalSoClient;
  try {
    const settings = await _services.settingsService.getSettings(soClient);
    const body = {
      item: settings
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Settings not found`
        }
      });
    }
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getSettingsHandler = getSettingsHandler;
const putSettingsHandler = async (context, request, response) => {
  const soClient = (await context.fleet).internalSoClient;
  const esClient = (await context.core).elasticsearch.client.asInternalUser;
  const user = _services.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  try {
    const settings = await _services.settingsService.saveSettings(soClient, request.body);
    await _services.agentPolicyService.bumpAllAgentPolicies(esClient, {
      user
    });
    const body = {
      item: settings
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Settings not found`
        }
      });
    }
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.putSettingsHandler = putSettingsHandler;
const registerRoutes = router => {
  router.versioned.get({
    path: _constants2.SETTINGS_API_ROUTES.INFO_PATTERN,
    fleetAuthz: {
      fleet: {
        readSettings: true
      }
    },
    description: `Get settings`
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetSettingsRequestSchema
    }
  }, getSettingsHandler);
  router.versioned.put({
    path: _constants2.SETTINGS_API_ROUTES.UPDATE_PATTERN,
    fleetAuthz: {
      fleet: {
        allSettings: true
      }
    },
    description: `Update settings`
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PutSettingsRequestSchema
    }
  }, putSettingsHandler);
  router.versioned.get({
    path: _constants2.SETTINGS_API_ROUTES.ENROLLMENT_INFO_PATTERN,
    fleetAuthz: authz => {
      return authz.fleet.addAgents || authz.fleet.addFleetServers;
    },
    description: `Get enrollment settings`
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetEnrollmentSettingsRequestSchema
    }
  }, _enrollment_settings_handler.getEnrollmentSettingsHandler);
};
exports.registerRoutes = registerRoutes;