"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentDiagnosticsTab = void 0;
var _eui = require("@elastic/eui");
var _react = _interopRequireWildcard(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _i18n = require("@kbn/i18n");
var _services = require("../../../../../../../../common/services");
var _hooks = require("../../../../../hooks");
var _agent_request_diagnostics_modal = require("../../../components/agent_request_diagnostics_modal");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AgentDiagnosticsTab = ({
  agent
}) => {
  const authz = (0, _hooks.useAuthz)();
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const {
    getAbsolutePath
  } = (0, _hooks.useLink)();
  const [isLoading, setIsLoading] = (0, _react.useState)(true);
  const [isShowingExpiredEntries, setIsShowingExpiredEntries] = (0, _react.useState)(false);
  const [visibleDiagnosticsEntries, setVisibleDiagnosticEntries] = (0, _react.useState)([]);
  const [allDiagnosticsEntries, setAllDiagnosticEntries] = (0, _react.useState)([]);
  const [prevDiagnosticsEntries, setPrevDiagnosticEntries] = (0, _react.useState)([]);
  const [loadInterval, setLoadInterval] = (0, _react.useState)(10000);
  const [isRequestDiagnosticsModalOpen, setIsRequestDiagnosticsModalOpen] = (0, _react.useState)(false);
  const loadData = (0, _react.useCallback)(async () => {
    try {
      const uploadsResponse = await (0, _hooks.sendGetAgentUploads)(agent.id);
      const error = uploadsResponse.error;
      if (error) {
        throw error;
      }
      if (!uploadsResponse.data) {
        throw new Error('No data');
      }
      const entries = uploadsResponse.data.items;
      setAllDiagnosticEntries(entries);
      setIsLoading(false);

      // query faster if an action is in progress, for quicker feedback
      if (entries.some(entry => entry.status === 'IN_PROGRESS' || entry.status === 'AWAITING_UPLOAD')) {
        setLoadInterval(3000);
      } else {
        setLoadInterval(10000);
      }
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.errorLoadingUploadsNotificationTitle', {
          defaultMessage: 'Error loading diagnostics uploads'
        })
      });
    }
  }, [agent.id, notifications.toasts, setLoadInterval]);
  const deleteFile = fileId => {
    (0, _hooks.sendDeleteAgentUpload)(fileId).then(({
      data,
      error
    }) => {
      if (error || (data === null || data === void 0 ? void 0 : data.deleted) === false) {
        notifications.toasts.addError(error || new Error('Request returned `deleted: false`'), {
          title: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.errorDeletingUploadNotificationTitle', {
            defaultMessage: 'Error deleting diagnostics file'
          })
        });
      } else {
        notifications.toasts.addSuccess({
          title: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.successDeletingUploadNotificationTitle', {
            defaultMessage: 'Diagnostics file deleted'
          })
        });
      }
      loadData();
    });
  };
  (0, _react.useEffect)(() => {
    loadData();
    const interval = setInterval(async () => {
      loadData();
    }, loadInterval);
    const cleanup = () => {
      if (interval) {
        clearInterval(interval);
      }
    };
    return cleanup;
  }, [loadData, loadInterval]);
  (0, _react.useEffect)(() => {
    setPrevDiagnosticEntries(allDiagnosticsEntries);
    if (prevDiagnosticsEntries.length > 0) {
      allDiagnosticsEntries.filter(newEntry => {
        const oldEntry = prevDiagnosticsEntries.find(entry => entry.id === newEntry.id);
        return newEntry.status === 'READY' && (!oldEntry || (oldEntry === null || oldEntry === void 0 ? void 0 : oldEntry.status) !== 'READY');
      }).forEach(entry => {
        notifications.toasts.addSuccess({
          title: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.readyNotificationTitle', {
            defaultMessage: 'Agent diagnostics {name} ready',
            values: {
              name: entry.name
            }
          })
        }, {
          toastLifeTimeMs: 5000
        });
      });
    }
  }, [prevDiagnosticsEntries, allDiagnosticsEntries, notifications.toasts]);
  (0, _react.useEffect)(() => {
    if (isShowingExpiredEntries) {
      setVisibleDiagnosticEntries(allDiagnosticsEntries);
    } else {
      setVisibleDiagnosticEntries(allDiagnosticsEntries.filter(entry => entry.status !== 'EXPIRED'));
    }
  }, [allDiagnosticsEntries, isShowingExpiredEntries]);
  const columns = [{
    field: 'id',
    name: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.tableColumns.fileLabelText', {
      defaultMessage: 'File'
    }),
    render: id => {
      const currentItem = allDiagnosticsEntries.find(item => item.id === id);
      return (currentItem === null || currentItem === void 0 ? void 0 : currentItem.status) === 'READY' ? /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
        href: getAbsolutePath(currentItem === null || currentItem === void 0 ? void 0 : currentItem.filePath),
        download: true,
        target: "_blank"
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        type: "download"
      }), " \xA0 ", currentItem === null || currentItem === void 0 ? void 0 : currentItem.name) : (currentItem === null || currentItem === void 0 ? void 0 : currentItem.status) === 'IN_PROGRESS' || (currentItem === null || currentItem === void 0 ? void 0 : currentItem.status) === 'AWAITING_UPLOAD' ? /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
        color: "subdued",
        disabled: true
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null), " \xA0", /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.fleet.requestDiagnostics.generatingText",
        defaultMessage: "Generating diagnostics file..."
      })) : /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
        color: "subdued",
        disabled: true
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
        gutterSize: "s",
        direction: "row",
        alignItems: "center"
      }, currentItem !== null && currentItem !== void 0 && currentItem.error ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
        grow: false
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
        content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.fleet.requestDiagnostics.errorGeneratingFileMessage",
          defaultMessage: "Error generating file: {reason}",
          values: {
            reason: currentItem.error
          }
        })
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        type: "warning",
        color: "danger"
      }))) : currentItem !== null && currentItem !== void 0 && currentItem.status ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
        grow: false
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
        content: currentItem.status
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        type: "warning",
        color: "danger"
      }))) : null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, currentItem === null || currentItem === void 0 ? void 0 : currentItem.name)));
    }
  }, {
    field: 'id',
    name: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.tableColumns.dateLabelText', {
      defaultMessage: 'Date'
    }),
    dataType: 'date',
    render: id => {
      const currentItem = allDiagnosticsEntries.find(item => item.id === id);
      return /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
        size: "s",
        color: (currentItem === null || currentItem === void 0 ? void 0 : currentItem.status) === 'READY' ? 'default' : 'subdued'
      }, (0, _eui.formatDate)(currentItem === null || currentItem === void 0 ? void 0 : currentItem.createTime, 'lll'));
    }
  }, ...(authz.fleet.allAgents ? [{
    name: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.tableColumns.actionsLabelText', {
      defaultMessage: 'Actions'
    }),
    width: '70px',
    actions: [{
      type: 'icon',
      icon: 'trash',
      color: 'danger',
      name: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.tableColumns.deleteButtonText', {
        defaultMessage: 'Delete'
      }),
      available: item => item.status === 'READY',
      description: _i18n.i18n.translate('xpack.fleet.requestDiagnostics.tableColumns.deleteButtonDesc', {
        defaultMessage: 'Delete diagnostics file'
      }),
      onClick: item => {
        deleteFile(item.id);
      }
    }]
  }] : [])];
  const requestDiagnosticsButton = /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
    fill: true,
    size: "s",
    onClick: () => {
      setIsRequestDiagnosticsModalOpen(true);
    },
    disabled: !(0, _services.isAgentRequestDiagnosticsSupported)(agent) || !authz.fleet.readAgents
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.fleet.requestDiagnostics.diagnosticsOneButton",
    defaultMessage: "Request diagnostics .zip"
  }));
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, isRequestDiagnosticsModalOpen && /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_agent_request_diagnostics_modal.AgentRequestDiagnosticsModal, {
    agents: [agent],
    agentCount: 1,
    onClose: () => {
      setIsRequestDiagnosticsModalOpen(false);
      loadData();
    }
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
    size: "s"
  }, /*#__PURE__*/_react.default.createElement("p", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.fleet.requestDiagnostics.calloutText",
    defaultMessage: "Diagnostics files are stored in Elasticsearch, and as such can incur storage costs. By default, files are periodically deleted via an ILM policy."
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "row",
    gutterSize: "m",
    alignItems: "center",
    justifyContent: "spaceBetween"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, (0, _services.isAgentRequestDiagnosticsSupported)(agent) ? requestDiagnosticsButton : /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
    content: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.requestDiagnostics.notSupportedTooltip",
      defaultMessage: "Requesting agent diagnostics is not supported for agents before version {version}.",
      values: {
        version: _services.MINIMUM_DIAGNOSTICS_AGENT_VERSION
      }
    })
  }, requestDiagnosticsButton)), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    label: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.requestDiagnostics.showExpiredFilesLabel",
      defaultMessage: "Show expired file requests"
    }),
    checked: isShowingExpiredEntries,
    onChange: e => setIsShowingExpiredEntries(e.target.checked)
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  }), isLoading ? /*#__PURE__*/_react.default.createElement(_eui.EuiSkeletonText, {
    lines: 3
  }) : /*#__PURE__*/_react.default.createElement(_eui.EuiBasicTable, {
    items: visibleDiagnosticsEntries,
    columns: columns
  }));
};
exports.AgentDiagnosticsTab = AgentDiagnosticsTab;