"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNotUpgradeableMessage = exports.differsOnlyInPatch = exports.VERSION_MISSING_ERROR = exports.UNENROLLED_ERROR = exports.SELECTED_VERSION_ERROR = exports.RUNNING_SELECTED_VERSION_ERROR = exports.ONGOING_UNEROLLMENT_ERROR = exports.NOT_UPGRADEABLE_ERROR = exports.LATEST_VERSION_NOT_VALID_ERROR = exports.INVALID_VERSION_ERROR = exports.DOWNGRADE_NOT_ALLOWED_ERROR = exports.ALREADY_UPGRADED_ERROR = exports.AGENT_UPGRADE_COOLDOWN_IN_MIN = exports.AGENT_UPGARDE_DETAILS_SUPPORTED_VERSION = exports.AGENT_ON_GREATER_VERSION_ERROR = exports.AGENT_ALREADY_ON_LATEST_ERROR = void 0;
exports.getRecentUpgradeInfoForAgent = getRecentUpgradeInfoForAgent;
exports.isAgentInWatchingState = isAgentInWatchingState;
exports.isAgentUpgradeAvailable = isAgentUpgradeAvailable;
exports.isAgentUpgradeable = isAgentUpgradeable;
exports.isAgentUpgradeableToVersion = isAgentUpgradeableToVersion;
exports.isAgentUpgrading = isAgentUpgrading;
exports.isAgentVersionLessThanLatest = void 0;
var _coerce = _interopRequireDefault(require("semver/functions/coerce"));
var _lt = _interopRequireDefault(require("semver/functions/lt"));
var _gt = _interopRequireDefault(require("semver/functions/gt"));
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _eq = _interopRequireDefault(require("semver/functions/eq"));
var _moment = _interopRequireDefault(require("moment"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AGENT_UPGRADE_COOLDOWN_IN_MIN = exports.AGENT_UPGRADE_COOLDOWN_IN_MIN = 10;
const AGENT_UPGARDE_DETAILS_SUPPORTED_VERSION = exports.AGENT_UPGARDE_DETAILS_SUPPORTED_VERSION = '8.12.0';

// Error messages for agent not upgradeable
const VERSION_MISSING_ERROR = exports.VERSION_MISSING_ERROR = `agent version is missing.`;
const UNENROLLED_ERROR = exports.UNENROLLED_ERROR = `agent has been unenrolled.`;
const ONGOING_UNEROLLMENT_ERROR = exports.ONGOING_UNEROLLMENT_ERROR = `agent is being unenrolled.`;
const NOT_UPGRADEABLE_ERROR = exports.NOT_UPGRADEABLE_ERROR = `agent cannot be upgraded through Fleet. It may be running in a container or it is not installed as a service.`;
const ALREADY_UPGRADED_ERROR = exports.ALREADY_UPGRADED_ERROR = `agent is already being upgraded.`;
const INVALID_VERSION_ERROR = exports.INVALID_VERSION_ERROR = 'agent version is not valid.';
const SELECTED_VERSION_ERROR = exports.SELECTED_VERSION_ERROR = 'the selected version is not valid.';
const RUNNING_SELECTED_VERSION_ERROR = exports.RUNNING_SELECTED_VERSION_ERROR = `agent is already running on the selected version.`;
const DOWNGRADE_NOT_ALLOWED_ERROR = exports.DOWNGRADE_NOT_ALLOWED_ERROR = `agent does not support downgrades.`;
const LATEST_VERSION_NOT_VALID_ERROR = exports.LATEST_VERSION_NOT_VALID_ERROR = 'latest version is not valid.';
const AGENT_ALREADY_ON_LATEST_ERROR = exports.AGENT_ALREADY_ON_LATEST_ERROR = `agent is already running on the latest available version.`;
const AGENT_ON_GREATER_VERSION_ERROR = exports.AGENT_ON_GREATER_VERSION_ERROR = `agent is running on a version greater than the latest available version.`;
function isAgentUpgradeAvailable(agent, latestAgentVersion) {
  var _agent$local_metadata, _agent$local_metadata2, _agent$local_metadata3;
  return latestAgentVersion && isAgentUpgradeable(agent) && (agent === null || agent === void 0 ? void 0 : (_agent$local_metadata = agent.local_metadata) === null || _agent$local_metadata === void 0 ? void 0 : (_agent$local_metadata2 = _agent$local_metadata.elastic) === null || _agent$local_metadata2 === void 0 ? void 0 : (_agent$local_metadata3 = _agent$local_metadata2.agent) === null || _agent$local_metadata3 === void 0 ? void 0 : _agent$local_metadata3.version) && isAgentVersionLessThanLatest(agent.local_metadata.elastic.agent.version, latestAgentVersion);
}
function isAgentUpgradeable(agent) {
  var _agent$local_metadata4, _agent$local_metadata5, _agent$local_metadata6;
  if (agent.unenrollment_started_at || agent.unenrolled_at) {
    return false;
  }
  if (!((_agent$local_metadata4 = agent.local_metadata) !== null && _agent$local_metadata4 !== void 0 && (_agent$local_metadata5 = _agent$local_metadata4.elastic) !== null && _agent$local_metadata5 !== void 0 && (_agent$local_metadata6 = _agent$local_metadata5.agent) !== null && _agent$local_metadata6 !== void 0 && _agent$local_metadata6.upgradeable)) {
    return false;
  }
  if (isAgentUpgrading(agent)) {
    return false;
  }
  if (isAgentInWatchingState(agent)) {
    return false;
  }
  if (getRecentUpgradeInfoForAgent(agent).hasBeenUpgradedRecently) {
    return false;
  }
  return true;
}
function isAgentUpgradeableToVersion(agent, versionToUpgrade) {
  var _agent$local_metadata7, _agent$local_metadata8, _agent$local_metadata9;
  const isAgentUpgradeableCheck = isAgentUpgradeable(agent);
  if (!isAgentUpgradeableCheck) return false;
  let agentVersion;
  if (typeof (agent === null || agent === void 0 ? void 0 : (_agent$local_metadata7 = agent.local_metadata) === null || _agent$local_metadata7 === void 0 ? void 0 : (_agent$local_metadata8 = _agent$local_metadata7.elastic) === null || _agent$local_metadata8 === void 0 ? void 0 : (_agent$local_metadata9 = _agent$local_metadata8.agent) === null || _agent$local_metadata9 === void 0 ? void 0 : _agent$local_metadata9.version) === 'string') {
    agentVersion = agent.local_metadata.elastic.agent.version;
  } else {
    return false;
  }
  if (!versionToUpgrade) {
    return false;
  }
  return isNotDowngrade(agentVersion, versionToUpgrade);
}
const isAgentVersionLessThanLatest = (agentVersion, latestAgentVersion) => {
  // make sure versions are only the number before comparison
  const agentVersionNumber = (0, _coerce.default)(agentVersion);
  if (!agentVersionNumber) throw new Error(`${INVALID_VERSION_ERROR}`);
  const latestAgentVersionNumber = (0, _coerce.default)(latestAgentVersion);
  if (!latestAgentVersionNumber) throw new Error(`${LATEST_VERSION_NOT_VALID_ERROR}`);
  return (0, _lt.default)(agentVersionNumber, latestAgentVersionNumber);
};

// Based on the previous, returns a detailed message explaining why the agent is not upgradeable
exports.isAgentVersionLessThanLatest = isAgentVersionLessThanLatest;
const getNotUpgradeableMessage = (agent, latestAgentVersion, versionToUpgrade) => {
  var _agent$local_metadata10, _agent$local_metadata11, _agent$local_metadata12;
  let agentVersion;
  if (typeof (agent === null || agent === void 0 ? void 0 : (_agent$local_metadata10 = agent.local_metadata) === null || _agent$local_metadata10 === void 0 ? void 0 : (_agent$local_metadata11 = _agent$local_metadata10.elastic) === null || _agent$local_metadata11 === void 0 ? void 0 : (_agent$local_metadata12 = _agent$local_metadata11.agent) === null || _agent$local_metadata12 === void 0 ? void 0 : _agent$local_metadata12.version) === 'string') {
    agentVersion = agent.local_metadata.elastic.agent.version;
  } else {
    return VERSION_MISSING_ERROR;
  }
  if (agent.unenrolled_at) {
    return UNENROLLED_ERROR;
  }
  if (agent.unenrollment_started_at) {
    return ONGOING_UNEROLLMENT_ERROR;
  }
  if (!agent.local_metadata.elastic.agent.upgradeable) {
    return NOT_UPGRADEABLE_ERROR;
  }
  if (isAgentUpgrading(agent)) {
    return ALREADY_UPGRADED_ERROR;
  }
  if (getRecentUpgradeInfoForAgent(agent).hasBeenUpgradedRecently) {
    const timeToWaitMins = getRecentUpgradeInfoForAgent(agent).timeToWaitMins;
    const elapsedMinsSinceUpgrade = getRecentUpgradeInfoForAgent(agent).elapsedMinsSinceUpgrade;
    return `agent was upgraded ${elapsedMinsSinceUpgrade} minutes ago, please wait ${timeToWaitMins} minutes before attempting the upgrade again.`;
  }
  const agentVersionNumber = (0, _coerce.default)(agentVersion);
  if (!agentVersionNumber) return INVALID_VERSION_ERROR;
  if (versionToUpgrade !== undefined) {
    const versionToUpgradeNumber = (0, _coerce.default)(versionToUpgrade);
    if (!versionToUpgradeNumber) return SELECTED_VERSION_ERROR;
    if ((0, _eq.default)(agentVersionNumber, versionToUpgradeNumber)) return RUNNING_SELECTED_VERSION_ERROR;
    if ((0, _lt.default)(versionToUpgradeNumber, agentVersionNumber)) return DOWNGRADE_NOT_ALLOWED_ERROR;

    // explicitly allow this case - the agent is upgradeable
    if ((0, _gt.default)(versionToUpgradeNumber, agentVersionNumber)) return undefined;
  }
  const latestAgentVersionNumber = (0, _coerce.default)(latestAgentVersion);
  if (!latestAgentVersionNumber) return LATEST_VERSION_NOT_VALID_ERROR;
  if ((0, _gt.default)(agentVersionNumber, latestAgentVersionNumber)) return AGENT_ON_GREATER_VERSION_ERROR;

  // in all the other cases, the agent is upgradeable; don't return any message.
  return undefined;
};
exports.getNotUpgradeableMessage = getNotUpgradeableMessage;
const isNotDowngrade = (agentVersion, versionToUpgrade) => {
  const agentVersionNumber = (0, _coerce.default)(agentVersion);
  if (!agentVersionNumber) throw new Error(`${INVALID_VERSION_ERROR}`);
  const versionToUpgradeNumber = (0, _coerce.default)(versionToUpgrade);
  if (!versionToUpgradeNumber) return true;
  return (0, _gt.default)(versionToUpgradeNumber, agentVersionNumber);
};
function getRecentUpgradeInfoForAgent(agent) {
  // no need for 10m wait if agent has upgrade details watching state
  const agentHasUpgradeDetailsSupport = (0, _gte.default)(agent.local_metadata.elastic.agent.version, AGENT_UPGARDE_DETAILS_SUPPORTED_VERSION);
  if (!agent.upgraded_at || agentHasUpgradeDetailsSupport) {
    return {
      hasBeenUpgradedRecently: false,
      timeToWaitMs: 0,
      timeToWaitMins: 0,
      elapsedMinsSinceUpgrade: 0
    };
  }
  const elapsedSinceUpgradeInMillis = Date.now() - Date.parse(agent.upgraded_at);
  const elapsedMins = _moment.default.duration(elapsedSinceUpgradeInMillis, 'milliseconds').asMinutes();
  const elapsedMinsSinceUpgrade = Math.ceil(elapsedMins);
  const timeToWaitMs = AGENT_UPGRADE_COOLDOWN_IN_MIN * 6e4 - elapsedSinceUpgradeInMillis;
  const hasBeenUpgradedRecently = elapsedSinceUpgradeInMillis / 6e4 < AGENT_UPGRADE_COOLDOWN_IN_MIN;
  const timeToWait = _moment.default.duration(timeToWaitMs, 'milliseconds').asMinutes();
  const timeToWaitMins = Math.ceil(timeToWait);
  return {
    hasBeenUpgradedRecently,
    timeToWaitMs,
    elapsedMinsSinceUpgrade,
    timeToWaitMins
  };
}
function isAgentInWatchingState(agent) {
  var _agent$upgrade_detail;
  return ((_agent$upgrade_detail = agent.upgrade_details) === null || _agent$upgrade_detail === void 0 ? void 0 : _agent$upgrade_detail.state) === 'UPG_WATCHING';
}
function isAgentUpgrading(agent) {
  if (agent.upgrade_details) {
    return agent.upgrade_details.state !== 'UPG_FAILED';
  }
  return agent.upgrade_started_at && !agent.upgraded_at;
}
const differsOnlyInPatch = (versionA, versionB, allowEqualPatch = true) => {
  const [majorA, minorA, patchA] = versionA.split('.');
  const [majorB, minorB, patchB] = versionB.split('.');
  return majorA === majorB && minorA === minorB && (allowEqualPatch ? patchA >= patchB : patchA > patchB);
};
exports.differsOnlyInPatch = differsOnlyInPatch;