"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_BLOB_STORE_SIZE_BYTES = exports.ElasticsearchBlobStorageClient = exports.BLOB_STORAGE_SYSTEM_INDEX_NAME = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
var _elasticsearch = require("@elastic/elasticsearch");
var _std = require("@kbn/std");
var _promises = require("stream/promises");
var _util = require("util");
var _rxjs = require("rxjs");
var _ebtTools = require("@kbn/ebt-tools");
var _lodash = require("lodash");
var _plugin = require("../../../plugin");
var _performance = require("../../../performance");
var _content_stream = require("./content_stream");
var _mappings = require("./mappings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Export this value for convenience to be used in tests. Do not use outside of
 * this adapter.
 * @internal
 */
const BLOB_STORAGE_SYSTEM_INDEX_NAME = exports.BLOB_STORAGE_SYSTEM_INDEX_NAME = '.kibana_blob_storage';
const MAX_BLOB_STORE_SIZE_BYTES = exports.MAX_BLOB_STORE_SIZE_BYTES = 50 * 1024 * 1024 * 1024; // 50 GiB

class ElasticsearchBlobStorageClient {
  /**
   * Call this function once to globally set the concurrent transfer (upload/download) limit for
   * all {@link ElasticsearchBlobStorageClient} instances.
   */
  static configureConcurrentTransfers(capacity) {
    if (Array.isArray(capacity)) {
      this.defaultUploadSemaphore = new _std.Semaphore(capacity[0]);
      this.defaultDownloadSemaphore = new _std.Semaphore(capacity[1]);
    } else {
      this.defaultUploadSemaphore = new _std.Semaphore(capacity);
      this.defaultDownloadSemaphore = new _std.Semaphore(capacity);
    }
  }
  constructor(esClient, index = BLOB_STORAGE_SYSTEM_INDEX_NAME, chunkSize, logger,
  /**
   * Override the default concurrent upload limit by passing in a different
   * semaphore
   */
  uploadSemaphore = ElasticsearchBlobStorageClient.defaultUploadSemaphore,
  /**
   * Override the default concurrent download limit by passing in a different
   * semaphore
   */
  downloadSemaphore = ElasticsearchBlobStorageClient.defaultDownloadSemaphore, /** Indicates that the index provided is an alias (changes how content is retrieved internally) */
  indexIsAlias = false) {
    this.esClient = esClient;
    this.index = index;
    this.chunkSize = chunkSize;
    this.logger = logger;
    this.uploadSemaphore = uploadSemaphore;
    this.downloadSemaphore = downloadSemaphore;
    this.indexIsAlias = indexIsAlias;
    (0, _assert.default)(this.uploadSemaphore, `No default semaphore provided and no semaphore was passed in for uploads.`);
    (0, _assert.default)(this.downloadSemaphore, `No default semaphore provided and no semaphore was passed in for downloads.`);
  }

  /**
   * This function acts as a singleton i.t.o. execution: it can only be called once per index.
   * Subsequent calls for the same index should not re-execute it.
   */

  async upload(src, options = {}) {
    const {
      transforms,
      id
    } = options;
    await ElasticsearchBlobStorageClient.createIndexIfNotExists(this.index, this.esClient, this.logger, this.indexIsAlias);
    const processUpload = async () => {
      try {
        const analytics = _plugin.FilesPlugin.getAnalytics();
        const dest = (0, _content_stream.getWritableContentStream)({
          id,
          client: this.esClient,
          index: this.index,
          logger: this.logger.get('content-stream-upload'),
          parameters: {
            maxChunkSize: this.chunkSize
          },
          indexIsAlias: this.indexIsAlias
        });
        const start = performance.now();
        await (0, _promises.pipeline)([src, ...(transforms !== null && transforms !== void 0 ? transforms : []), dest]);
        const end = performance.now();
        const _id = dest.getContentReferenceId();
        const size = dest.getBytesWritten();
        const perfArgs = {
          eventName: _performance.FILE_UPLOAD_PERFORMANCE_EVENT_NAME,
          duration: end - start,
          key1: 'size',
          value1: size,
          meta: {
            datasource: 'es',
            id: _id,
            index: this.index,
            chunkSize: this.chunkSize
          }
        };
        if (analytics) {
          (0, _ebtTools.reportPerformanceMetricEvent)(analytics, perfArgs);
        }
        return {
          id: _id,
          size
        };
      } catch (e) {
        this.logger.error(`Could not write chunks to Elasticsearch for id ${id}: ${e}`);
        throw e;
      }
    };
    return (0, _rxjs.lastValueFrom)((0, _rxjs.defer)(processUpload).pipe(this.uploadSemaphore.acquire()));
  }
  getReadableContentStream(id, size) {
    return (0, _content_stream.getReadableContentStream)({
      id,
      client: this.esClient,
      index: this.index,
      logger: this.logger.get('content-stream-download'),
      parameters: {
        size
      },
      indexIsAlias: this.indexIsAlias
    });
  }
  async download({
    id,
    size
  }) {
    // The refresh interval is set to 10s. To avoid throwing an error if the user tries to download a file
    // right after uploading it, we refresh the index before downloading the file.
    await this.esClient.indices.refresh({
      index: this.index
    });
    return (0, _rxjs.firstValueFrom)((0, _rxjs.defer)(() => Promise.resolve(this.getReadableContentStream(id, size))).pipe(this.downloadSemaphore.acquire()));
  }
  async delete(id) {
    try {
      // The refresh interval is set to 10s. To avoid throwing an error if the user tries to delete a file
      // right after uploading it, we refresh the index before deleting the file.
      await this.esClient.indices.refresh({
        index: this.index
      });
      const dest = (0, _content_stream.getWritableContentStream)({
        id,
        client: this.esClient,
        index: this.index,
        logger: this.logger.get('content-stream-delete'),
        indexIsAlias: this.indexIsAlias
      });
      /** @note Overwriting existing content with an empty buffer to remove all the chunks. */
      await (0, _util.promisify)(dest.end.bind(dest, '', 'utf8'))();
    } catch (e) {
      this.logger.error(`Could not delete file: ${e}`);
      throw e;
    }
  }
}
exports.ElasticsearchBlobStorageClient = ElasticsearchBlobStorageClient;
(0, _defineProperty2.default)(ElasticsearchBlobStorageClient, "defaultUploadSemaphore", void 0);
(0, _defineProperty2.default)(ElasticsearchBlobStorageClient, "defaultDownloadSemaphore", void 0);
(0, _defineProperty2.default)(ElasticsearchBlobStorageClient, "createIndexIfNotExists", (0, _lodash.memoize)(async (index, esClient, logger, indexIsAlias) => {
  // We don't attempt to create the index if it is an Alias/DS
  if (indexIsAlias) {
    logger.debug(`No need to create index [${index}] as it is an Alias or DS.`);
    return;
  }
  try {
    if (await esClient.indices.exists({
      index
    })) {
      logger.debug(`[${index}] already exists. Nothing to do`);
      return;
    }
    logger.info(`Creating [${index}] index for Elasticsearch blob store.`);
    await esClient.indices.create({
      index,
      wait_for_active_shards: 'all',
      body: {
        settings: {
          number_of_shards: 1,
          auto_expand_replicas: '0-1'
        },
        mappings: _mappings.mappings
      }
    });
  } catch (e) {
    if (e instanceof _elasticsearch.errors.ResponseError && e.statusCode === 400) {
      logger.warn(`Unable to create blob storage index [${index}], it may have been created already.`);
    }
    // best effort
  }
}));