"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationFieldsRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fields_metadata = require("../../../../common/fields_metadata");
var _hashed_cache = require("../../../../common/hashed_cache");
var _common = require("../../../../common");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class IntegrationFieldsRepository {
  constructor(fieldsExtractor) {
    (0, _defineProperty2.default)(this, "cache", void 0);
    (0, _defineProperty2.default)(this, "storeFieldsInCache", (cacheKey, extractedFieldsMetadata) => {
      const cachedIntegration = this.cache.get(cacheKey);
      if (!cachedIntegration) {
        this.cache.set(cacheKey, extractedFieldsMetadata);
      } else {
        this.cache.set(cacheKey, {
          ...cachedIntegration,
          ...extractedFieldsMetadata
        });
      }
    });
    (0, _defineProperty2.default)(this, "getCacheKey", params => params);
    (0, _defineProperty2.default)(this, "mapExtractedFieldsToFieldMetadataTree", extractedFields => {
      const datasetGroups = Object.entries(extractedFields);
      return datasetGroups.reduce((integrationGroup, [datasetName, datasetGroup]) => {
        const datasetFieldsEntries = Object.entries(datasetGroup);
        integrationGroup[datasetName] = datasetFieldsEntries.reduce((datasetFields, [fieldName, field]) => {
          datasetFields[fieldName] = _common.FieldMetadata.create({
            ...field,
            source: 'integration'
          });
          return datasetFields;
        }, {});
        return integrationGroup;
      }, {});
    });
    this.fieldsExtractor = fieldsExtractor;
    this.cache = new _hashed_cache.HashedCache();
  }
  async getByName(fieldName, {
    integration,
    dataset
  }) {
    let field = this.getCachedField(fieldName, {
      integration,
      dataset
    });
    if (!field) {
      try {
        await this.extractFields({
          integration,
          dataset
        });
      } catch (error) {
        throw new _errors.PackageNotFoundError(error.message);
      }
      field = this.getCachedField(fieldName, {
        integration,
        dataset
      });
    }
    return field;
  }
  static create({
    integrationFieldsExtractor
  }) {
    return new IntegrationFieldsRepository(integrationFieldsExtractor);
  }
  async extractFields({
    integration,
    dataset
  }) {
    const cacheKey = this.getCacheKey({
      integration,
      dataset
    });
    const cachedIntegration = this.cache.get(cacheKey);
    if (cachedIntegration) {
      return undefined;
    }
    return this.fieldsExtractor({
      integration,
      dataset
    }).then(this.mapExtractedFieldsToFieldMetadataTree).then(fieldMetadataTree => this.storeFieldsInCache(cacheKey, fieldMetadataTree));
  }
  getCachedField(fieldName, {
    integration,
    dataset
  }) {
    const cacheKey = this.getCacheKey({
      integration,
      dataset
    });
    const cachedIntegration = this.cache.get(cacheKey);
    const datasetName = dataset === _fields_metadata.ANY_DATASET ? null : dataset;

    // 1. Integration fields were never fetched
    if (!cachedIntegration) {
      return undefined;
    }

    // 2. Dataset is passed but was never fetched before
    if (datasetName && !cachedIntegration.hasOwnProperty(datasetName)) {
      return undefined;
    }

    // 3. Dataset is passed and it was previously fetched, should return the field
    if (datasetName && cachedIntegration.hasOwnProperty(datasetName)) {
      const targetDataset = cachedIntegration[datasetName];
      return targetDataset[fieldName];
    }

    // 4. Dataset is not passed, we attempt search on all stored datasets
    if (!datasetName) {
      // Merge all the available datasets into a unique field list. Overriding fields might occur in the process.
      const cachedDatasetsFields = Object.assign({}, ...Object.values(cachedIntegration));
      return cachedDatasetsFields[fieldName];
    }
  }
}
exports.IntegrationFieldsRepository = IntegrationFieldsRepository;