"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _common = require("../common");
var _feature_schema = require("./feature_schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class FeatureRegistry {
  constructor() {
    (0, _defineProperty2.default)(this, "locked", false);
    (0, _defineProperty2.default)(this, "kibanaFeatures", {});
    (0, _defineProperty2.default)(this, "esFeatures", {});
  }
  lockRegistration() {
    this.locked = true;
  }
  registerKibanaFeature(feature) {
    if (this.locked) {
      throw new Error(`Features are locked, can't register new features. Attempt to register ${feature.id} failed.`);
    }
    (0, _feature_schema.validateKibanaFeature)(feature);
    if (feature.id in this.kibanaFeatures || feature.id in this.esFeatures) {
      throw new Error(`Feature with id ${feature.id} is already registered.`);
    }
    const featureCopy = (0, _lodash.cloneDeep)(feature);
    this.kibanaFeatures[feature.id] = applyAutomaticPrivilegeGrants(featureCopy);
  }
  registerElasticsearchFeature(feature) {
    if (this.locked) {
      throw new Error(`Features are locked, can't register new features. Attempt to register ${feature.id} failed.`);
    }
    if (feature.id in this.kibanaFeatures || feature.id in this.esFeatures) {
      throw new Error(`Feature with id ${feature.id} is already registered.`);
    }
    (0, _feature_schema.validateElasticsearchFeature)(feature);
    const featureCopy = (0, _lodash.cloneDeep)(feature);
    this.esFeatures[feature.id] = featureCopy;
  }

  /**
   * Updates definitions for the registered features using configuration overrides, if any.
   */
  applyOverrides(overrides) {
    for (const [featureId, featureOverride] of Object.entries(overrides)) {
      var _featureOverride$subF;
      const feature = this.kibanaFeatures[featureId];
      if (!feature) {
        throw new Error(`Cannot override feature "${featureId}" since feature with such ID is not registered.`);
      }
      if (featureOverride.hidden) {
        feature.hidden = featureOverride.hidden;
      }

      // Note that the name doesn't currently support localizable strings. We'll revisit this approach when i18n support
      // becomes necessary.
      if (featureOverride.name) {
        feature.name = featureOverride.name;
      }
      if (featureOverride.category) {
        feature.category = _server.DEFAULT_APP_CATEGORIES[featureOverride.category];
      }
      if (featureOverride.order != null) {
        feature.order = featureOverride.order;
      }
      if (featureOverride.privileges) {
        for (const [privilegeId, privilegeOverride] of Object.entries(featureOverride.privileges)) {
          var _feature$privileges;
          const typedPrivilegeId = privilegeId;
          const targetPrivilege = (_feature$privileges = feature.privileges) === null || _feature$privileges === void 0 ? void 0 : _feature$privileges[typedPrivilegeId];
          if (!targetPrivilege) {
            throw new Error(`Cannot override privilege "${privilegeId}" of feature "${featureId}" since "${privilegeId}" privilege is not registered.`);
          }
          for (const featureReference of (_privilegeOverride$co = privilegeOverride.composedOf) !== null && _privilegeOverride$co !== void 0 ? _privilegeOverride$co : []) {
            var _privilegeOverride$co, _referencedFeature$pr;
            const referencedFeature = this.kibanaFeatures[featureReference.feature];
            if (!referencedFeature) {
              throw new Error(`Cannot compose privilege "${privilegeId}" of feature "${featureId}" with privileges of feature "${featureReference.feature}" since such feature is not registered.`);
            }

            // Collect all known feature and sub-feature privileges for the referenced feature.
            const knownPrivileges = new Map(Object.entries((_referencedFeature$pr = referencedFeature.privileges) !== null && _referencedFeature$pr !== void 0 ? _referencedFeature$pr : {}).concat(collectSubFeaturesPrivileges(referencedFeature)));
            for (const privilegeReference of featureReference.privileges) {
              const referencedPrivilege = knownPrivileges.get(privilegeReference);
              if (!referencedPrivilege) {
                throw new Error(`Cannot compose privilege "${privilegeId}" of feature "${featureId}" with privilege "${privilegeReference}" of feature "${featureReference.feature}" since such privilege is not registered.`);
              }
            }
          }

          // It's safe to assume that `feature.privileges` is defined here since we've checked it above.
          feature.privileges[typedPrivilegeId] = {
            ...targetPrivilege,
            ...privilegeOverride
          };
        }
      }
      if ((_featureOverride$subF = featureOverride.subFeatures) !== null && _featureOverride$subF !== void 0 && _featureOverride$subF.privileges) {
        // Collect all known sub-feature privileges for the feature.
        const knownPrivileges = new Map(collectSubFeaturesPrivileges(feature));
        if (knownPrivileges.size === 0) {
          throw new Error(`Cannot override sub-feature privileges of feature "${featureId}" since it didn't register any.`);
        }
        for (const [privilegeId, privilegeOverride] of Object.entries(featureOverride.subFeatures.privileges)) {
          const targetPrivilege = knownPrivileges.get(privilegeId);
          if (!targetPrivilege) {
            throw new Error(`Cannot override sub-feature privilege "${privilegeId}" of feature "${featureId}" since "${privilegeId}" sub-feature privilege is not registered. Known sub-feature privileges are: ${Array.from(knownPrivileges.keys())}.`);
          }
          targetPrivilege.disabled = privilegeOverride.disabled;
          if (privilegeOverride.includeIn) {
            targetPrivilege.includeIn = privilegeOverride.includeIn;
          }
        }
      }
    }
  }
  getAllKibanaFeatures(license, ignoreLicense = false) {
    if (!this.locked) {
      throw new Error('Cannot retrieve Kibana features while registration is still open');
    }
    let features = Object.values(this.kibanaFeatures);
    const performLicenseCheck = license && !ignoreLicense;
    if (performLicenseCheck) {
      features = features.filter(feature => {
        var _feature$subFeatures;
        const filter = !feature.minimumLicense || license.hasAtLeast(feature.minimumLicense);
        if (!filter) return false;
        (_feature$subFeatures = feature.subFeatures) === null || _feature$subFeatures === void 0 ? void 0 : _feature$subFeatures.forEach(subFeature => {
          subFeature.privilegeGroups.forEach(group => {
            group.privileges = group.privileges.filter(privilege => !privilege.minimumLicense || license.hasAtLeast(privilege.minimumLicense));
          });
        });
        return true;
      });
    }
    return features.map(featureConfig => new _common.KibanaFeature(featureConfig));
  }
  getAllElasticsearchFeatures() {
    if (!this.locked) {
      throw new Error('Cannot retrieve elasticsearch features while registration is still open');
    }
    return Object.values(this.esFeatures).map(featureConfig => new _common.ElasticsearchFeature(featureConfig));
  }
}
exports.FeatureRegistry = FeatureRegistry;
function applyAutomaticPrivilegeGrants(feature) {
  var _feature$privileges2, _feature$privileges3, _feature$reserved$pri, _feature$reserved;
  const allPrivilege = (_feature$privileges2 = feature.privileges) === null || _feature$privileges2 === void 0 ? void 0 : _feature$privileges2.all;
  const readPrivilege = (_feature$privileges3 = feature.privileges) === null || _feature$privileges3 === void 0 ? void 0 : _feature$privileges3.read;
  const reservedPrivileges = ((_feature$reserved$pri = (_feature$reserved = feature.reserved) === null || _feature$reserved === void 0 ? void 0 : _feature$reserved.privileges) !== null && _feature$reserved$pri !== void 0 ? _feature$reserved$pri : []).map(rp => rp.privilege);
  applyAutomaticAllPrivilegeGrants(allPrivilege, ...reservedPrivileges);
  applyAutomaticReadPrivilegeGrants(readPrivilege);
  return feature;
}
function applyAutomaticAllPrivilegeGrants(...allPrivileges) {
  allPrivileges.forEach(allPrivilege => {
    if (allPrivilege) {
      allPrivilege.savedObject.all = (0, _lodash.uniq)([...allPrivilege.savedObject.all, 'telemetry']);
      allPrivilege.savedObject.read = (0, _lodash.uniq)([...allPrivilege.savedObject.read, 'config', 'config-global', 'url']);
    }
  });
}
function applyAutomaticReadPrivilegeGrants(...readPrivileges) {
  readPrivileges.forEach(readPrivilege => {
    if (readPrivilege) {
      readPrivilege.savedObject.read = (0, _lodash.uniq)([...readPrivilege.savedObject.read, 'config', 'config-global', 'telemetry', 'url']);
    }
  });
}
function collectSubFeaturesPrivileges(feature) {
  var _feature$subFeatures$, _feature$subFeatures2;
  return (_feature$subFeatures$ = (_feature$subFeatures2 = feature.subFeatures) === null || _feature$subFeatures2 === void 0 ? void 0 : _feature$subFeatures2.flatMap(subFeature => subFeature.privilegeGroups.flatMap(({
    privileges
  }) => privileges.map(privilege => [privilege.id, privilege])))) !== null && _feature$subFeatures$ !== void 0 ? _feature$subFeatures$ : [];
}