"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAxesConfiguration = getAxesConfiguration;
exports.getAxisPosition = getAxisPosition;
exports.getOriginalAxisPosition = getOriginalAxisPosition;
exports.groupAxesByType = groupAxesByType;
exports.isFormatterCompatible = isFormatterCompatible;
var _charts = require("@elastic/charts");
var _utils = require("@kbn/visualizations-plugin/common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function isFormatterCompatible(formatter1, formatter2) {
  return (formatter1 === null || formatter1 === void 0 ? void 0 : formatter1.id) === (formatter2 === null || formatter2 === void 0 ? void 0 : formatter2.id);
}
const LEFT_GLOBAL_AXIS_ID = 'left';
const RIGHT_GLOBAL_AXIS_ID = 'right';
function isAxisSeriesAppliedForFormatter(series, currentSeries) {
  return series.every(leftSeries => isFormatterCompatible(leftSeries.fieldFormat, currentSeries.fieldFormat));
}
function groupAxesByType(layers, fieldFormats, yAxisConfigs) {
  const series = {
    auto: [],
    left: [],
    right: []
  };
  const leftSeriesKeys = [];
  const rightSeriesKeys = [];
  layers.forEach(layer => {
    const {
      layerId,
      table
    } = layer;
    layer.accessors.forEach(accessor => {
      const dataDecorations = layer.decorations;
      const yAccessor = (0, _utils.getAccessorByDimension)(accessor, table.columns);
      const decorationByAccessor = dataDecorations === null || dataDecorations === void 0 ? void 0 : dataDecorations.find(decorationConfig => decorationConfig.forAccessor === yAccessor);
      const axisConfigById = yAxisConfigs === null || yAxisConfigs === void 0 ? void 0 : yAxisConfigs.find(axis => (decorationByAccessor === null || decorationByAccessor === void 0 ? void 0 : decorationByAccessor.axisId) && axis.id && axis.id === (decorationByAccessor === null || decorationByAccessor === void 0 ? void 0 : decorationByAccessor.axisId));
      const key = axisConfigById !== null && axisConfigById !== void 0 && axisConfigById.id ? `axis-${axisConfigById === null || axisConfigById === void 0 ? void 0 : axisConfigById.id}` : 'auto';
      const fieldFormat = fieldFormats[layerId].yAccessors[yAccessor];
      if (!series[key]) {
        series[key] = [];
      }
      series[key].push({
        layer: layer.layerId,
        accessor: yAccessor,
        fieldFormat
      });
      if ((axisConfigById === null || axisConfigById === void 0 ? void 0 : axisConfigById.position) === _charts.Position.Left) {
        leftSeriesKeys.push(key);
      } else if ((axisConfigById === null || axisConfigById === void 0 ? void 0 : axisConfigById.position) === _charts.Position.Right) {
        rightSeriesKeys.push(key);
      }
    });
  });
  const tablesExist = layers.filter(({
    table
  }) => Boolean(table)).length > 0;
  if (!leftSeriesKeys.length) {
    leftSeriesKeys.push(LEFT_GLOBAL_AXIS_ID);
  }
  if (!rightSeriesKeys.length) {
    rightSeriesKeys.push(RIGHT_GLOBAL_AXIS_ID);
  }
  series.auto.forEach(currentSeries => {
    const leftAxisGroupId = tablesExist ? leftSeriesKeys.find(leftSeriesKey => isAxisSeriesAppliedForFormatter(series[leftSeriesKey], currentSeries)) : undefined;
    const rightAxisGroupId = tablesExist ? rightSeriesKeys.find(rightSeriesKey => isAxisSeriesAppliedForFormatter(series[rightSeriesKey], currentSeries)) : undefined;
    const rightSeriesCount = rightSeriesKeys.reduce((acc, key) => {
      return acc + series[key].length;
    }, 0);
    const leftSeriesCount = leftSeriesKeys.reduce((acc, key) => {
      return acc + series[key].length;
    }, 0);
    let axisGroupId;
    if (leftSeriesCount === 0 || leftAxisGroupId) {
      axisGroupId = leftAxisGroupId || leftSeriesKeys[0];
    } else if (rightSeriesCount === 0 || rightAxisGroupId) {
      axisGroupId = rightAxisGroupId || rightSeriesKeys[0];
    } else if (rightSeriesCount >= leftSeriesCount) {
      axisGroupId = leftSeriesKeys[0];
    } else {
      axisGroupId = rightSeriesKeys[0];
    }
    series[axisGroupId].push(currentSeries);
  });
  return series;
}
function getAxisPosition(position, shouldRotate) {
  if (shouldRotate) {
    switch (position) {
      case _charts.Position.Bottom:
        {
          return _charts.Position.Right;
        }
      case _charts.Position.Right:
        {
          return _charts.Position.Top;
        }
      case _charts.Position.Top:
        {
          return _charts.Position.Left;
        }
      case _charts.Position.Left:
        {
          return _charts.Position.Bottom;
        }
    }
  }
  return position;
}
function getOriginalAxisPosition(position, shouldRotate) {
  if (shouldRotate) {
    switch (position) {
      case _charts.Position.Bottom:
        {
          return _charts.Position.Left;
        }
      case _charts.Position.Right:
        {
          return _charts.Position.Bottom;
        }
      case _charts.Position.Top:
        {
          return _charts.Position.Right;
        }
      case _charts.Position.Left:
        {
          return _charts.Position.Top;
        }
    }
  }
  return position;
}
function axisGlobalConfig(position, yAxisConfigs) {
  return (yAxisConfigs === null || yAxisConfigs === void 0 ? void 0 : yAxisConfigs.find(axis => !axis.id && axis.position === position)) || {};
}
const getXAxisConfig = (axisConfigs = []) => axisConfigs.find(({
  type
}) => type === 'xAxisConfig');
function getAxesConfiguration(layers, shouldRotate, formatFactory, fieldFormats, axisConfigs) {
  const series = groupAxesByType(layers, fieldFormats, axisConfigs);
  const axisGroups = [];
  let position;
  axisConfigs === null || axisConfigs === void 0 ? void 0 : axisConfigs.forEach(axis => {
    const groupId = axis.id ? `axis-${axis.id}` : undefined;
    if (groupId && series[groupId] && series[groupId].length > 0) {
      position = getAxisPosition(axis.position || _charts.Position.Left, shouldRotate);
      axisGroups.push({
        groupId,
        formatter: formatFactory === null || formatFactory === void 0 ? void 0 : formatFactory(series[groupId][0].fieldFormat),
        series: series[groupId].map(({
          fieldFormat,
          ...currentSeries
        }) => currentSeries),
        ...axisGlobalConfig(axis.position || _charts.Position.Left, axisConfigs),
        ...axis,
        position
      });
    }
  });
  if (series[LEFT_GLOBAL_AXIS_ID].length > 0) {
    position = shouldRotate ? _charts.Position.Bottom : _charts.Position.Left;
    axisGroups.push({
      groupId: LEFT_GLOBAL_AXIS_ID,
      formatter: formatFactory === null || formatFactory === void 0 ? void 0 : formatFactory(series.left[0].fieldFormat),
      series: series.left.map(({
        fieldFormat,
        ...currentSeries
      }) => currentSeries),
      ...axisGlobalConfig(_charts.Position.Left, axisConfigs),
      position
    });
  }
  if (series[RIGHT_GLOBAL_AXIS_ID].length > 0) {
    position = shouldRotate ? _charts.Position.Top : _charts.Position.Right;
    axisGroups.push({
      groupId: RIGHT_GLOBAL_AXIS_ID,
      formatter: formatFactory === null || formatFactory === void 0 ? void 0 : formatFactory(series.right[0].fieldFormat),
      series: series.right.map(({
        fieldFormat,
        ...currentSeries
      }) => currentSeries),
      ...axisGlobalConfig(_charts.Position.Right, axisConfigs),
      position
    });
  }
  const xAxisConfig = getXAxisConfig(axisConfigs);
  if (xAxisConfig) {
    position = getAxisPosition(xAxisConfig.position || _charts.Position.Bottom, shouldRotate);
    axisGroups.push({
      groupId: 'bottom',
      series: [],
      ...xAxisConfig,
      position
    });
  }
  return axisGroups;
}