"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getColor = exports.byDataColorPaletteMap = void 0;
var _lodash = require("lodash");
var _public = require("@kbn/charts-plugin/public");
var _types = require("../../../common/types");
var _get_node_labels = require("./get_node_labels");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const isTreemapOrMosaicChart = shape => [_types.ChartTypes.MOSAIC, _types.ChartTypes.TREEMAP].includes(shape);
const byDataColorPaletteMap = (rows, column, paletteDefinition, {
  params
}, formatters, formatter) => {
  const colorMap = new Map(rows.map(item => {
    var _column$id;
    const formattedName = (0, _get_node_labels.getNodeLabel)(item[(_column$id = column.id) !== null && _column$id !== void 0 ? _column$id : ''], column, formatters, formatter.deserialize);
    return [formattedName, undefined];
  }));
  let rankAtDepth = 0;
  return {
    getColor: item => {
      const key = String(item);
      if (!colorMap.has(key)) return;
      let color = colorMap.get(key);
      if (color) {
        return color;
      }
      color = paletteDefinition.getCategoricalColor([{
        name: key,
        totalSeriesAtDepth: colorMap.size,
        rankAtDepth: rankAtDepth++
      }], {
        behindText: false
      }, params) || undefined;
      colorMap.set(key, color);
      return color;
    }
  };
};
exports.byDataColorPaletteMap = byDataColorPaletteMap;
const getDistinctColor = (categoricalKey, isSplitChart, overwriteColors = {}, visParams, palettes, syncColors, {
  parentSeries,
  allSeries
}, formattedCategoricalKey) => {
  var _visParams$palette$pa, _visParams$palette;
  // TODO move away from Record to a Map to avoid issues with reserved JS keywords
  if (overwriteColors.hasOwnProperty(categoricalKey)) {
    return overwriteColors[categoricalKey];
  }
  // this is for supporting old visualizations (created by vislib plugin)
  // it seems that there for some aggs, the uiState saved from vislib is
  // different from how es-charts handles it
  if (overwriteColors.hasOwnProperty(formattedCategoricalKey)) {
    return overwriteColors[formattedCategoricalKey];
  }
  const index = allSeries.findIndex(d => (0, _lodash.isEqual)(d, categoricalKey));
  const isSplitParentLayer = isSplitChart && parentSeries.includes(categoricalKey);
  return palettes === null || palettes === void 0 ? void 0 : palettes.get(visParams.palette.name).getCategoricalColor([{
    name: categoricalKey,
    rankAtDepth: isSplitParentLayer ? parentSeries.findIndex(d => d === categoricalKey) : index > -1 ? index : 0,
    totalSeriesAtDepth: isSplitParentLayer ? parentSeries.length : allSeries.length || 1
  }], {
    maxDepth: 1,
    totalSeries: allSeries.length || 1,
    behindText: visParams.labels.show,
    syncColors
  }, (_visParams$palette$pa = (_visParams$palette = visParams.palette) === null || _visParams$palette === void 0 ? void 0 : _visParams$palette.params) !== null && _visParams$palette$pa !== void 0 ? _visParams$palette$pa : {
    colors: []
  });
};

/**
 * This interface is introduced to simplify the used logic on testing an ArrayNode outside elastic-charts.
 * The SimplifiedArrayNode structure resembles the hierarchical configuration of an ArrayNode,
 * by presenting only the necessary fields used by the functions in this file.
 * The main difference is in the parent node, that to simplify this infinite tree structure we configured it as optional
 * so that in test I don't need to add a type assertion on an undefined parent as in elastic-charts.
 * The children are slight different in implementation and they accept `unknown` as key
 * due to the situation described in https://github.com/elastic/kibana/issues/153437
 */

/**
 * This method returns the path of each hierarchical layer encountered from the given node
 * (a node of a hierarchical tree, currently a partition tree) up to the root of the hierarchy tree.
 * The resulting array only shows, for each parent, the name of the node, its child index within the parent branch
 * (called rankInDepth) and the total number of children of the parent.
 *
 */
const createSeriesLayers = (arrayNode, parentSeries, isSplitChart, formatters, formatter, column) => {
  const seriesLayers = [];
  let tempParent = arrayNode;
  while (tempParent.parent && tempParent.depth > 0) {
    const nodeKey = tempParent.parent.children[tempParent.sortIndex][0];
    const seriesName = String(nodeKey);
    /**
     * FIXME this is a bad implementation: The `parentSeries` is an array of both `string` and `RangeKey` even if its type
     * is marked as `string[]` in `DistinctSeries`. Here instead we are checking if a stringified `RangeKey` is included into this array that
     * is conceptually wrong.
     * see https://github.com/elastic/kibana/issues/153437
     */
    const isSplitParentLayer = isSplitChart && parentSeries.includes(seriesName);
    const formattedName = (0, _get_node_labels.getNodeLabel)(nodeKey, column, formatters, formatter.deserialize);
    seriesLayers.unshift({
      // by construction and types `formattedName` should be always be a string, but I leave this Nullish Coalescing
      // because I don't trust much our formatting functions
      name: formattedName !== null && formattedName !== void 0 ? formattedName : seriesName,
      rankAtDepth: isSplitParentLayer ?
      // FIXME as described above this will not work correctly if the `nodeKey` is a `RangeKey`
      parentSeries.findIndex(name => name === seriesName) : tempParent.sortIndex,
      totalSeriesAtDepth: isSplitParentLayer ? parentSeries.length : tempParent.parent.children.length
    });
    tempParent = tempParent.parent;
  }
  return seriesLayers;
};
const overrideColors = (seriesLayers, overwriteColors, name) => {
  let overwriteColor;
  if (overwriteColors.hasOwnProperty(name)) {
    overwriteColor = overwriteColors[name];
  }
  seriesLayers.forEach(layer => {
    if (Object.keys(overwriteColors).includes(layer.name)) {
      overwriteColor = overwriteColors[layer.name];
    }
  });
  return overwriteColor;
};
const getColor = (chartType, categoricalKey, arrayNode, layerIndex, isSplitChart, overwriteColors = {}, distinctSeries, {
  columnsLength,
  rowsLength
}, visParams, palettes, byDataPalette, syncColors, isDarkMode, formatter, column, formatters) => {
  var _formatter$deserializ, _visParams$palette$pa2, _visParams$palette2;
  // Mind the difference here: the contrast computation for the text ignores the alpha/opacity
  // therefore change it for dark mode
  const defaultColor = isDarkMode ? 'rgba(0,0,0,0)' : 'rgba(255,255,255,0)';
  const name = column.format ? (_formatter$deserializ = formatter.deserialize(column.format).convert(categoricalKey)) !== null && _formatter$deserializ !== void 0 ? _formatter$deserializ : '' : '';
  if (visParams.distinctColors) {
    return getDistinctColor(categoricalKey, isSplitChart, overwriteColors, visParams, palettes, syncColors, distinctSeries, name) || defaultColor;
  }
  const seriesLayers = createSeriesLayers(arrayNode, distinctSeries.parentSeries, isSplitChart, formatters, formatter, column);
  const overriddenColor = overrideColors(seriesLayers, overwriteColors, name);
  if (overriddenColor) {
    // this is necessary for supporting some old visualizations that defined their own colors (created by vislib plugin)
    return (0, _public.lightenColor)(overriddenColor, seriesLayers.length, columnsLength);
  }
  if (chartType === _types.ChartTypes.MOSAIC && byDataPalette && seriesLayers[1]) {
    return byDataPalette.getColor(seriesLayers[1].name) || defaultColor;
  }
  if (isTreemapOrMosaicChart(chartType)) {
    if (layerIndex < columnsLength - 1) {
      return defaultColor;
    }
    // for treemap use the top layer for coloring, for mosaic use the second layer
    if (seriesLayers.length > 1) {
      if (chartType === _types.ChartTypes.MOSAIC) {
        seriesLayers.shift();
      } else {
        seriesLayers.pop();
      }
    }
  }
  const outputColor = palettes === null || palettes === void 0 ? void 0 : palettes.get(visParams.palette.name).getCategoricalColor(seriesLayers, {
    behindText: visParams.labels.show || isTreemapOrMosaicChart(chartType),
    maxDepth: columnsLength,
    totalSeries: rowsLength,
    syncColors
  }, (_visParams$palette$pa2 = (_visParams$palette2 = visParams.palette) === null || _visParams$palette2 === void 0 ? void 0 : _visParams$palette2.params) !== null && _visParams$palette$pa2 !== void 0 ? _visParams$palette$pa2 : {
    colors: []
  });
  return outputColor || defaultColor;
};
exports.getColor = getColor;