"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSeriesValueColumnIndex = exports.getMultiFilterCells = exports.getFilterPopoverTitle = exports.getFilterEventData = exports.getFilterClickData = exports.getAccessor = exports.canFilter = void 0;
var _utils = require("@kbn/visualizations-plugin/common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const canFilter = async (event, actions) => {
  if (!event) {
    return false;
  }
  const filters = await actions.createFiltersFromValueClickAction(event.data);
  return Boolean(filters.length);
};
exports.canFilter = canFilter;
const getMultiFilterCells = (tooltipSelectedValues, bucketColumns, visData) => {
  const row = visData.rows.findIndex(r => tooltipSelectedValues.every(({
    valueAccessor,
    seriesIdentifier
  }) => {
    if (typeof valueAccessor !== 'number' || valueAccessor < 1) return;
    const index = valueAccessor - 1;
    const bucketColumnId = bucketColumns[index].id;
    if (!bucketColumnId) return;
    return r[bucketColumnId] === seriesIdentifier.key;
  }));
  return tooltipSelectedValues.map(({
    valueAccessor
  }) => {
    if (typeof valueAccessor !== 'number' || valueAccessor < 1) return;
    const index = valueAccessor - 1;
    const bucketColumnId = bucketColumns[index].id;
    if (!bucketColumnId) return;
    const column = visData.columns.findIndex(c => c.id === bucketColumnId);
    if (column === -1) {
      return;
    }
    return {
      column,
      row
    };
  }).filter(nonNullable);
};
exports.getMultiFilterCells = getMultiFilterCells;
function nonNullable(v) {
  return v != null;
}
const getFilterClickData = (clickedLayers, bucketColumns, metricColId, visData, originalVisData, numOriginalMetrics, splitChartDimension, splitChartFormatter) => {
  const data = [];
  const rowIndex = visData.rows.findIndex(row => clickedLayers.every((layer, index) => {
    const columnId = bucketColumns[index].id;
    if (!columnId && !splitChartDimension) return;
    // if there is no column id it means there is no actual bucket column, just the metric column and potentially a split chart column
    const isCurrentLayer = !columnId || row[columnId] === layer.groupByRollup;
    if (!splitChartDimension) {
      return isCurrentLayer;
    }
    const value = (splitChartFormatter === null || splitChartFormatter === void 0 ? void 0 : splitChartFormatter.convert(row[splitChartDimension.id])) || row[splitChartDimension.id];
    return isCurrentLayer && value === layer.smAccessorValue;
  }));
  const originalRowIndex = Math.floor(rowIndex / numOriginalMetrics);
  data.push(...clickedLayers.map((clickedLayer, index) => {
    var _currentColumn$meta, _currentColumn$meta$s;
    const currentColumnIndex = visData.columns.findIndex(col => col.id === bucketColumns[index].id);
    if (currentColumnIndex === -1) {
      return undefined;
    }
    const currentColumn = visData.columns[currentColumnIndex];

    // this logic maps the indices of the elements in the
    // visualization's table to the indices in the table before
    // any multiple metrics were collapsed into one metric column
    const originalColumnIndex = (_currentColumn$meta = currentColumn.meta) !== null && _currentColumn$meta !== void 0 && (_currentColumn$meta$s = _currentColumn$meta.sourceParams) !== null && _currentColumn$meta$s !== void 0 && _currentColumn$meta$s.consolidatedMetricsColumn ? currentColumnIndex + rowIndex % numOriginalMetrics : currentColumnIndex;
    return {
      column: originalColumnIndex,
      row: originalRowIndex,
      value: clickedLayer.groupByRollup,
      table: originalVisData
    };
  }).filter(Boolean));

  // Allows filtering with the small multiples value
  if (splitChartDimension) {
    if (!bucketColumns[0].id) {
      var _visData$columns$find;
      // this is a split chart without any real bucket columns, so filter by the metric column
      data.push({
        column: visData.columns.findIndex(col => col.id === metricColId),
        row: rowIndex,
        table: visData,
        value: (_visData$columns$find = visData.columns.find(col => col.id === metricColId)) === null || _visData$columns$find === void 0 ? void 0 : _visData$columns$find.name
      });
    }
    data.push({
      column: visData.columns.findIndex(col => col.id === splitChartDimension.id),
      row: rowIndex,
      table: visData,
      value: clickedLayers[0].smAccessorValue
    });
  }
  return data;
};
exports.getFilterClickData = getFilterClickData;
const getFilterEventData = (visData, series) => {
  return visData.columns.reduce((acc, {
    id
  }, column) => {
    const value = series.key;
    const row = visData.rows.findIndex(r => r[id] === value);
    if (row > -1) {
      acc.push({
        table: visData,
        column,
        row,
        value
      });
    }
    return acc;
  }, []);
};
exports.getFilterEventData = getFilterEventData;
const getSeriesValueColumnIndex = (value, visData) => {
  return visData.columns.findIndex(({
    id
  }) => !!visData.rows.find(r => r[id] === value));
};
exports.getSeriesValueColumnIndex = getSeriesValueColumnIndex;
const getAccessor = (buckets, index) => {
  const accessorForDimensionBuckets = buckets === null || buckets === void 0 ? void 0 : buckets.find(b => {
    return typeof b !== 'string' && b.accessor === index;
  });
  return accessorForDimensionBuckets || (buckets === null || buckets === void 0 ? void 0 : buckets[index]);
};
exports.getAccessor = getAccessor;
const getFilterPopoverTitle = (visParams, visData, columnIndex, formatter, seriesKey) => {
  let formattedTitle = '';
  if (visParams.dimensions.buckets) {
    const accessor = getAccessor(visParams.dimensions.buckets, columnIndex);
    formattedTitle = accessor ? formatter((0, _utils.getFormatByAccessor)(accessor, visData.columns)).convert(seriesKey) : '';
  }
  return formattedTitle || seriesKey;
};
exports.getFilterPopoverTitle = getFilterPopoverTitle;