"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ignoreErrorsMap = void 0;
exports.validateQuery = validateQuery;
exports.validateSources = validateSources;
var _uniqBy = _interopRequireDefault(require("lodash/uniqBy"));
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("../shared/helpers");
var _variables = require("../shared/variables");
var _errors = require("./errors");
var _resources = require("./resources");
var _helpers2 = require("./helpers");
var _helper = require("../autocomplete/helper");
var _constants = require("../shared/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function validateFunctionLiteralArg(astFunction, actualArg, argDef, references, parentCommand) {
  const messages = [];
  if ((0, _helpers.isLiteralItem)(actualArg)) {
    if (actualArg.literalType === 'string' && argDef.literalOptions && (0, _helpers.isValidLiteralOption)(actualArg, argDef)) {
      var _argDef$literalOption;
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedLiteralOption',
        values: {
          name: astFunction.name,
          value: actualArg.value,
          supportedOptions: (_argDef$literalOption = argDef.literalOptions) === null || _argDef$literalOption === void 0 ? void 0 : _argDef$literalOption.map(option => `"${option}"`).join(', ')
        },
        locations: actualArg.location
      }));
    }
    if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, argDef, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: argDef.type,
          value: typeof actualArg.value === 'number' ? actualArg.value : String(actualArg.value),
          givenType: actualArg.literalType
        },
        locations: actualArg.location
      }));
    }
  }
  if ((0, _helpers.isTimeIntervalItem)(actualArg)) {
    // check first if it's a valid interval string
    if (!(0, _helpers.inKnownTimeInterval)(actualArg)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownInterval',
        values: {
          value: actualArg.unit
        },
        locations: actualArg.location
      }));
    } else {
      if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, argDef, references, parentCommand)) {
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'wrongArgumentType',
          values: {
            name: astFunction.name,
            argType: argDef.type,
            value: actualArg.name,
            givenType: 'duration'
          },
          locations: actualArg.location
        }));
      }
    }
  }
  return messages;
}
function validateInlineCastArg(astFunction, arg, parameterDefinition, references, parentCommand) {
  if (!(0, _helpers.isInlineCastItem)(arg)) {
    return [];
  }
  if (!(0, _helpers.checkFunctionArgMatchesDefinition)(arg, parameterDefinition, references, parentCommand)) {
    return [(0, _errors.getMessageFromId)({
      messageId: 'wrongArgumentType',
      values: {
        name: astFunction.name,
        argType: parameterDefinition.type,
        value: arg.text,
        givenType: arg.castType
      },
      locations: arg.location
    })];
  }
  return [];
}
function validateNestedFunctionArg(astFunction, actualArg, parameterDefinition, references, parentCommand) {
  const messages = [];
  if ((0, _helpers.isFunctionItem)(actualArg) &&
  // no need to check the reason here, it is checked already above
  (0, _helpers.isSupportedFunction)(actualArg.name, parentCommand).supported) {
    // The isSupported check ensure the definition exists
    const argFn = (0, _helpers.getFunctionDefinition)(actualArg.name);
    const fnDef = (0, _helpers.getFunctionDefinition)(astFunction.name);
    // no nestying criteria should be enforced only for same type function
    if ('noNestingFunctions' in parameterDefinition && parameterDefinition.noNestingFunctions && fnDef.type === argFn.type) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noNestedArgumentSupport',
        values: {
          name: actualArg.text,
          argType: argFn.signatures[0].returnType
        },
        locations: actualArg.location
      }));
    }
    if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, parameterDefinition, references, parentCommand)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentType',
        values: {
          name: astFunction.name,
          argType: parameterDefinition.type,
          value: actualArg.text,
          givenType: argFn.signatures[0].returnType
        },
        locations: actualArg.location
      }));
    }
  }
  return messages;
}
function validateFunctionColumnArg(astFunction, actualArg, parameterDefinition, references, parentCommand) {
  const messages = [];
  if (!(0, _helpers.isColumnItem)(actualArg)) {
    return messages;
  }
  const columnName = (0, _helpers.getQuotedColumnName)(actualArg);
  const columnExists = (0, _helpers.getColumnExists)(actualArg, references);
  if (parameterDefinition.constantOnly) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'expectedConstant',
      values: {
        fn: astFunction.name,
        given: columnName
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (!columnExists) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownColumn',
      values: {
        name: actualArg.name
      },
      locations: actualArg.location
    }));
    return messages;
  }
  if (actualArg.name === '*') {
    // if function does not support wildcards return a specific error
    if (!('supportsWildcard' in parameterDefinition) || !parameterDefinition.supportsWildcard) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'noWildcardSupportAsArg',
        values: {
          name: astFunction.name
        },
        locations: actualArg.location
      }));
    }
    return messages;
  }
  if (!(0, _helpers.checkFunctionArgMatchesDefinition)(actualArg, parameterDefinition, references, parentCommand)) {
    const columnHit = (0, _helpers.lookupColumn)(actualArg, references);
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'wrongArgumentType',
      values: {
        name: astFunction.name,
        argType: parameterDefinition.type,
        value: actualArg.name,
        givenType: columnHit.type
      },
      locations: actualArg.location
    }));
  }
  return messages;
}
function extractCompatibleSignaturesForFunction(fnDef, astFunction) {
  return fnDef.signatures.filter(def => {
    if (def.minParams) {
      return astFunction.args.length >= def.minParams;
    }
    return astFunction.args.length >= def.params.filter(({
      optional
    }) => !optional).length && astFunction.args.length <= def.params.length;
  });
}
function removeInlineCasts(arg) {
  if ((0, _helpers.isInlineCastItem)(arg)) {
    return removeInlineCasts(arg.value);
  }
  return arg;
}
function validateFunction(astFunction, parentCommand, parentOption, references, forceConstantOnly = false, isNested) {
  const messages = [];
  if (astFunction.incomplete) {
    return messages;
  }
  const fnDefinition = (0, _helpers.getFunctionDefinition)(astFunction.name);
  const isFnSupported = (0, _helpers.isSupportedFunction)(astFunction.name, parentCommand, parentOption);
  if (!isFnSupported.supported) {
    if (isFnSupported.reason === 'unknownFunction') {
      messages.push(_errors.errors.unknownFunction(astFunction));
    }
    // for nested functions skip this check and make the nested check fail later on
    if (isFnSupported.reason === 'unsupportedFunction' && !isNested) {
      messages.push(parentOption ? (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommandOption',
        values: {
          name: astFunction.name,
          command: parentCommand.toUpperCase(),
          option: parentOption.toUpperCase()
        },
        locations: astFunction.location
      }) : (0, _errors.getMessageFromId)({
        messageId: 'unsupportedFunctionForCommand',
        values: {
          name: astFunction.name,
          command: parentCommand.toUpperCase()
        },
        locations: astFunction.location
      }));
    }
    if (messages.length) {
      return messages;
    }
  }
  const matchingSignatures = extractCompatibleSignaturesForFunction(fnDefinition, astFunction);
  if (!matchingSignatures.length) {
    const {
      max,
      min
    } = (0, _helpers2.getMaxMinNumberOfParams)(fnDefinition);
    if (max === min) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumber',
        values: {
          fn: astFunction.name,
          numArgs: max,
          passedArgs: astFunction.args.length
        },
        locations: astFunction.location
      }));
    } else if (astFunction.args.length > max) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooMany',
        values: {
          fn: astFunction.name,
          numArgs: max,
          passedArgs: astFunction.args.length,
          extraArgs: astFunction.args.length - max
        },
        locations: astFunction.location
      }));
    } else {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'wrongArgumentNumberTooFew',
        values: {
          fn: astFunction.name,
          numArgs: min,
          passedArgs: astFunction.args.length,
          missingArgs: min - astFunction.args.length
        },
        locations: astFunction.location
      }));
    }
  }
  // now perform the same check on all functions args
  for (let i = 0; i < astFunction.args.length; i++) {
    const arg = astFunction.args[i];
    const allMatchingArgDefinitionsAreConstantOnly = matchingSignatures.every(signature => {
      var _signature$params$i;
      return (_signature$params$i = signature.params[i]) === null || _signature$params$i === void 0 ? void 0 : _signature$params$i.constantOnly;
    });
    const wrappedArray = Array.isArray(arg) ? arg : [arg];
    for (const _subArg of wrappedArray) {
      /**
       * we need to remove the inline casts
       * to see if there's a function under there
       *
       * e.g. for ABS(CEIL(numberField)::int), we need to validate CEIL(numberField)
       */
      const subArg = removeInlineCasts(_subArg);
      if ((0, _helpers.isFunctionItem)(subArg)) {
        const messagesFromArg = validateFunction(subArg, parentCommand, parentOption, references,
        /**
         * The constantOnly constraint needs to be enforced for arguments that
         * are functions as well, regardless of whether the definition for the
         * sub function's arguments includes the constantOnly flag.
         *
         * Example:
         * bucket(@timestamp, abs(bytes), "", "")
         *
         * In the above example, the abs function is not defined with the
         * constantOnly flag, but the second parameter in bucket _is_ defined
         * with the constantOnly flag.
         *
         * Because of this, the abs function's arguments inherit the constraint
         * and each should be validated as if each were constantOnly.
         */
        allMatchingArgDefinitionsAreConstantOnly || forceConstantOnly,
        // use the nesting flag for now just for stats and metrics
        // TODO: revisit this part later on to make it more generic
        parentCommand === 'stats' || parentCommand === 'metrics' ? isNested || !(0, _helpers.isAssignment)(astFunction) : false);
        if (messagesFromArg.some(({
          code
        }) => code === 'expectedConstant')) {
          const consolidatedMessage = (0, _errors.getMessageFromId)({
            messageId: 'expectedConstant',
            values: {
              fn: astFunction.name,
              given: subArg.text
            },
            locations: subArg.location
          });
          messages.push(consolidatedMessage, ...messagesFromArg.filter(({
            code
          }) => code !== 'expectedConstant'));
        } else {
          messages.push(...messagesFromArg);
        }
      }
    }
  }
  // check if the definition has some specific validation to apply:
  if (fnDefinition.validate) {
    const payloads = fnDefinition.validate(astFunction);
    if (payloads.length) {
      messages.push(...payloads);
    }
  }
  // at this point we're sure that at least one signature is matching
  const failingSignatures = [];
  for (const signature of matchingSignatures) {
    const failingSignature = [];
    astFunction.args.forEach((outerArg, index) => {
      const argDef = (0, _helper.getParamAtPosition)(signature, index);
      if (!outerArg && argDef !== null && argDef !== void 0 && argDef.optional || !argDef) {
        // that's ok, just skip it
        // the else case is already catched with the argument counts check
        // few lines above
        return;
      }

      // check every element of the argument (may be an array of elements, or may be a single element)
      const hasMultipleElements = Array.isArray(outerArg);
      const argElements = hasMultipleElements ? outerArg : [outerArg];
      const singularType = (0, _helpers.extractSingularType)(argDef.type);
      const messagesFromAllArgElements = argElements.flatMap(arg => {
        return [validateFunctionLiteralArg, validateNestedFunctionArg, validateFunctionColumnArg, validateInlineCastArg].flatMap(validateFn => {
          return validateFn(astFunction, arg, {
            ...argDef,
            type: singularType,
            constantOnly: forceConstantOnly || argDef.constantOnly
          }, references, parentCommand);
        });
      });
      const shouldCollapseMessages = (0, _helpers.isArrayType)(argDef.type) && hasMultipleElements;
      failingSignature.push(...(shouldCollapseMessages ? (0, _helpers2.collapseWrongArgumentTypeMessages)(messagesFromAllArgElements, outerArg, astFunction.name, argDef.type, parentCommand, references) : messagesFromAllArgElements));
    });
    if (failingSignature.length) {
      failingSignatures.push(failingSignature);
    }
  }
  if (failingSignatures.length && failingSignatures.length === matchingSignatures.length) {
    const failingSignatureOrderedByErrorCount = failingSignatures.map((arr, index) => ({
      index,
      count: arr.length
    })).sort((a, b) => a.count - b.count);
    const indexForShortestFailingsignature = failingSignatureOrderedByErrorCount[0].index;
    messages.push(...failingSignatures[indexForShortestFailingsignature]);
  }
  // This is due to a special case in enrich where an implicit assignment is possible
  // so the AST needs to store an explicit "columnX = columnX" which duplicates the message
  return (0, _uniqBy.default)(messages, ({
    location
  }) => `${location.min}-${location.max}`);
}
function validateSetting(setting, settingDef, command, referenceMaps) {
  const messages = [];
  if (setting.incomplete || command.incomplete) {
    return messages;
  }
  if (!settingDef) {
    const commandDef = (0, _helpers.getCommandDefinition)(command.name);
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unsupportedSetting',
      values: {
        setting: setting.name,
        expected: commandDef.modes.map(({
          name
        }) => name).join(', ')
      },
      locations: setting.location
    }));
    return messages;
  }
  if (settingDef.values.every(({
    name
  }) => name !== setting.name) ||
  // enforce the check on the prefix if present
  settingDef.prefix && !setting.text.startsWith(settingDef.prefix)) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unsupportedSettingCommandValue',
      values: {
        command: command.name.toUpperCase(),
        value: setting.text,
        // for some reason all this enums are uppercase in ES
        expected: settingDef.values.map(({
          name
        }) => `${settingDef.prefix || ''}${name}`).join(', ').toUpperCase()
      },
      locations: setting.location
    }));
  }
  return messages;
}

/**
 * Validate that a function is an aggregate function or that all children
 * recursively terminate at either a literal or an aggregate function.
 */
const isFunctionAggClosed = fn => (0, _helpers.isAggFunction)(fn) || areFunctionArgsAggClosed(fn);
const areFunctionArgsAggClosed = fn => fn.args.every(arg => (0, _helpers.isLiteralItem)(arg) || (0, _helpers.isFunctionItem)(arg) && isFunctionAggClosed(arg));

/**
 * Looks for first nested aggregate function in an aggregate function, recursively.
 */
const findNestedAggFunctionInAggFunction = agg => {
  for (const arg of agg.args) {
    if ((0, _helpers.isFunctionItem)(arg)) {
      return (0, _helpers.isAggFunction)(arg) ? arg : findNestedAggFunctionInAggFunction(arg);
    }
  }
};

/**
 * Looks for first nested aggregate function in another aggregate a function,
 * recursively.
 *
 * @param fn Function to check for nested aggregate functions.
 * @param parentIsAgg Whether the parent function of `fn` is an aggregate function.
 * @returns The first nested aggregate function in `fn`, or `undefined` if none is found.
 */
const findNestedAggFunction = (fn, parentIsAgg = false) => {
  if ((0, _helpers.isAggFunction)(fn)) {
    return parentIsAgg ? fn : findNestedAggFunctionInAggFunction(fn);
  }
  for (const arg of fn.args) {
    if ((0, _helpers.isFunctionItem)(arg)) {
      const nestedAgg = findNestedAggFunction(arg, parentIsAgg || (0, _helpers.isAggFunction)(fn));
      if (nestedAgg) return nestedAgg;
    }
  }
};

/**
 * Validates aggregates fields: `... <aggregates> ...`.
 */
const validateAggregates = (command, aggregates, references) => {
  const messages = [];

  // Should never happen.
  if (!aggregates.length) {
    messages.push(_errors.errors.unexpected(command.location));
    return messages;
  }
  let hasMissingAggregationFunctionError = false;
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      messages.push(...validateFunction(aggregate, command.name, undefined, references));
      let hasAggregationFunction = false;
      (0, _esqlAst.walk)(aggregate, {
        visitFunction: fn => {
          const definition = (0, _helpers.getFunctionDefinition)(fn.name);
          if (!definition) return;
          if (definition.type === 'agg') hasAggregationFunction = true;
        }
      });
      if (!hasAggregationFunction) {
        hasMissingAggregationFunctionError = true;
        messages.push(_errors.errors.noAggFunction(command, aggregate));
      }
    } else if ((0, _helpers.isColumnItem)(aggregate)) {
      messages.push(_errors.errors.unknownAggFunction(aggregate));
    } else {
      // Should never happen.
    }
  }
  if (hasMissingAggregationFunctionError) {
    return messages;
  }
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      const fn = (0, _helpers.isAssignment)(aggregate) ? aggregate.args[1] : aggregate;
      if ((0, _helpers.isFunctionItem)(fn) && !isFunctionAggClosed(fn)) {
        messages.push(_errors.errors.expressionNotAggClosed(command, fn));
      }
    }
  }
  if (messages.length) {
    return messages;
  }
  for (const aggregate of aggregates) {
    if ((0, _helpers.isFunctionItem)(aggregate)) {
      const aggInAggFunction = findNestedAggFunction(aggregate);
      if (aggInAggFunction) {
        messages.push(_errors.errors.aggInAggFunction(aggInAggFunction));
        break;
      }
    }
  }
  return messages;
};

/**
 * Validates grouping fields of the BY clause: `... BY <grouping>`.
 */
const validateByGrouping = (fields, commandName, referenceMaps, multipleParams) => {
  const messages = [];
  for (const field of fields) {
    if (!Array.isArray(field)) {
      if (!multipleParams) {
        if ((0, _helpers.isColumnItem)(field)) {
          messages.push(...validateColumnForCommand(field, commandName, referenceMaps));
        }
      } else {
        if ((0, _helpers.isColumnItem)(field)) {
          messages.push(...validateColumnForCommand(field, commandName, referenceMaps));
        }
        if ((0, _helpers.isFunctionItem)(field)) {
          messages.push(...validateFunction(field, commandName, 'by', referenceMaps));
        }
      }
    }
  }
  return messages;
};
function validateOption(option, optionDef, command, referenceMaps) {
  // check if the arguments of the option are of the correct type
  const messages = [];
  if (option.incomplete || command.incomplete) {
    return messages;
  }
  if (!optionDef) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'unknownOption',
      values: {
        command: command.name.toUpperCase(),
        option: option.name
      },
      locations: option.location
    }));
    return messages;
  }
  // use dedicate validate fn if provided
  if (optionDef.validate) {
    const fields = _constants.METADATA_FIELDS;
    messages.push(...optionDef.validate(option, command, new Set(fields)));
  }
  if (!optionDef.skipCommonValidation) {
    option.args.forEach(arg => {
      if (!Array.isArray(arg)) {
        if (!optionDef.signature.multipleParams) {
          if ((0, _helpers.isColumnItem)(arg)) {
            messages.push(...validateColumnForCommand(arg, command.name, referenceMaps));
          }
        } else {
          if ((0, _helpers.isColumnItem)(arg)) {
            messages.push(...validateColumnForCommand(arg, command.name, referenceMaps));
          }
          if ((0, _helpers.isFunctionItem)(arg)) {
            messages.push(...validateFunction(arg, command.name, option.name, referenceMaps));
          }
        }
      }
    });
  }
  return messages;
}
function validateSource(source, commandName, {
  sources,
  policies
}) {
  const messages = [];
  if (source.incomplete) {
    return messages;
  }
  const hasCCS = (0, _helpers.hasCCSSource)(source.name);
  if (hasCCS) {
    return messages;
  }
  const commandDef = (0, _helpers.getCommandDefinition)(commandName);
  const isWildcardAndNotSupported = (0, _helpers.hasWildcard)(source.name) && !commandDef.signature.params.some(({
    wildcards
  }) => wildcards);
  if (isWildcardAndNotSupported) {
    messages.push((0, _errors.getMessageFromId)({
      messageId: 'wildcardNotSupportedForCommand',
      values: {
        command: commandName.toUpperCase(),
        value: source.name
      },
      locations: source.location
    }));
  } else {
    if (source.sourceType === 'index' && !(0, _helpers.sourceExists)(source.name, sources)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownIndex',
        values: {
          name: source.name
        },
        locations: source.location
      }));
    } else if (source.sourceType === 'policy' && !policies.has(source.name)) {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unknownPolicy',
        values: {
          name: source.name
        },
        locations: source.location
      }));
    }
  }
  return messages;
}
function validateColumnForCommand(column, commandName, references) {
  const messages = [];
  if (commandName === 'row') {
    if (!references.variables.has(column.name)) {
      messages.push(_errors.errors.unknownColumn(column));
    }
  } else {
    const columnName = (0, _helpers.getQuotedColumnName)(column);
    if ((0, _helpers.getColumnExists)(column, references)) {
      const commandDef = (0, _helpers.getCommandDefinition)(commandName);
      const columnParamsWithInnerTypes = commandDef.signature.params.filter(({
        type,
        innerType
      }) => type === 'column' && innerType);
      // this should be guaranteed by the columnCheck above
      const columnRef = (0, _helpers.lookupColumn)(column, references);
      if (columnParamsWithInnerTypes.length) {
        const hasSomeWrongInnerTypes = columnParamsWithInnerTypes.every(({
          innerType
        }) => {
          return innerType !== 'any' && innerType !== columnRef.type;
        });
        if (hasSomeWrongInnerTypes) {
          const supportedTypes = columnParamsWithInnerTypes.map(({
            innerType
          }) => innerType);
          messages.push((0, _errors.getMessageFromId)({
            messageId: 'unsupportedColumnTypeForCommand',
            values: {
              command: commandName.toUpperCase(),
              type: supportedTypes.join(', '),
              typeCount: supportedTypes.length,
              givenType: columnRef.type,
              column: columnName
            },
            locations: column.location
          }));
        }
      }
      if ((0, _helpers.hasWildcard)(columnName) && !(0, _helpers.isVariable)(columnRef) && !commandDef.signature.params.some(({
        type,
        wildcards
      }) => type === 'column' && wildcards)) {
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'wildcardNotSupportedForCommand',
          values: {
            command: commandName.toUpperCase(),
            value: columnName
          },
          locations: column.location
        }));
      }
    } else {
      if (column.name) {
        messages.push(_errors.errors.unknownColumn(column));
      }
    }
  }
  return messages;
}
function validateSources(command, sources, references) {
  const messages = [];
  for (const source of sources) {
    messages.push(...validateSource(source, command.name, references));
  }
  return messages;
}

/**
 * Validates the METRICS source command:
 *
 *     METRICS <sources> [ <aggregates> [ BY <grouping> ]]
 */
const validateMetricsCommand = (command, references) => {
  const messages = [];
  const {
    sources,
    aggregates,
    grouping
  } = command;

  // METRICS <sources> ...
  messages.push(...validateSources(command, sources, references));

  // ... <aggregates> ...
  if (aggregates && aggregates.length) {
    messages.push(...validateAggregates(command, aggregates, references));

    // ... BY <grouping>
    if (grouping && grouping.length) {
      messages.push(...validateByGrouping(grouping, 'metrics', references, true));
    }
  }
  return messages;
};
function validateCommand(command, references) {
  const messages = [];
  if (command.incomplete) {
    return messages;
  }
  // do not check the command exists, the grammar is already picking that up
  const commandDef = (0, _helpers.getCommandDefinition)(command.name);
  if (commandDef !== null && commandDef !== void 0 && commandDef.validate) {
    messages.push(...commandDef.validate(command));
  }
  switch (commandDef.name) {
    case 'metrics':
      {
        const metrics = command;
        messages.push(...validateMetricsCommand(metrics, references));
        break;
      }
    default:
      {
        // Now validate arguments
        for (const commandArg of command.args) {
          const wrappedArg = Array.isArray(commandArg) ? commandArg : [commandArg];
          for (const arg of wrappedArg) {
            if ((0, _helpers.isFunctionItem)(arg)) {
              messages.push(...validateFunction(arg, command.name, undefined, references));
            }
            if ((0, _helpers.isSettingItem)(arg)) {
              messages.push(...validateSetting(arg, commandDef.modes[0], command, references));
            }
            if ((0, _helpers.isOptionItem)(arg)) {
              messages.push(...validateOption(arg, commandDef.options.find(({
                name
              }) => name === arg.name), command, references));
            }
            if ((0, _helpers.isColumnItem)(arg)) {
              if (command.name === 'stats') {
                messages.push(_errors.errors.unknownAggFunction(arg));
              } else {
                messages.push(...validateColumnForCommand(arg, command.name, references));
              }
            }
            if ((0, _helpers.isTimeIntervalItem)(arg)) {
              messages.push((0, _errors.getMessageFromId)({
                messageId: 'unsupportedTypeForCommand',
                values: {
                  command: command.name.toUpperCase(),
                  type: 'date_period',
                  value: arg.name
                },
                locations: arg.location
              }));
            }
            if ((0, _helpers.isSourceItem)(arg)) {
              messages.push(...validateSource(arg, command.name, references));
            }
          }
        }
      }
  }

  // no need to check for mandatory options passed
  // as they are already validated at syntax level
  return messages;
}
function validateFieldsShadowing(fields, variables) {
  const messages = [];
  for (const variable of variables.keys()) {
    if (fields.has(variable)) {
      var _fields$get;
      const variableHits = variables.get(variable);
      if (!(0, _helpers.areFieldAndVariableTypesCompatible)((_fields$get = fields.get(variable)) === null || _fields$get === void 0 ? void 0 : _fields$get.type, variableHits[0].type)) {
        const fieldType = fields.get(variable).type;
        const variableType = variableHits[0].type;
        const flatFieldType = fieldType;
        const flatVariableType = variableType;
        messages.push((0, _errors.getMessageFromId)({
          messageId: 'shadowFieldType',
          values: {
            field: variable,
            fieldType: flatFieldType,
            newType: flatVariableType
          },
          locations: variableHits[0].location
        }));
      }
    }
  }
  return messages;
}
function validateUnsupportedTypeFields(fields) {
  const messages = [];
  for (const field of fields.values()) {
    if (field.type === 'unsupported') {
      messages.push((0, _errors.getMessageFromId)({
        messageId: 'unsupportedFieldType',
        values: {
          field: field.name
        },
        locations: {
          min: 1,
          max: 1
        }
      }));
    }
  }
  return messages;
}
const ignoreErrorsMap = exports.ignoreErrorsMap = {
  getFieldsFor: ['unknownColumn', 'wrongArgumentType', 'unsupportedFieldType'],
  getSources: ['unknownIndex'],
  getPolicies: ['unknownPolicy']
};

/**
 * ES|QL validation public API
 * It takes a query string and returns a list of messages (errors and warnings) after validate
 * The astProvider is optional, but if not provided the default one from '@kbn/esql-validation-autocomplete' will be used.
 * This is useful for async loading the ES|QL parser and reduce the bundle size, or to swap grammar version.
 * As for the callbacks, while optional, the validation function will selectively ignore some errors types based on each callback missing.
 */
async function validateQuery(queryString, astProvider, options = {}, callbacks) {
  const result = await validateAst(queryString, astProvider, callbacks);
  // early return if we do not want to ignore errors
  if (!options.ignoreOnMissingCallbacks) {
    return result;
  }
  const finalCallbacks = callbacks || {};
  const errorTypoesToIgnore = Object.entries(ignoreErrorsMap).reduce((acc, [key, errorCodes]) => {
    if (!(key in finalCallbacks) || key in finalCallbacks && finalCallbacks[key] == null) {
      for (const e of errorCodes) {
        acc[e] = true;
      }
    }
    return acc;
  }, {});
  const filteredErrors = result.errors.filter(error => {
    if ('severity' in error) {
      return true;
    }
    return !errorTypoesToIgnore[error.code];
  }).map(error => 'severity' in error ? {
    text: error.message,
    code: error.code,
    type: 'error',
    location: {
      min: error.startColumn,
      max: error.endColumn
    }
  } : error);
  return {
    errors: filteredErrors,
    warnings: result.warnings
  };
}

/**
 * This function will perform an high level validation of the
 * query AST. An initial syntax validation is already performed by the parser
 * while here it can detect things like function names, types correctness and potential warnings
 * @param ast A valid AST data structure
 */
async function validateAst(queryString, astProvider, callbacks) {
  const messages = [];
  const parsingResult = await astProvider(queryString);
  const {
    ast
  } = parsingResult;
  const [sources, availableFields, availablePolicies] = await Promise.all([
  // retrieve the list of available sources
  (0, _resources.retrieveSources)(ast, callbacks),
  // retrieve available fields (if a source command has been defined)
  (0, _resources.retrieveFields)(queryString, ast, callbacks),
  // retrieve available policies (if an enrich command has been defined)
  (0, _resources.retrievePolicies)(ast, callbacks)]);
  if (availablePolicies.size) {
    const fieldsFromPoliciesMap = await (0, _resources.retrievePoliciesFields)(ast, availablePolicies, callbacks);
    fieldsFromPoliciesMap.forEach((value, key) => availableFields.set(key, value));
  }
  if (ast.some(({
    name
  }) => ['grok', 'dissect'].includes(name))) {
    const fieldsFromGrokOrDissect = await (0, _resources.retrieveFieldsFromStringSources)(queryString, ast, callbacks);
    fieldsFromGrokOrDissect.forEach((value, key) => {
      // if the field is already present, do not overwrite it
      // Note: this can also overlap with some variables
      if (!availableFields.has(key)) {
        availableFields.set(key, value);
      }
    });
  }
  const variables = (0, _variables.collectVariables)(ast, availableFields, queryString);
  // notify if the user is rewriting a column as variable with another type
  messages.push(...validateFieldsShadowing(availableFields, variables));
  messages.push(...validateUnsupportedTypeFields(availableFields));
  for (const command of ast) {
    const references = {
      sources,
      fields: availableFields,
      policies: availablePolicies,
      variables,
      query: queryString
    };
    const commandMessages = validateCommand(command, references);
    messages.push(...commandMessages);
  }
  return {
    errors: [...parsingResult.errors, ...messages.filter(({
      type
    }) => type === 'error')],
    warnings: messages.filter(({
      type
    }) => type === 'warning')
  };
}