"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectVariables = collectVariables;
exports.collectVariablesFromList = void 0;
exports.excludeVariablesFromCurrentCommand = excludeVariablesFromCurrentCommand;
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function addToVariableOccurrencies(variables, instance) {
  if (!variables.has(instance.name)) {
    variables.set(instance.name, []);
  }
  const variablesOccurrencies = variables.get(instance.name);
  variablesOccurrencies.push(instance);
}
function addToVariables(oldArg, newArg, fields, variables) {
  if ((0, _helpers.isColumnItem)(oldArg) && (0, _helpers.isColumnItem)(newArg)) {
    const newVariable = {
      name: newArg.name,
      type: 'number' /* fallback to number */,
      location: newArg.location
    };
    // Now workout the exact type
    // it can be a rename of another variable as well
    const oldRef = fields.get(oldArg.name) || variables.get(oldArg.quoted ? oldArg.text : oldArg.name);
    if (oldRef) {
      addToVariableOccurrencies(variables, newVariable);
      newVariable.type = Array.isArray(oldRef) ? oldRef[0].type : oldRef.type;
    }
  }
}
function getAssignRightHandSideType(item, fields) {
  if (Array.isArray(item)) {
    const firstArg = item[0];
    if (Array.isArray(firstArg) || !firstArg) {
      return;
    }
    if (firstArg.type === 'literal') {
      return firstArg.literalType;
    }
    if ((0, _helpers.isColumnItem)(firstArg)) {
      const field = fields.get(firstArg.name);
      if (field) {
        return field.type;
      }
    }
    if ((0, _helpers.isFunctionItem)(firstArg)) {
      const fnDefinition = (0, _helpers.getFunctionDefinition)(firstArg.name);
      return fnDefinition === null || fnDefinition === void 0 ? void 0 : fnDefinition.signatures[0].returnType;
    }
    return firstArg.type;
  }
}
function excludeVariablesFromCurrentCommand(commands, currentCommand, fieldsMap, queryString) {
  const anyVariables = collectVariables(commands, fieldsMap, queryString);
  const currentCommandVariables = collectVariables([currentCommand], fieldsMap, queryString);
  const resultVariables = new Map();
  anyVariables.forEach((value, key) => {
    if (!currentCommandVariables.has(key)) {
      resultVariables.set(key, value);
    }
  });
  return resultVariables;
}
function extractExpressionAsQuotedVariable(originalQuery, location) {
  const extractExpressionText = originalQuery.substring(location.min, location.max + 1);
  // now inject quotes and save it as variable
  return `\`${extractExpressionText.replaceAll(_constants.SINGLE_BACKTICK, _constants.DOUBLE_BACKTICK)}\``;
}
function addVariableFromAssignment(assignOperation, variables, fields) {
  if ((0, _helpers.isColumnItem)(assignOperation.args[0])) {
    const rightHandSideArgType = getAssignRightHandSideType(assignOperation.args[1], fields);
    addToVariableOccurrencies(variables, {
      name: assignOperation.args[0].name,
      type: rightHandSideArgType || 'number' /* fallback to number */,
      location: assignOperation.args[0].location
    });
  }
}
function addVariableFromExpression(expressionOperation, queryString, variables) {
  if (!expressionOperation.text.includes(_constants.EDITOR_MARKER)) {
    // save the variable in its quoted usable way
    // (a bit of forward thinking here to simplyfy lookups later)
    const forwardThinkingVariableName = extractExpressionAsQuotedVariable(queryString, expressionOperation.location);
    const expressionType = 'number';
    addToVariableOccurrencies(variables, {
      name: forwardThinkingVariableName,
      type: expressionType,
      location: expressionOperation.location
    });
  }
}
const collectVariablesFromList = (list, fields, queryString, variables) => {
  for (const arg of list) {
    if ((0, _helpers.isAssignment)(arg)) {
      addVariableFromAssignment(arg, variables, fields);
    } else if ((0, _helpers.isExpression)(arg)) {
      addVariableFromExpression(arg, queryString, variables);
    }
  }
};
exports.collectVariablesFromList = collectVariablesFromList;
function collectVariables(commands, fields, queryString) {
  const variables = new Map();
  for (const command of commands) {
    if (['row', 'eval', 'stats', 'metrics'].includes(command.name)) {
      collectVariablesFromList(command.args, fields, queryString, variables);
      if (command.name === 'stats') {
        const commandOptionsWithAssignment = command.args.filter(arg => (0, _helpers.isOptionItem)(arg) && arg.name === 'by');
        for (const commandOption of commandOptionsWithAssignment) {
          collectVariablesFromList(commandOption.args, fields, queryString, variables);
        }
      }
    }
    if (command.name === 'enrich') {
      const commandOptionsWithAssignment = command.args.filter(arg => (0, _helpers.isOptionItem)(arg) && arg.name === 'with');
      for (const commandOption of commandOptionsWithAssignment) {
        // Enrich assignment has some special behaviour, so do not use the version above here...
        for (const assignFn of commandOption.args) {
          if ((0, _helpers.isFunctionItem)(assignFn)) {
            const [newArg, oldArg] = (assignFn === null || assignFn === void 0 ? void 0 : assignFn.args) || [];
            if (Array.isArray(oldArg)) {
              addToVariables(oldArg[0], newArg, fields, variables);
            }
          }
        }
      }
    }
    if (command.name === 'rename') {
      const commandOptionsWithAssignment = command.args.filter(arg => (0, _helpers.isOptionItem)(arg) && arg.name === 'as');
      for (const commandOption of commandOptionsWithAssignment) {
        const [oldArg, newArg] = commandOption.args;
        addToVariables(oldArg, newArg, fields, variables);
      }
    }
  }
  return variables;
}