"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.areFieldAndVariableTypesCompatible = areFieldAndVariableTypesCompatible;
exports.checkFunctionArgMatchesDefinition = checkFunctionArgMatchesDefinition;
exports.createMapFromList = createMapFromList;
exports.extractSingularType = extractSingularType;
exports.findPreviousWord = findPreviousWord;
exports.getAllArrayTypes = getAllArrayTypes;
exports.getAllArrayValues = getAllArrayValues;
exports.getAllCommands = getAllCommands;
exports.getAllFunctions = getAllFunctions;
exports.getColumnExists = getColumnExists;
exports.getCommandDefinition = getCommandDefinition;
exports.getCommandOption = getCommandOption;
exports.getFunctionDefinition = getFunctionDefinition;
exports.getLastCharFromTrimmed = getLastCharFromTrimmed;
exports.getUnquotedColumnName = exports.getQuotedColumnName = void 0;
exports.hasCCSSource = hasCCSSource;
exports.hasWildcard = hasWildcard;
exports.inKnownTimeInterval = inKnownTimeInterval;
exports.isAggFunction = void 0;
exports.isArrayType = isArrayType;
exports.isAssignment = isAssignment;
exports.isAssignmentComplete = isAssignmentComplete;
exports.isColumnItem = isColumnItem;
exports.isComma = isComma;
exports.isExpression = isExpression;
exports.isFunctionItem = isFunctionItem;
exports.isIncompleteItem = isIncompleteItem;
exports.isInlineCastItem = isInlineCastItem;
exports.isLiteralItem = isLiteralItem;
exports.isMathFunction = isMathFunction;
exports.isOptionItem = isOptionItem;
exports.isRestartingExpression = isRestartingExpression;
exports.isSettingItem = isSettingItem;
exports.isSingleItem = isSingleItem;
exports.isSourceCommand = isSourceCommand;
exports.isSourceItem = isSourceItem;
exports.isSupportedFunction = isSupportedFunction;
exports.isTimeIntervalItem = isTimeIntervalItem;
exports.isValidLiteralOption = isValidLiteralOption;
exports.isVariable = isVariable;
exports.lookupColumn = lookupColumn;
exports.nonNullable = nonNullable;
exports.pipePrecedesCurrentWord = pipePrecedesCurrentWord;
exports.printFunctionSignature = printFunctionSignature;
exports.shouldBeQuotedSource = shouldBeQuotedSource;
exports.shouldBeQuotedText = shouldBeQuotedText;
exports.sourceExists = sourceExists;
var _aggs = require("../definitions/aggs");
var _builtin = require("../definitions/builtin");
var _commands = require("../definitions/commands");
var _functions = require("../definitions/functions");
var _grouping = require("../definitions/grouping");
var _helpers = require("../definitions/helpers");
var _literals = require("../definitions/literals");
var _options = require("../definitions/options");
var _context = require("./context");
var _esql_to_kibana_type = require("./esql_to_kibana_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function nonNullable(v) {
  return v != null;
}
function isSingleItem(arg) {
  return arg && !Array.isArray(arg);
}
function isSettingItem(arg) {
  return isSingleItem(arg) && arg.type === 'mode';
}
function isFunctionItem(arg) {
  return isSingleItem(arg) && arg.type === 'function';
}
function isOptionItem(arg) {
  return isSingleItem(arg) && arg.type === 'option';
}
function isSourceItem(arg) {
  return isSingleItem(arg) && arg.type === 'source';
}
function isColumnItem(arg) {
  return isSingleItem(arg) && arg.type === 'column';
}
function isLiteralItem(arg) {
  return isSingleItem(arg) && arg.type === 'literal';
}
function isInlineCastItem(arg) {
  return isSingleItem(arg) && arg.type === 'inlineCast';
}
function isTimeIntervalItem(arg) {
  return isSingleItem(arg) && arg.type === 'timeInterval';
}
function isAssignment(arg) {
  return isFunctionItem(arg) && arg.name === '=';
}
function isAssignmentComplete(node) {
  var _removeMarkerArgFromA, _removeMarkerArgFromA2;
  const assignExpression = (_removeMarkerArgFromA = (0, _context.removeMarkerArgFromArgsList)(node)) === null || _removeMarkerArgFromA === void 0 ? void 0 : (_removeMarkerArgFromA2 = _removeMarkerArgFromA.args) === null || _removeMarkerArgFromA2 === void 0 ? void 0 : _removeMarkerArgFromA2[1];
  return Boolean(assignExpression && Array.isArray(assignExpression) && assignExpression.length);
}
function isExpression(arg) {
  return isFunctionItem(arg) && arg.name !== '=';
}
function isIncompleteItem(arg) {
  return !arg || !Array.isArray(arg) && arg.incomplete;
}
function isMathFunction(query, offset) {
  const queryTrimmed = query.trimEnd();
  // try to get the full operation token (e.g. "+", "in", "like", etc...) but it requires the token
  // to be spaced out from a field/function (e.g. "field + ") so it is subject to issues
  const [opString] = queryTrimmed.split(' ').reverse();
  // compare last char for all math functions
  // limit only to 2 chars operators
  const fns = _builtin.builtinFunctions.filter(({
    name
  }) => name.length < 3).map(({
    name
  }) => name);
  const tokenMatch = fns.some(op => opString === op);
  // there's a match, that's good
  if (tokenMatch) {
    return true;
  }
  // either there's no match or it is the case where field/function and op are not spaced out
  // e.g "field+" or "fn()+"
  // so try to extract the last char and compare it with the single char math functions
  const singleCharFns = fns.filter(name => name.length === 1);
  return singleCharFns.some(c => c === opString[opString.length - 1]);
}
function isComma(char) {
  return char === ',';
}
function isSourceCommand({
  label
}) {
  return ['FROM', 'ROW', 'SHOW'].includes(label);
}
let fnLookups;
let commandLookups;
function buildFunctionLookup() {
  if (!fnLookups) {
    fnLookups = _builtin.builtinFunctions.concat(_functions.evalFunctionDefinitions, _aggs.statsAggregationFunctionDefinitions, _grouping.groupingFunctionDefinitions).reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        for (const alias of def.alias) {
          memo.set(alias, def);
        }
      }
      return memo;
    }, new Map());
  }
  return fnLookups;
}
function isSupportedFunction(name, parentCommand, option) {
  var _fn$supportedOptions;
  if (!parentCommand) {
    return {
      supported: false,
      reason: 'missingCommand'
    };
  }
  const fn = buildFunctionLookup().get(name);
  const isSupported = Boolean(option == null ? fn === null || fn === void 0 ? void 0 : fn.supportedCommands.includes(parentCommand) : fn === null || fn === void 0 ? void 0 : (_fn$supportedOptions = fn.supportedOptions) === null || _fn$supportedOptions === void 0 ? void 0 : _fn$supportedOptions.includes(option));
  return {
    supported: isSupported,
    reason: isSupported ? undefined : fn ? 'unsupportedFunction' : 'unknownFunction'
  };
}
function getAllFunctions(options) {
  const fns = buildFunctionLookup();
  if (!(options !== null && options !== void 0 && options.type)) {
    return Array.from(fns.values());
  }
  const types = new Set(Array.isArray(options.type) ? options.type : [options.type]);
  return Array.from(fns.values()).filter(fn => types.has(fn.type));
}
function getFunctionDefinition(name) {
  return buildFunctionLookup().get(name.toLowerCase());
}
const unwrapStringLiteralQuotes = value => value.slice(1, -1);
function buildCommandLookup() {
  if (!commandLookups) {
    commandLookups = _commands.commandDefinitions.reduce((memo, def) => {
      memo.set(def.name, def);
      if (def.alias) {
        memo.set(def.alias, def);
      }
      return memo;
    }, new Map());
  }
  return commandLookups;
}
function getCommandDefinition(name) {
  return buildCommandLookup().get(name.toLowerCase());
}
function getAllCommands() {
  return Array.from(buildCommandLookup().values());
}
function getCommandOption(optionName) {
  return [_options.byOption, _options.metadataOption, _options.asOption, _options.onOption, _options.withOption, _options.appendSeparatorOption].find(({
    name
  }) => name === optionName);
}
function compareLiteralType(argType, item) {
  if (item.literalType === 'null') {
    return true;
  }
  if (item.literalType !== 'string') {
    if (argType === item.literalType) {
      return true;
    }
    return false;
  }
  if (argType === 'chrono_literal') {
    return _literals.chronoLiterals.some(({
      name
    }) => name === item.text);
  }
  // date-type parameters accept string literals because of ES auto-casting
  return ['string', 'date'].includes(argType);
}

/**
 * This function returns the variable or field matching a column
 */
function lookupColumn(column, {
  fields,
  variables
}) {
  var _variables$get;
  const columnName = getQuotedColumnName(column);
  return fields.get(columnName) || ((_variables$get = variables.get(columnName)) === null || _variables$get === void 0 ? void 0 : _variables$get[0]);
}
const ARRAY_REGEXP = /\[\]$/;
function isArrayType(type) {
  return ARRAY_REGEXP.test(type);
}
const arrayToSingularMap = new Map([['number[]', 'number'], ['date[]', 'date'], ['boolean[]', 'boolean'], ['string[]', 'string'], ['any[]', 'any']]);

/**
 * Given an array type for example `string[]` it will return `string`
 */
function extractSingularType(type) {
  var _arrayToSingularMap$g;
  return (_arrayToSingularMap$g = arrayToSingularMap.get(type)) !== null && _arrayToSingularMap$g !== void 0 ? _arrayToSingularMap$g : type;
}
function createMapFromList(arr) {
  const arrMap = new Map();
  for (const item of arr) {
    arrMap.set(item.name, item);
  }
  return arrMap;
}
function areFieldAndVariableTypesCompatible(fieldType, variableType) {
  if (fieldType == null) {
    return false;
  }
  return fieldType === variableType;
}
function printFunctionSignature(arg) {
  const fnDef = getFunctionDefinition(arg.name);
  if (fnDef) {
    const signature = (0, _helpers.getFunctionSignatures)({
      ...fnDef,
      signatures: [{
        ...(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures[0]),
        params: arg.args.map(innerArg => Array.isArray(innerArg) ? {
          name: `InnerArgument[]`,
          type: 'any'
        } :
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        {
          name: innerArg.text,
          type: innerArg.type
        }),
        // this cast isn't actually correct, but we're abusing the
        // getFunctionSignatures API anyways
        returnType: ''
      }]
    }, {
      withTypes: false,
      capitalize: true
    });
    return signature[0].declaration;
  }
  return '';
}
function getAllArrayValues(arg) {
  const values = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        values.push(String(subArg.value));
      }
      if (isColumnItem(subArg) || isTimeIntervalItem(subArg)) {
        values.push(subArg.name);
      }
      if (subArg.type === 'function') {
        const signature = printFunctionSignature(subArg);
        if (signature) {
          values.push(signature);
        }
      }
    }
  }
  return values;
}
function getAllArrayTypes(arg, parentCommand, references) {
  const types = [];
  if (Array.isArray(arg)) {
    for (const subArg of arg) {
      if (Array.isArray(subArg)) {
        break;
      }
      if (subArg.type === 'literal') {
        types.push(subArg.literalType);
      }
      if (subArg.type === 'column') {
        const hit = lookupColumn(subArg, references);
        types.push((hit === null || hit === void 0 ? void 0 : hit.type) || 'unsupported');
      }
      if (subArg.type === 'timeInterval') {
        types.push('time_literal');
      }
      if (subArg.type === 'function') {
        if (isSupportedFunction(subArg.name, parentCommand).supported) {
          const fnDef = buildFunctionLookup().get(subArg.name);
          types.push(fnDef.signatures[0].returnType);
        }
      }
    }
  }
  return types;
}
function inKnownTimeInterval(item) {
  return _literals.timeUnits.some(unit => unit === item.unit.toLowerCase());
}

/**
 * Checks if this argument is one of the possible options
 * if they are defined on the arg definition.
 *
 * TODO - Consider merging with isEqualType to create a unified arg validation function
 */
function isValidLiteralOption(arg, argDef) {
  return arg.literalType === 'string' && argDef.literalOptions && !argDef.literalOptions.map(option => option.toLowerCase()).includes(unwrapStringLiteralQuotes(arg.value).toLowerCase());
}

/**
 * Checks if an AST function argument is of the correct type
 * given the definition.
 */
function checkFunctionArgMatchesDefinition(arg, parameterDefinition, references, parentCommand) {
  const argType = parameterDefinition.type;
  if (argType === 'any') {
    return true;
  }
  if (arg.type === 'literal') {
    return compareLiteralType(argType, arg);
  }
  if (arg.type === 'function') {
    if (isSupportedFunction(arg.name, parentCommand).supported) {
      const fnDef = buildFunctionLookup().get(arg.name);
      return fnDef.signatures.some(signature => signature.returnType === 'any' || argType === signature.returnType);
    }
  }
  if (arg.type === 'timeInterval') {
    return argType === 'time_literal' && inKnownTimeInterval(arg);
  }
  if (arg.type === 'column') {
    const hit = lookupColumn(arg, references);
    const validHit = hit;
    if (!validHit) {
      return false;
    }
    const wrappedTypes = Array.isArray(validHit.type) ? validHit.type : [validHit.type];
    // if final type is of type any make it pass for now
    return wrappedTypes.some(ct => ['any', 'null'].includes(ct) || argType === ct);
  }
  if (arg.type === 'inlineCast') {
    // TODO - remove with https://github.com/elastic/kibana/issues/174710
    return argType === (0, _esql_to_kibana_type.esqlToKibanaType)(arg.castType);
  }
}
function fuzzySearch(fuzzyName, resources) {
  const wildCardPosition = getWildcardPosition(fuzzyName);
  if (wildCardPosition !== 'none') {
    const matcher = getMatcher(fuzzyName, wildCardPosition);
    for (const resourceName of resources) {
      if (matcher(resourceName)) {
        return true;
      }
    }
  }
}
function getMatcher(name, position) {
  if (position === 'start') {
    const prefix = name.substring(1);
    return resource => resource.endsWith(prefix);
  }
  if (position === 'end') {
    const prefix = name.substring(0, name.length - 1);
    return resource => resource.startsWith(prefix);
  }
  if (position === 'multiple-within') {
    // make sure to remove the * at the beginning of the name if present
    const safeName = name.startsWith('*') ? name.slice(1) : name;
    // replace 2 ore more consecutive wildcards with a single one
    const setOfChars = safeName.replace(/\*{2+}/g, '*').split('*');
    return resource => {
      let index = -1;
      return setOfChars.every(char => {
        index = resource.indexOf(char, index + 1);
        return index !== -1;
      });
    };
  }
  const [prefix, postFix] = name.split('*');
  return resource => resource.startsWith(prefix) && resource.endsWith(postFix);
}
function getWildcardPosition(name) {
  if (!hasWildcard(name)) {
    return 'none';
  }
  const wildCardCount = name.match(/\*/g).length;
  if (wildCardCount > 1) {
    return 'multiple-within';
  }
  if (name.startsWith('*')) {
    return 'start';
  }
  if (name.endsWith('*')) {
    return 'end';
  }
  return 'middle';
}
function hasWildcard(name) {
  return /\*/.test(name);
}
function isVariable(column) {
  return Boolean(column && 'location' in column);
}
function hasCCSSource(name) {
  return name.includes(':');
}

/**
 * This will return the name without any quotes.
 *
 * E.g. "`bytes`" will become "bytes"
 *
 * @param column
 * @returns
 */
const getUnquotedColumnName = column => column.name;

/**
 * This returns the name with any quotes that were present.
 *
 * E.g. "`bytes`" will be "`bytes`"
 *
 * @param column
 * @returns
 */
exports.getUnquotedColumnName = getUnquotedColumnName;
const getQuotedColumnName = column => column.quoted ? column.text : column.name;

/**
 * TODO - consider calling lookupColumn under the hood of this function. Seems like they should really do the same thing.
 */
exports.getQuotedColumnName = getQuotedColumnName;
function getColumnExists(column, {
  fields,
  variables
}) {
  const namesToCheck = [getUnquotedColumnName(column), getQuotedColumnName(column)];
  for (const name of namesToCheck) {
    if (fields.has(name) || variables.has(name)) {
      return true;
    }

    // TODO — I don't see this fuzzy searching in lookupColumn... should it be there?
    if (Boolean(fuzzySearch(name, fields.keys()) || fuzzySearch(name, variables.keys()))) {
      return true;
    }
  }
  return false;
}
function sourceExists(index, sources) {
  if (sources.has(index) || index.startsWith('-')) {
    return true;
  }
  return Boolean(fuzzySearch(index, sources.keys()));
}

/**
 * Works backward from the cursor position to determine if
 * the final character of the previous word matches the given character.
 */
function characterPrecedesCurrentWord(text, char) {
  let inCurrentWord = true;
  for (let i = text.length - 1; i >= 0; i--) {
    if (inCurrentWord && /\s/.test(text[i])) {
      inCurrentWord = false;
    }
    if (!inCurrentWord && !/\s/.test(text[i])) {
      return text[i] === char;
    }
  }
}
function pipePrecedesCurrentWord(text) {
  return characterPrecedesCurrentWord(text, '|');
}
function getLastCharFromTrimmed(text) {
  return text[text.trimEnd().length - 1];
}

/**
 * Are we after a comma? i.e. STATS fieldA, <here>
 */
function isRestartingExpression(text) {
  return getLastCharFromTrimmed(text) === ',' || characterPrecedesCurrentWord(text, ',');
}
function findPreviousWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 2];
}
function shouldBeQuotedSource(text) {
  // Based on lexer `fragment UNQUOTED_SOURCE_PART`
  return /[:"=|,[\]\/ \t\r\n]/.test(text);
}
function shouldBeQuotedText(text, {
  dashSupported
} = {}) {
  return dashSupported ? /[^a-zA-Z\d_\.@-]/.test(text) : /[^a-zA-Z\d_\.@]/.test(text);
}
const isAggFunction = arg => {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = getFunctionDefinition(arg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === 'agg';
};
exports.isAggFunction = isAggFunction;