"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCompatibleFunctionDefinition = exports.buildVariablesDefinitions = exports.buildValueDefinitions = exports.buildSourcesDefinitions = exports.buildSettingDefinitions = exports.buildPoliciesDefinitions = exports.buildOptionDefinition = exports.buildNoPoliciesAvailableDefinition = exports.buildNewVarDefinition = exports.buildMatchingFieldsDefinition = exports.buildFieldsDefinitions = exports.buildConstantsDefinitions = exports.TRIGGER_SUGGESTION_COMMAND = void 0;
exports.getCompatibleLiterals = getCompatibleLiterals;
exports.getQuotedText = getQuotedText;
exports.getSafeInsertText = getSafeInsertText;
exports.getSuggestionBuiltinDefinition = getSuggestionBuiltinDefinition;
exports.getSuggestionCommandDefinition = getSuggestionCommandDefinition;
exports.getSuggestionFunctionDefinition = getSuggestionFunctionDefinition;
exports.getUnitDuration = getUnitDuration;
var _i18n = require("@kbn/i18n");
var _grouping = require("../definitions/grouping");
var _aggs = require("../definitions/aggs");
var _functions = require("../definitions/functions");
var _helpers = require("../definitions/helpers");
var _literals = require("../definitions/literals");
var _helpers2 = require("../shared/helpers");
var _documentation_util = require("./documentation_util");
var _constants = require("../shared/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const allFunctions = _aggs.statsAggregationFunctionDefinitions.concat(_functions.evalFunctionDefinitions).concat(_grouping.groupingFunctionDefinitions);
const TRIGGER_SUGGESTION_COMMAND = exports.TRIGGER_SUGGESTION_COMMAND = {
  title: 'Trigger Suggestion Dialog',
  id: 'editor.action.triggerSuggest'
};
function getSafeInsertText(text, options = {}) {
  return (0, _helpers2.shouldBeQuotedText)(text, options) ? `\`${text.replace(_constants.SINGLE_TICK_REGEX, _constants.DOUBLE_BACKTICK)}\`` : text;
}
function getQuotedText(text) {
  return text.startsWith(`"`) && text.endsWith(`"`) ? text : `"${text}"`;
}
function getSafeInsertSourceText(text) {
  return (0, _helpers2.shouldBeQuotedSource)(text) ? getQuotedText(text) : text;
}
function getSuggestionFunctionDefinition(fn) {
  const fullSignatures = (0, _helpers.getFunctionSignatures)(fn, {
    capitalize: true,
    withTypes: true
  });
  return {
    label: fullSignatures[0].declaration,
    text: `${fn.name.toUpperCase()}($0)`,
    asSnippet: true,
    kind: 'Function',
    detail: fn.description,
    documentation: {
      value: (0, _documentation_util.buildFunctionDocumentation)(fullSignatures, fn.examples)
    },
    // agg functgions have priority over everything else
    sortText: fn.type === 'agg' ? '1A' : 'C',
    // trigger a suggestion follow up on selection
    command: TRIGGER_SUGGESTION_COMMAND
  };
}
function getSuggestionBuiltinDefinition(fn) {
  const hasArgs = fn.signatures.some(({
    params
  }) => params.length > 1);
  return {
    label: fn.name.toUpperCase(),
    text: hasArgs ? `${fn.name.toUpperCase()} $0` : fn.name.toUpperCase(),
    asSnippet: hasArgs,
    kind: 'Operator',
    detail: fn.description,
    documentation: {
      value: ''
    },
    sortText: 'D',
    command: hasArgs ? TRIGGER_SUGGESTION_COMMAND : undefined
  };
}
const getCompatibleFunctionDefinition = (command, option, returnTypes, ignored = []) => {
  const fnSupportedByCommand = allFunctions.filter(({
    name,
    supportedCommands,
    supportedOptions
  }) => (option ? supportedOptions === null || supportedOptions === void 0 ? void 0 : supportedOptions.includes(option) : supportedCommands.includes(command)) && !ignored.includes(name)).sort((a, b) => a.name.localeCompare(b.name));
  if (!returnTypes) {
    return fnSupportedByCommand.map(getSuggestionFunctionDefinition);
  }
  return fnSupportedByCommand.filter(mathDefinition => mathDefinition.signatures.some(signature => returnTypes[0] === 'any' || returnTypes.includes(signature.returnType))).map(getSuggestionFunctionDefinition);
};
exports.getCompatibleFunctionDefinition = getCompatibleFunctionDefinition;
function getSuggestionCommandDefinition(command) {
  const commandDefinition = (0, _helpers2.getCommandDefinition)(command.name);
  const commandSignature = (0, _helpers.getCommandSignature)(commandDefinition);
  return {
    label: commandDefinition.name.toUpperCase(),
    text: commandDefinition.signature.params.length ? `${commandDefinition.name.toUpperCase()} $0` : commandDefinition.name.toUpperCase(),
    asSnippet: true,
    kind: 'Method',
    detail: commandDefinition.description,
    documentation: {
      value: (0, _documentation_util.buildDocumentation)(commandSignature.declaration, commandSignature.examples)
    },
    sortText: 'A',
    command: TRIGGER_SUGGESTION_COMMAND
  };
}
const buildFieldsDefinitions = fields => fields.map(label => ({
  label,
  text: getSafeInsertText(label),
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.fieldDefinition', {
    defaultMessage: `Field specified by the input table`
  }),
  sortText: 'D'
}));
exports.buildFieldsDefinitions = buildFieldsDefinitions;
const buildVariablesDefinitions = variables => variables.map(label => ({
  label,
  text: label,
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.variableDefinition', {
    defaultMessage: `Variable specified by the user within the ES|QL query`
  }),
  sortText: 'D'
}));
exports.buildVariablesDefinitions = buildVariablesDefinitions;
const buildSourcesDefinitions = sources => sources.map(({
  name,
  isIntegration,
  title
}) => ({
  label: title !== null && title !== void 0 ? title : name,
  text: getSafeInsertSourceText(name),
  isSnippet: isIntegration,
  ...(isIntegration && {
    command: TRIGGER_SUGGESTION_COMMAND
  }),
  kind: isIntegration ? 'Class' : 'Issue',
  detail: isIntegration ? _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.integrationDefinition', {
    defaultMessage: `Integration`
  }) : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.sourceDefinition', {
    defaultMessage: `Index`
  }),
  sortText: 'A'
}));
exports.buildSourcesDefinitions = buildSourcesDefinitions;
const buildConstantsDefinitions = (userConstants, detail) => userConstants.map(label => ({
  label,
  text: label,
  kind: 'Constant',
  detail: detail !== null && detail !== void 0 ? detail : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.constantDefinition', {
    defaultMessage: `Constant`
  }),
  sortText: 'A'
}));
exports.buildConstantsDefinitions = buildConstantsDefinitions;
const buildValueDefinitions = (values, detail) => values.map(value => ({
  label: `"${value}"`,
  text: `"${value}"`,
  detail: detail !== null && detail !== void 0 ? detail : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.valueDefinition', {
    defaultMessage: 'Literal value'
  }),
  kind: 'Value'
}));
exports.buildValueDefinitions = buildValueDefinitions;
const buildNewVarDefinition = label => {
  return {
    label,
    text: `${label} =`,
    kind: 'Variable',
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.newVarDoc', {
      defaultMessage: 'Define a new variable'
    }),
    sortText: '1'
  };
};
exports.buildNewVarDefinition = buildNewVarDefinition;
const buildPoliciesDefinitions = policies => policies.map(({
  name: label,
  sourceIndices
}) => ({
  label,
  text: getSafeInsertText(label, {
    dashSupported: true
  }),
  kind: 'Class',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.policyDefinition', {
    defaultMessage: `Policy defined on {count, plural, one {index} other {indices}}: {indices}`,
    values: {
      count: sourceIndices.length,
      indices: sourceIndices.join(', ')
    }
  }),
  sortText: 'D'
}));
exports.buildPoliciesDefinitions = buildPoliciesDefinitions;
const buildMatchingFieldsDefinition = (matchingField, fields) => fields.map(label => ({
  label,
  text: getSafeInsertText(label),
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.matchingFieldDefinition', {
    defaultMessage: `Use to match on {matchingField} on the policy`,
    values: {
      matchingField
    }
  }),
  sortText: 'D'
}));
exports.buildMatchingFieldsDefinition = buildMatchingFieldsDefinition;
const buildOptionDefinition = (option, isAssignType = false) => {
  const completeItem = {
    label: option.name.toUpperCase(),
    text: option.name.toUpperCase(),
    kind: 'Reference',
    detail: option.description,
    sortText: '1'
  };
  if (isAssignType || option.signature.params.length) {
    completeItem.text = isAssignType ? `${option.name.toUpperCase()} = $0` : `${option.name.toUpperCase()} $0`;
    completeItem.asSnippet = true;
    completeItem.command = TRIGGER_SUGGESTION_COMMAND;
  }
  return completeItem;
};
exports.buildOptionDefinition = buildOptionDefinition;
const buildSettingDefinitions = setting => {
  // for now there's just a single setting with one argument
  return setting.values.map(({
    name,
    description
  }) => ({
    label: `${setting.prefix || ''}${name}`,
    text: `${setting.prefix || ''}${name}:$0`,
    asSnippet: true,
    kind: 'Reference',
    detail: description ? `${setting.description} - ${description}` : setting.description,
    sortText: 'D',
    command: TRIGGER_SUGGESTION_COMMAND
  }));
};
exports.buildSettingDefinitions = buildSettingDefinitions;
const buildNoPoliciesAvailableDefinition = () => ({
  label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.noPoliciesLabel', {
    defaultMessage: 'No available policy'
  }),
  text: '',
  kind: 'Issue',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.noPoliciesLabelsFound', {
    defaultMessage: 'Click to create'
  }),
  sortText: 'D',
  command: {
    id: 'esql.policies.create',
    title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createNewPolicy', {
      defaultMessage: 'Click to create'
    })
  }
});
exports.buildNoPoliciesAvailableDefinition = buildNoPoliciesAvailableDefinition;
function getUnitDuration(unit = 1) {
  const filteredTimeLiteral = _literals.timeUnitsToSuggest.filter(({
    name
  }) => {
    const result = /s$/.test(name);
    return unit > 1 ? result : !result;
  });
  return filteredTimeLiteral.map(({
    name
  }) => `${unit} ${name}`);
}

/**
 * Given information about the current command and the parameter type, suggest
 * some literals that may make sense.
 *
 * TODO — this currently tries to cover both command-specific suggestions and type
 * suggestions. We could consider separating the two... or just using parameter types
 * and forgetting about command-specific suggestions altogether.
 *
 * Another thought... should literal suggestions be defined in the definitions file?
 * That approach might allow for greater specificity in the suggestions and remove some
 * "magical" logic. Maybe this is really the same thing as the literalOptions parameter
 * definition property...
 */
function getCompatibleLiterals(commandName, types, names) {
  const suggestions = [];
  if (types.includes('number')) {
    if (commandName === 'limit') {
      // suggest 10/100/1000 for limit
      suggestions.push(...buildConstantsDefinitions(['10', '100', '1000'], ''));
    }
  }
  if (types.includes('time_literal')) {
    // filter plural for now and suggest only unit + singular
    suggestions.push(...buildConstantsDefinitions(getUnitDuration(1))); // i.e. 1 year
  }
  // this is a special type built from the suggestion system, not inherited from the AST
  if (types.includes('time_literal_unit')) {
    suggestions.push(...buildConstantsDefinitions(_literals.timeUnitsToSuggest.map(({
      name
    }) => name))); // i.e. year, month, ...
  }
  if (types.includes('chrono_literal')) {
    suggestions.push(...buildConstantsDefinitions(_literals.chronoLiterals.map(({
      name
    }) => name))); // i.e. EPOC_DAY, ...
  }
  if (types.includes('string')) {
    if (names) {
      const index = types.indexOf('string');
      if (/pattern/.test(names[index])) {
        suggestions.push(...buildConstantsDefinitions([commandName === 'grok' ? '"%{WORD:firstWord}"' : '"%{firstWord}"'], _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.aPatternString', {
          defaultMessage: 'A pattern string'
        })));
      } else {
        suggestions.push(...buildConstantsDefinitions(['string'], ''));
      }
    }
  }
  return suggestions;
}