"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggest = suggest;
var _uniqBy = _interopRequireDefault(require("lodash/uniqBy"));
var _lodash = require("lodash");
var _helpers = require("../shared/helpers");
var _variables = require("../shared/variables");
var _complete_items = require("./complete_items");
var _factories = require("./factories");
var _constants = require("../shared/constants");
var _context = require("../shared/context");
var _resources_helpers = require("../shared/resources_helpers");
var _helper = require("./helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function hasSameArgBothSides(assignFn) {
  if (assignFn.name === '=' && (0, _helpers.isColumnItem)(assignFn.args[0]) && assignFn.args[1]) {
    const assignValue = assignFn.args[1];
    if (Array.isArray(assignValue) && (0, _helpers.isColumnItem)(assignValue[0])) {
      return assignFn.args[0].name === assignValue[0].name;
    }
  }
}
function appendEnrichFields(fieldsMap, policyMetadata) {
  if (!policyMetadata) {
    return fieldsMap;
  }
  // @TODO: improve this
  const newMap = new Map(fieldsMap);
  for (const field of policyMetadata.enrichFields) {
    newMap.set(field, {
      name: field,
      type: 'number'
    });
  }
  return newMap;
}
function getFinalSuggestions({
  comma
} = {
  comma: true
}) {
  const finalSuggestions = [_complete_items.pipeCompleteItem];
  if (comma) {
    finalSuggestions.push(_complete_items.commaCompleteItem);
  }
  return finalSuggestions;
}

/**
 * This function count the number of unclosed brackets in order to
 * locally fix the queryString to generate a valid AST
 * A known limitation of this is that is not aware of commas "," or pipes "|"
 * so it is not yet helpful on a multiple commands errors (a workaround it to pass each command here...)
 * @param bracketType
 * @param text
 * @returns
 */
function countBracketsUnclosed(bracketType, text) {
  const stack = [];
  const closingBrackets = {
    '(': ')',
    '[': ']',
    '"': '"',
    '"""': '"""'
  };
  for (let i = 0; i < text.length; i++) {
    const substr = text.substring(i, i + bracketType.length);
    if (substr === closingBrackets[bracketType] && stack.length) {
      stack.pop();
    } else if (substr === bracketType) {
      stack.push(bracketType);
    }
  }
  return stack.length;
}

/**
 * This function attempts to correct the syntax of a partial query to make it valid.
 *
 * This is important because a syntactically-invalid query will not generate a good AST.
 *
 * @param _query
 * @param context
 * @returns
 */
function correctQuerySyntax(_query, context) {
  let query = _query;
  // check if all brackets are closed, otherwise close them
  const unclosedRoundBrackets = countBracketsUnclosed('(', query);
  const unclosedSquaredBrackets = countBracketsUnclosed('[', query);
  const unclosedQuotes = countBracketsUnclosed('"', query);
  const unclosedTripleQuotes = countBracketsUnclosed('"""', query);
  // if it's a comma by the user or a forced trigger by a function argument suggestion
  // add a marker to make the expression still valid
  const charThatNeedMarkers = [',', ':'];
  if (context.triggerCharacter && charThatNeedMarkers.includes(context.triggerCharacter) ||
  // monaco.editor.CompletionTriggerKind['Invoke'] === 0
  context.triggerKind === 0 && unclosedRoundBrackets === 0 || context.triggerCharacter === ' ' && ((0, _helpers.isMathFunction)(query, query.length) || (0, _helpers.isComma)(query.trimEnd()[query.trimEnd().length - 1]))) {
    query += _constants.EDITOR_MARKER;
  }

  // if there are unclosed brackets, close them
  if (unclosedRoundBrackets || unclosedSquaredBrackets || unclosedQuotes) {
    for (const [char, count] of [['"""', unclosedTripleQuotes], ['"', unclosedQuotes], [')', unclosedRoundBrackets], [']', unclosedSquaredBrackets]]) {
      if (count) {
        // inject the closing brackets
        query += Array(count).fill(char).join('');
      }
    }
  }
  return query;
}
async function suggest(fullText, offset, context, astProvider, resourceRetriever) {
  const innerText = fullText.substring(0, offset);
  const correctedQuery = correctQuerySyntax(innerText, context);
  const {
    ast
  } = await astProvider(correctedQuery);
  const astContext = (0, _context.getAstContext)(innerText, ast, offset);
  // build the correct query to fetch the list of fields
  const queryForFields = (0, _helper.getQueryForFields)((0, _resources_helpers.buildQueryUntilPreviousCommand)(ast, correctedQuery), ast);
  const {
    getFieldsByType,
    getFieldsMap
  } = getFieldsByTypeRetriever(queryForFields, resourceRetriever);
  const getSources = getSourcesRetriever(resourceRetriever);
  const getDatastreamsForIntegration = getDatastreamsForIntegrationRetriever(resourceRetriever);
  const {
    getPolicies,
    getPolicyMetadata
  } = getPolicyRetriever(resourceRetriever);
  if (astContext.type === 'newCommand') {
    // propose main commands here
    // filter source commands if already defined
    const suggestions = _complete_items.commandAutocompleteDefinitions;
    if (!ast.length) {
      return suggestions.filter(_helpers.isSourceCommand);
    }
    return suggestions.filter(def => !(0, _helpers.isSourceCommand)(def));
  }
  if (astContext.type === 'expression') {
    // suggest next possible argument, or option
    // otherwise a variable
    return getExpressionSuggestionsByType(innerText, ast, astContext, getSources, getDatastreamsForIntegration, getFieldsByType, getFieldsMap, getPolicies, getPolicyMetadata);
  }
  if (astContext.type === 'setting') {
    return getSettingArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  if (astContext.type === 'option') {
    // need this wrap/unwrap thing to make TS happy
    const {
      option,
      ...rest
    } = astContext;
    if (option && (0, _helpers.isOptionItem)(option)) {
      return getOptionArgsSuggestions(innerText, ast, {
        option,
        ...rest
      }, getFieldsByType, getFieldsMap, getPolicyMetadata);
    }
  }
  if (astContext.type === 'function') {
    return getFunctionArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  if (astContext.type === 'list') {
    return getListArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  return [];
}
function getFieldsByTypeRetriever(queryString, resourceRetriever) {
  const helpers = (0, _resources_helpers.getFieldsByTypeHelper)(queryString, resourceRetriever);
  return {
    getFieldsByType: async (expectedType = 'any', ignored = []) => {
      const fields = await helpers.getFieldsByType(expectedType, ignored);
      return (0, _factories.buildFieldsDefinitions)(fields);
    },
    getFieldsMap: helpers.getFieldsMap
  };
}
function getPolicyRetriever(resourceRetriever) {
  const helpers = (0, _resources_helpers.getPolicyHelper)(resourceRetriever);
  return {
    getPolicies: async () => {
      const policies = await helpers.getPolicies();
      return (0, _factories.buildPoliciesDefinitions)(policies);
    },
    getPolicyMetadata: helpers.getPolicyMetadata
  };
}
function getSourcesRetriever(resourceRetriever) {
  const helper = (0, _resources_helpers.getSourcesHelper)(resourceRetriever);
  return async () => {
    const list = (await helper()) || [];
    // hide indexes that start with .
    return (0, _factories.buildSourcesDefinitions)(list.filter(({
      hidden
    }) => !hidden).map(({
      name,
      dataStreams,
      title
    }) => {
      return {
        name,
        isIntegration: Boolean(dataStreams && dataStreams.length),
        title
      };
    }));
  };
}
function getDatastreamsForIntegrationRetriever(resourceRetriever) {
  const helper = (0, _resources_helpers.getSourcesHelper)(resourceRetriever);
  return async sourceName => {
    var _list$find;
    const list = (await helper()) || [];
    return (_list$find = list.find(({
      name
    }) => name === sourceName)) === null || _list$find === void 0 ? void 0 : _list$find.dataStreams;
  };
}
function findNewVariable(variables) {
  let autoGeneratedVariableCounter = 0;
  let name = `var${autoGeneratedVariableCounter++}`;
  while (variables.has(name)) {
    name = `var${autoGeneratedVariableCounter++}`;
  }
  return name;
}
function workoutBuiltinOptions(nodeArg, references) {
  // skip assign operator if it's a function or an existing field to avoid promoting shadowing
  return {
    skipAssign: Boolean(!(0, _helpers.isColumnItem)(nodeArg) || (0, _helpers.lookupColumn)(nodeArg, references))
  };
}
function areCurrentArgsValid(command, node, references) {
  // unfortunately here we need to bake some command-specific logic
  if (command.name === 'stats') {
    if (node) {
      // consider the following expressions not complete yet
      // ... | stats a
      // ... | stats a =
      if ((0, _helpers.isColumnItem)(node) || (0, _helpers.isAssignment)(node) && !(0, _helpers.isAssignmentComplete)(node)) {
        return false;
      }
    }
  }
  if (command.name === 'eval') {
    if (node) {
      if ((0, _helpers.isFunctionItem)(node)) {
        if ((0, _helpers.isAssignment)(node)) {
          return (0, _helpers.isAssignmentComplete)(node);
        } else {
          return isFunctionArgComplete(node, references).complete;
        }
      }
    }
  }
  if (command.name === 'where') {
    if (node) {
      if ((0, _helpers.isColumnItem)(node) || (0, _helpers.isFunctionItem)(node) && !isFunctionArgComplete(node, references).complete) {
        return false;
      } else {
        return extractFinalTypeFromArg(node, references) === (0, _helpers.getCommandDefinition)(command.name).signature.params[0].type;
      }
    }
  }
  if (command.name === 'rename') {
    if (node) {
      if ((0, _helpers.isColumnItem)(node)) {
        return true;
      }
    }
  }
  return true;
}
function extractFinalTypeFromArg(arg, references) {
  if (Array.isArray(arg)) {
    return extractFinalTypeFromArg(arg[0], references);
  }
  if ((0, _helpers.isColumnItem)(arg) || (0, _helpers.isLiteralItem)(arg)) {
    if ((0, _helpers.isLiteralItem)(arg)) {
      return arg.literalType;
    }
    if ((0, _helpers.isColumnItem)(arg)) {
      const hit = (0, _helpers.lookupColumn)(arg, references);
      if (hit) {
        return hit.type;
      }
    }
  }
  if ((0, _helpers.isTimeIntervalItem)(arg)) {
    return arg.type;
  }
  if ((0, _helpers.isFunctionItem)(arg)) {
    const fnDef = (0, _helpers.getFunctionDefinition)(arg.name);
    if (fnDef) {
      // @TODO: improve this to better filter down the correct return type based on existing arguments
      // just mind that this can be highly recursive...
      return fnDef.signatures[0].returnType;
    }
  }
}

// @TODO: refactor this to be shared with validation
function isFunctionArgComplete(arg, references) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(arg.name);
  if (!fnDefinition) {
    return {
      complete: false
    };
  }
  const cleanedArgs = (0, _context.removeMarkerArgFromArgsList)(arg).args;
  const argLengthCheck = fnDefinition.signatures.some(def => {
    if (def.minParams && cleanedArgs.length >= def.minParams) {
      return true;
    }
    if (cleanedArgs.length === def.params.length) {
      return true;
    }
    return cleanedArgs.length >= def.params.filter(({
      optional
    }) => !optional).length;
  });
  if (!argLengthCheck) {
    return {
      complete: false,
      reason: 'fewArgs'
    };
  }
  if (fnDefinition.name === 'in' && Array.isArray(arg.args[1]) && !arg.args[1].length) {
    return {
      complete: false,
      reason: 'fewArgs'
    };
  }
  const hasCorrectTypes = fnDefinition.signatures.some(def => {
    return arg.args.every((a, index) => {
      return def.params[index].type === extractFinalTypeFromArg(a, references);
    });
  });
  if (!hasCorrectTypes) {
    return {
      complete: false,
      reason: 'wrongTypes'
    };
  }
  return {
    complete: true
  };
}
function extractArgMeta(commandOrOption, node) {
  let argIndex = commandOrOption.args.length;
  const prevIndex = Math.max(argIndex - 1, 0);
  const lastArg = (0, _context.removeMarkerArgFromArgsList)(commandOrOption).args[prevIndex];
  if ((0, _helpers.isIncompleteItem)(lastArg)) {
    argIndex = prevIndex;
  }

  // if a node is not specified use the lastArg
  // mind to give priority to node as lastArg might be a function root
  // => "a > b and c == d" gets translated into and( gt(a, b) , eq(c, d) ) => hence "and" is lastArg
  const nodeArg = node || lastArg;
  return {
    argIndex,
    prevIndex,
    lastArg,
    nodeArg
  };
}
async function getExpressionSuggestionsByType(innerText, commands, {
  command,
  option,
  node
}, getSources, getDatastreamsForIntegration, getFieldsByType, getFieldsMap, getPolicies, getPolicyMetadata) {
  var _argDef;
  const commandDef = (0, _helpers.getCommandDefinition)(command.name);
  const {
    argIndex,
    prevIndex,
    lastArg,
    nodeArg
  } = extractArgMeta(command, node);

  // TODO - this is a workaround because it was too difficult to handle this case in a generic way :(
  if (commandDef.name === 'from' && node && (0, _helpers.isSourceItem)(node) && /\s/.test(node.name)) {
    // FROM " <suggest>"
    return [];
  }

  // A new expression is considered either
  // * just after a command name => i.e. ... | STATS <here>
  // * or after a comma => i.e. STATS fieldA, <here>
  const isNewExpression = (0, _helpers.isRestartingExpression)(innerText) || argIndex === 0 && (!(0, _helpers.isFunctionItem)(nodeArg) || !(nodeArg !== null && nodeArg !== void 0 && nodeArg.args.length));

  // the not function is a special operator that can be used in different ways,
  // and not all these are mapped within the AST data structure: in particular
  // <COMMAND> <field> NOT <here>
  // is an incomplete statement and it results in a missing AST node, so we need to detect
  // from the query string itself
  const endsWithNot = / not$/i.test(innerText.trimEnd()) && !command.args.some(arg => (0, _helpers.isFunctionItem)(arg) && arg.name === 'not');

  // early exit in case of a missing function
  if ((0, _helpers.isFunctionItem)(lastArg) && !(0, _helpers.getFunctionDefinition)(lastArg.name)) {
    return [];
  }

  // Are options already declared? This is useful to suggest only new ones
  const optionsAlreadyDeclared = command.args.filter(arg => (0, _helpers.isOptionItem)(arg)).map(({
    name
  }) => ({
    name,
    index: commandDef.options.findIndex(({
      name: defName
    }) => defName === name)
  }));
  const optionsAvailable = commandDef.options.filter(({
    name
  }, index) => {
    const optArg = optionsAlreadyDeclared.find(({
      name: optionName
    }) => optionName === name);
    return !optArg && !optionsAlreadyDeclared.length || optArg && index > optArg.index;
  });
  // get the next definition for the given command
  let argDef = commandDef.signature.params[argIndex];
  // tune it for the variadic case
  if (!argDef) {
    // this is the case of a comma argument
    if (commandDef.signature.multipleParams) {
      if (isNewExpression || (0, _helpers.isAssignment)(lastArg) && !(0, _helpers.isAssignmentComplete)(lastArg)) {
        // i.e. ... | <COMMAND> a, <here>
        // i.e. ... | <COMMAND> a = ..., b = <here>
        argDef = commandDef.signature.params[0];
      }
    }

    // this is the case where there's an argument, but it's of the wrong type
    // i.e. ... | WHERE numberField <here> (WHERE wants a boolean expression!)
    // i.e. ... | STATS numberfield <here> (STATS wants a function expression!)
    if (!isNewExpression && nodeArg && !Array.isArray(nodeArg)) {
      const prevArg = commandDef.signature.params[prevIndex];
      // in some cases we do not want to go back as the command only accepts a literal
      // i.e. LIMIT 5 <suggest> -> that's it, so no argDef should be assigned

      // make an exception for STATS (STATS is the only command who accept a function type as arg)
      if (prevArg && (prevArg.type === 'function' || !Array.isArray(nodeArg) && prevArg.type !== nodeArg.type)) {
        if (!(0, _helpers.isLiteralItem)(nodeArg) || !prevArg.constantOnly) {
          argDef = prevArg;
        }
      }
    }
  }

  // collect all fields + variables to suggest
  const fieldsMap = await (argDef ? getFieldsMap() : new Map());
  const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);

  // enrich with assignment has some special rules who are handled somewhere else
  const canHaveAssignments = ['eval', 'stats', 'row'].includes(command.name);
  const references = {
    fields: fieldsMap,
    variables: anyVariables
  };
  const suggestions = [];

  // in this flow there's a clear plan here from argument definitions so try to follow it
  if (argDef) {
    if (argDef.type === 'column' || argDef.type === 'any' || argDef.type === 'function') {
      if (isNewExpression && canHaveAssignments) {
        if (endsWithNot) {
          // i.e.
          // ... | ROW field NOT <suggest>
          // ... | EVAL field NOT <suggest>
          // there's not way to know the type of the field here, so suggest anything
          suggestions.push(...(0, _complete_items.getNextTokenForNot)(command.name, option === null || option === void 0 ? void 0 : option.name, 'any'));
        } else {
          // i.e.
          // ... | ROW <suggest>
          // ... | STATS <suggest>
          // ... | STATS ..., <suggest>
          // ... | EVAL <suggest>
          // ... | EVAL ..., <suggest>
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyVariables)));
        }
      }
    }
    // Suggest fields or variables
    if (argDef.type === 'column' || argDef.type === 'any') {
      // ... | <COMMAND> <suggest>
      if ((!nodeArg || isNewExpression) && !endsWithNot) {
        suggestions.push(...(await getFieldsOrFunctionsSuggestions([argDef.innerType || 'any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          // TODO instead of relying on canHaveAssignments and other command name checks
          // we should have a more generic way to determine if a command can have functions.
          // I think it comes down to the definition of 'column' since 'any' should always
          // include functions.
          functions: canHaveAssignments || command.name === 'sort',
          fields: !argDef.constantOnly,
          variables: anyVariables,
          literals: argDef.constantOnly
        }, {
          ignoreFields: isNewExpression ? command.args.filter(_helpers.isColumnItem).map(({
            name
          }) => name) : []
        })));
      }
    }
    if (argDef.type === 'function' || argDef.type === 'any') {
      if ((0, _helpers.isColumnItem)(nodeArg)) {
        // ... | STATS a <suggest>
        // ... | EVAL a <suggest>
        const nodeArgType = extractFinalTypeFromArg(nodeArg, references);
        if (nodeArgType) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nodeArgType, undefined, workoutBuiltinOptions(nodeArg, references)));
        } else {
          suggestions.push((0, _complete_items.getAssignmentDefinitionCompletitionItem)());
        }
      }
      if (isNewExpression && !endsWithNot || (0, _helpers.isAssignment)(nodeArg) && !(0, _helpers.isAssignmentComplete)(nodeArg)) {
        // ... | STATS a = <suggest>
        // ... | EVAL a = <suggest>
        // ... | STATS a = ..., <suggest>
        // ... | EVAL a = ..., <suggest>
        // ... | STATS a = ..., b = <suggest>
        // ... | EVAL a = ..., b = <suggest>
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(['any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          functions: true,
          fields: false,
          variables: nodeArg ? undefined : anyVariables,
          literals: argDef.constantOnly
        })));
        if (['show', 'meta'].includes(command.name)) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, 'any'));
        }
      }
    }
    if (argDef.type === 'any') {
      // ... | EVAL var = field <suggest>
      // ... | EVAL var = fn(field) <suggest>
      // make sure we're still in the same assignment context and there's no comma (newExpression ensures that)
      if (!isNewExpression) {
        if ((0, _helpers.isAssignment)(nodeArg) && (0, _helpers.isAssignmentComplete)(nodeArg)) {
          const [rightArg] = nodeArg.args[1];
          const nodeArgType = extractFinalTypeFromArg(rightArg, references);
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nodeArgType || 'any', undefined, workoutBuiltinOptions(rightArg, references)));
          if (nodeArgType === 'number' && (0, _helpers.isLiteralItem)(rightArg)) {
            // ... EVAL var = 1 <suggest>
            suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
          }
          if ((0, _helpers.isFunctionItem)(rightArg)) {
            if (rightArg.args.some(_helpers.isTimeIntervalItem)) {
              const lastFnArg = rightArg.args[rightArg.args.length - 1];
              const lastFnArgType = extractFinalTypeFromArg(lastFnArg, references);
              if (lastFnArgType === 'number' && (0, _helpers.isLiteralItem)(lastFnArg))
                // ... EVAL var = 1 year + 2 <suggest>
                suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
            }
          }
        } else {
          if ((0, _helpers.isFunctionItem)(nodeArg)) {
            if (nodeArg.name === 'not') {
              suggestions.push(...(await getFieldsOrFunctionsSuggestions(['boolean'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
                functions: true,
                fields: true,
                variables: anyVariables
              })));
            } else {
              const nodeArgType = extractFinalTypeFromArg(nodeArg, references);
              suggestions.push(...(await getBuiltinFunctionNextArgument(command, option, argDef, nodeArg, nodeArgType || 'any', references, getFieldsByType)));
              if (nodeArg.args.some(_helpers.isTimeIntervalItem)) {
                const lastFnArg = nodeArg.args[nodeArg.args.length - 1];
                const lastFnArgType = extractFinalTypeFromArg(lastFnArg, references);
                if (lastFnArgType === 'number' && (0, _helpers.isLiteralItem)(lastFnArg))
                  // ... EVAL var = 1 year + 2 <suggest>
                  suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, ['time_literal_unit']));
              }
            }
          }
        }
      }
    }

    // if the definition includes a list of constants, suggest them
    if (argDef.values) {
      // ... | <COMMAND> ... <suggest enums>
      suggestions.push(...(0, _factories.buildConstantsDefinitions)(argDef.values));
    }
    // If the type is specified try to dig deeper in the definition to suggest the best candidate
    if (['string', 'number', 'boolean'].includes(argDef.type) && !argDef.values) {
      // it can be just literal values (i.e. "string")
      if (argDef.constantOnly) {
        // ... | <COMMAND> ... <suggest>
        suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, [argDef.type], [argDef.name]));
      } else {
        // or it can be anything else as long as it is of the right type and the end (i.e. column or function)
        if (!nodeArg) {
          if (endsWithNot) {
            // i.e.
            // ... | WHERE field NOT <suggest>
            // there's not way to know the type of the field here, so suggest anything
            suggestions.push(...(0, _complete_items.getNextTokenForNot)(command.name, option === null || option === void 0 ? void 0 : option.name, 'any'));
          } else {
            // ... | <COMMAND> <suggest>
            // In this case start suggesting something not strictly based on type
            suggestions.push(...(await getFieldsOrFunctionsSuggestions(['any'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
              functions: true,
              fields: true,
              variables: anyVariables
            })));
          }
        } else {
          // if something is already present, leverage its type to suggest something in context
          const nodeArgType = extractFinalTypeFromArg(nodeArg, references);
          // These cases can happen here, so need to identify each and provide the right suggestion
          // i.e. ... | <COMMAND> field <suggest>
          // i.e. ... | <COMMAND> field + <suggest>
          // i.e. ... | <COMMAND> field >= <suggest>
          // i.e. ... | <COMMAND> field > 0 <suggest>
          // i.e. ... | <COMMAND> field + otherN <suggest>

          if (nodeArgType) {
            if ((0, _helpers.isFunctionItem)(nodeArg)) {
              if (nodeArg.name === 'not') {
                suggestions.push(...(await getFieldsOrFunctionsSuggestions(['boolean'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
                  functions: true,
                  fields: true,
                  variables: anyVariables
                })));
              } else {
                suggestions.push(...(await getBuiltinFunctionNextArgument(command, option, argDef, nodeArg, nodeArgType, references, getFieldsByType)));
              }
            } else {
              // i.e. ... | <COMMAND> field <suggest>
              suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nodeArgType, undefined, workoutBuiltinOptions(nodeArg, references)));
            }
          }
        }
      }
    }
    if (argDef.type === 'source') {
      if (argDef.innerType === 'policy') {
        // ... | ENRICH <suggest>
        const policies = await getPolicies();
        suggestions.push(...(policies.length ? policies : [(0, _factories.buildNoPoliciesAvailableDefinition)()]));
      } else {
        const index = (0, _helper.getSourcesFromCommands)(commands, 'index');
        const canRemoveQuote = isNewExpression && innerText.includes('"');
        // Function to add suggestions based on canRemoveQuote
        const addSuggestionsBasedOnQuote = async definitions => {
          suggestions.push(...(canRemoveQuote ? (0, _helper.removeQuoteForSuggestedSources)(definitions) : definitions));
        };
        if (index && index.text && index.text !== _constants.EDITOR_MARKER) {
          const source = index.text.replace(_constants.EDITOR_MARKER, '');
          const dataStreams = await getDatastreamsForIntegration(source);
          if (dataStreams) {
            // Integration name, suggest the datastreams
            await addSuggestionsBasedOnQuote((0, _factories.buildSourcesDefinitions)(dataStreams.map(({
              name
            }) => ({
              name,
              isIntegration: false
            }))));
          } else {
            // Not an integration, just a partial source name
            await addSuggestionsBasedOnQuote(await getSources());
          }
        } else {
          // FROM <suggest> or no index/text
          await addSuggestionsBasedOnQuote(await getSources());
        }
      }
    }
  }
  const nonOptionArgs = command.args.filter(arg => !(0, _helpers.isOptionItem)(arg) && !(0, _helpers.isSettingItem)(arg) && !Array.isArray(arg) && !arg.incomplete);
  // Perform some checks on mandatory arguments
  const mandatoryArgsAlreadyPresent = commandDef.signature.multipleParams && nonOptionArgs.length > 1 || nonOptionArgs.length >= commandDef.signature.params.filter(({
    optional
  }) => !optional).length || ((_argDef = argDef) === null || _argDef === void 0 ? void 0 : _argDef.type) === 'function';

  // check if declared args are fully valid for the given command
  const currentArgsAreValidForCommand = areCurrentArgsValid(command, nodeArg, references);

  // latest suggestions: options and final ones
  if (!isNewExpression && mandatoryArgsAlreadyPresent && currentArgsAreValidForCommand || optionsAlreadyDeclared.length) {
    // suggest some command options
    if (optionsAvailable.length) {
      suggestions.push(...optionsAvailable.map(opt => (0, _factories.buildOptionDefinition)(opt, command.name === 'dissect')));
    }
    if (!optionsAvailable.length || optionsAvailable.every(({
      optional
    }) => optional)) {
      const shouldPushItDown = command.name === 'eval' && !command.args.some(_helpers.isFunctionItem);
      // now suggest pipe or comma
      const finalSuggestions = getFinalSuggestions({
        comma: commandDef.signature.multipleParams && optionsAvailable.length === commandDef.options.length
      }).map(({
        sortText,
        ...rest
      }) => ({
        ...rest,
        sortText: shouldPushItDown ? `Z${sortText}` : sortText
      }));
      suggestions.push(...finalSuggestions);
    }
  }
  // Due to some logic overlapping functions can be repeated
  // so dedupe here based on text string (it can differ from name)
  return (0, _uniqBy.default)(suggestions, suggestion => suggestion.text);
}
async function getBuiltinFunctionNextArgument(command, option, argDef, nodeArg, nodeArgType, references, getFieldsByType) {
  const suggestions = [];
  const isFnComplete = isFunctionArgComplete(nodeArg, references);
  if (isFnComplete.complete) {
    // i.e. ... | <COMMAND> field > 0 <suggest>
    // i.e. ... | <COMMAND> field + otherN <suggest>
    suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, option === null || option === void 0 ? void 0 : option.name, nodeArgType || 'any', undefined, workoutBuiltinOptions(nodeArg, references)));
  } else {
    // i.e. ... | <COMMAND> field >= <suggest>
    // i.e. ... | <COMMAND> field + <suggest>
    // i.e. ... | <COMMAND> field and <suggest>

    // Because it's an incomplete function, need to extract the type of the current argument
    // and suggest the next argument based on types

    // pick the last arg and check its type to verify whether is incomplete for the given function
    const cleanedArgs = (0, _context.removeMarkerArgFromArgsList)(nodeArg).args;
    const nestedType = extractFinalTypeFromArg(nodeArg.args[cleanedArgs.length - 1], references);
    if (isFnComplete.reason === 'fewArgs') {
      const fnDef = (0, _helpers.getFunctionDefinition)(nodeArg.name);
      if (fnDef !== null && fnDef !== void 0 && fnDef.signatures.every(({
        params
      }) => params.some(({
        type
      }) => (0, _helpers.isArrayType)(type)))) {
        suggestions.push(_complete_items.listCompleteItem);
      } else {
        var _getFunctionDefinitio;
        const finalType = nestedType || nodeArgType || 'any';
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(
        // this is a special case with AND/OR
        // <COMMAND> expression AND/OR <suggest>
        // technically another boolean value should be suggested, but it is a better experience
        // to actually suggest a wider set of fields/functions
        [finalType === 'boolean' && ((_getFunctionDefinitio = (0, _helpers.getFunctionDefinition)(nodeArg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === 'builtin' ? 'any' : finalType], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          functions: true,
          fields: true,
          variables: references.variables
        })));
      }
    }
    if (isFnComplete.reason === 'wrongTypes') {
      if (nestedType) {
        // suggest something to complete the builtin function
        if (nestedType !== argDef.type) {
          suggestions.push(...(0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, nestedType, [argDef.type], workoutBuiltinOptions(nodeArg, references)));
        }
      }
    }
  }
  return suggestions;
}
function pushItUpInTheList(suggestions, shouldPromote) {
  if (!shouldPromote) {
    return suggestions;
  }
  return suggestions.map(({
    sortText,
    ...rest
  }) => ({
    ...rest,
    sortText: `1${sortText}`
  }));
}

/**
 * TODO — split this into distinct functions, one for fields, one for functions, one for literals
 */
async function getFieldsOrFunctionsSuggestions(types, commandName, optionName, getFieldsByType, {
  functions,
  fields,
  variables,
  literals = false
}, {
  ignoreFn = [],
  ignoreFields = []
} = {}) {
  const filteredFieldsByType = pushItUpInTheList(await (fields ? getFieldsByType(types, ignoreFields) : []), functions);
  const filteredVariablesByType = [];
  if (variables) {
    for (const variable of variables.values()) {
      if (types.includes('any') || types.includes(variable[0].type)) {
        filteredVariablesByType.push(variable[0].name);
      }
    }
    // due to a bug on the ES|QL table side, filter out fields list with underscored variable names (??)
    // avg( numberField ) => avg_numberField_
    const ALPHANUMERIC_REGEXP = /[^a-zA-Z\d]/g;
    if (filteredVariablesByType.length && filteredVariablesByType.some(v => ALPHANUMERIC_REGEXP.test(v))) {
      for (const variable of filteredVariablesByType) {
        // remove backticks if present
        const sanitizedVariable = variable.startsWith(_constants.SINGLE_BACKTICK) ? variable.slice(1, variable.length - 1) : variable;
        const underscoredName = sanitizedVariable.replace(ALPHANUMERIC_REGEXP, '_');
        const index = filteredFieldsByType.findIndex(({
          label
        }) => underscoredName === label || `_${underscoredName}_` === label);
        if (index >= 0) {
          filteredFieldsByType.splice(index);
        }
      }
    }
  }
  const suggestions = filteredFieldsByType.concat(functions ? (0, _factories.getCompatibleFunctionDefinition)(commandName, optionName, types, ignoreFn) : [], variables ? pushItUpInTheList((0, _factories.buildVariablesDefinitions)(filteredVariablesByType), functions) : [], literals ? (0, _factories.getCompatibleLiterals)(commandName, types) : []);
  return suggestions;
}
const addCommaIf = (condition, text) => condition ? `${text},` : text;
async function getFunctionArgsSuggestions(innerText, commands, {
  command,
  option,
  node
}, getFieldsByType, getFieldsMap, getPolicyMetadata) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(node.name);
  // early exit on no hit
  if (!fnDefinition) {
    return [];
  }
  const fieldsMap = await getFieldsMap();
  const variablesExcludingCurrentCommandOnes = (0, _variables.excludeVariablesFromCurrentCommand)(commands, command, fieldsMap, innerText);
  // pick the type of the next arg
  const shouldGetNextArgument = node.text.includes(_constants.EDITOR_MARKER);
  let argIndex = Math.max(node.args.length, 0);
  if (!shouldGetNextArgument && argIndex) {
    argIndex -= 1;
  }
  const arg = node.args[argIndex];

  // the first signature is used as reference
  // TODO - take into consideration all signatures that match the current args
  const refSignature = fnDefinition.signatures[0];
  const hasMoreMandatoryArgs = refSignature.params.length >= argIndex && refSignature.params.filter(({
    optional
  }, index) => !optional && index > argIndex).length > 0 || ('minParams' in refSignature && refSignature.minParams ? refSignature.minParams - 1 > argIndex : false);
  const suggestedConstants = Array.from(new Set(fnDefinition.signatures.reduce((acc, signature) => {
    const p = signature.params[argIndex];
    if (!p) {
      return acc;
    }
    const _suggestions = p.literalSuggestions ? p.literalSuggestions : p.literalOptions ? p.literalOptions : [];
    return acc.concat(_suggestions);
  }, [])));
  if (suggestedConstants.length) {
    return (0, _factories.buildValueDefinitions)(suggestedConstants).map(suggestion => ({
      ...suggestion,
      text: addCommaIf(hasMoreMandatoryArgs && fnDefinition.type !== 'builtin', suggestion.text)
    }));
  }
  const suggestions = [];
  const noArgDefined = !arg;
  const isUnknownColumn = arg && (0, _helpers.isColumnItem)(arg) && !(0, _helpers.getColumnExists)(arg, {
    fields: fieldsMap,
    variables: variablesExcludingCurrentCommandOnes
  });
  if (noArgDefined || isUnknownColumn) {
    // ... | EVAL fn( <suggest>)
    // ... | EVAL fn( field, <suggest>)

    const commandArgIndex = command.args.findIndex(cmdArg => (0, _helpers.isSingleItem)(cmdArg) && cmdArg.location.max >= node.location.max);
    const finalCommandArgIndex = command.name !== 'stats' ? -1 : commandArgIndex < 0 ? Math.max(command.args.length - 1, 0) : commandArgIndex;
    const finalCommandArg = command.args[finalCommandArgIndex];
    const fnToIgnore = [];
    // just ignore the current function
    if (command.name !== 'stats' || (0, _helpers.isOptionItem)(finalCommandArg) && finalCommandArg.name === 'by') {
      fnToIgnore.push(node.name);
    } else {
      fnToIgnore.push(...(0, _helper.getFunctionsToIgnoreForStats)(command, finalCommandArgIndex), ...((0, _helper.isAggFunctionUsedAlready)(command, finalCommandArgIndex) ? (0, _helpers.getAllFunctions)({
        type: 'agg'
      }).map(({
        name
      }) => name) : []));
    }
    const existingTypes = node.args.map(nodeArg => extractFinalTypeFromArg(nodeArg, {
      fields: fieldsMap,
      variables: variablesExcludingCurrentCommandOnes
    })).filter(_helpers.nonNullable);
    const validSignatures = fnDefinition.signatures
    // if existing arguments are preset already, use them to filter out incompatible signatures
    .filter(signature => {
      if (existingTypes.length) {
        return existingTypes.every((type, index) => signature.params[index].type === type);
      }
      return true;
    });

    /**
     * Get all parameter definitions across all function signatures
     * for the current parameter position in the given function definition,
     */
    const allParamDefinitionsForThisPosition = validSignatures.map(signature => (0, _helper.getParamAtPosition)(signature, argIndex)).filter(_helpers.nonNullable);

    // Separate the param definitions into two groups:
    // fields should only be suggested if the param isn't constant-only,
    // and constant suggestions should only be given if it is.
    //
    // TODO - consider incorporating the literalOptions into this
    //
    // TODO — improve this to inherit the constant flag from the outer function
    // (e.g. if func1's first parameter is constant-only, any nested functions should
    // inherit that constraint: func1(func2(shouldBeConstantOnly)))
    //
    const [constantOnlyParamDefs, paramDefsWhichSupportFields] = (0, _lodash.partition)(allParamDefinitionsForThisPosition, paramDef => paramDef.constantOnly || /_literal$/.test(paramDef.type));
    const getTypesFromParamDefs = paramDefs => {
      return Array.from(new Set(paramDefs.map(({
        type
      }) => type)));
    };
    suggestions.push(...(0, _factories.getCompatibleLiterals)(command.name, getTypesFromParamDefs(constantOnlyParamDefs)));
    suggestions.push(...(await getFieldsOrFunctionsSuggestions(getTypesFromParamDefs(paramDefsWhichSupportFields), command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
      functions: true,
      fields: true,
      variables: variablesExcludingCurrentCommandOnes
    },
    // do not repropose the same function as arg
    // i.e. avoid cases like abs(abs(abs(...))) with suggestions
    {
      ignoreFn: fnToIgnore
    })));
  }

  // for eval and row commands try also to complete numeric literals with time intervals where possible
  if (arg) {
    if (command.name !== 'stats') {
      if ((0, _helpers.isLiteralItem)(arg) && arg.literalType === 'number') {
        // ... | EVAL fn(2 <suggest>)
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(['time_literal_unit'], command.name, option === null || option === void 0 ? void 0 : option.name, getFieldsByType, {
          functions: false,
          fields: false,
          variables: variablesExcludingCurrentCommandOnes,
          literals: true
        })));
      }
    }
    if (hasMoreMandatoryArgs) {
      // suggest a comma if there's another argument for the function
      suggestions.push(_complete_items.commaCompleteItem);
    }
    // if there are other arguments in the function, inject automatically a comma after each suggestion
    return suggestions.map(suggestion => suggestion !== _complete_items.commaCompleteItem ? {
      ...suggestion,
      text: hasMoreMandatoryArgs && fnDefinition.type !== 'builtin' ? `${suggestion.text},` : suggestion.text
    } : suggestion);
  }
  return suggestions.map(({
    text,
    ...rest
  }) => ({
    ...rest,
    text: addCommaIf(hasMoreMandatoryArgs && fnDefinition.type !== 'builtin', text)
  }));
}
async function getListArgsSuggestions(innerText, commands, {
  command,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const suggestions = [];
  // node is supposed to be the function who support a list argument (like the "in" operator)
  // so extract the type of the first argument and suggest fields of that type
  if (node && (0, _helpers.isFunctionItem)(node)) {
    const fieldsMap = await getFieldsMaps();
    const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
    // extract the current node from the variables inferred
    anyVariables.forEach((values, key) => {
      if (values.some(v => v.location === node.location)) {
        anyVariables.delete(key);
      }
    });
    const [firstArg] = node.args;
    if ((0, _helpers.isColumnItem)(firstArg)) {
      const argType = extractFinalTypeFromArg(firstArg, {
        fields: fieldsMap,
        variables: anyVariables
      });
      if (argType) {
        // do not propose existing columns again
        const otherArgs = node.args.filter(Array.isArray).flat().filter(_helpers.isColumnItem);
        suggestions.push(...(await getFieldsOrFunctionsSuggestions([argType], command.name, undefined, getFieldsByType, {
          functions: true,
          fields: true,
          variables: anyVariables
        }, {
          ignoreFields: [firstArg.name, ...otherArgs.map(({
            name
          }) => name)]
        })));
      }
    }
  }
  return suggestions;
}
async function getSettingArgsSuggestions(innerText, commands, {
  command,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const suggestions = [];
  const settingDefs = (0, _helpers.getCommandDefinition)(command.name).modes;
  if (settingDefs.length) {
    const lastChar = (0, _helpers.getLastCharFromTrimmed)(innerText);
    const matchingSettingDefs = settingDefs.filter(({
      prefix
    }) => lastChar === prefix);
    if (matchingSettingDefs.length) {
      // COMMAND _<here>
      suggestions.push(...matchingSettingDefs.flatMap(_factories.buildSettingDefinitions));
    }
  }
  return suggestions;
}
async function getOptionArgsSuggestions(innerText, commands, {
  command,
  option,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const optionDef = (0, _helpers.getCommandOption)(option.name);
  const {
    nodeArg,
    argIndex,
    lastArg
  } = extractArgMeta(option, node);
  const suggestions = [];
  const isNewExpression = (0, _helpers.isRestartingExpression)(innerText) || option.args.length === 0;
  const fieldsMap = await getFieldsMaps();
  const anyVariables = (0, _variables.collectVariables)(commands, fieldsMap, innerText);
  const references = {
    fields: fieldsMap,
    variables: anyVariables
  };
  if (command.name === 'enrich') {
    if (option.name === 'on') {
      // if it's a new expression, suggest fields to match on
      if (isNewExpression || (0, _helpers.findPreviousWord)(innerText) === 'ON' || option && (0, _helpers.isAssignment)(option.args[0]) && !option.args[1]) {
        const policyName = (0, _helpers.isSourceItem)(command.args[0]) ? command.args[0].name : undefined;
        if (policyName) {
          const policyMetadata = await getPolicyMetadata(policyName);
          if (policyMetadata) {
            suggestions.push(...(0, _factories.buildMatchingFieldsDefinition)(policyMetadata.matchField, Array.from(fieldsMap.keys())));
          }
        }
      } else {
        // propose the with option
        suggestions.push((0, _factories.buildOptionDefinition)((0, _helpers.getCommandOption)('with')), ...getFinalSuggestions({
          comma: true
        }));
      }
    }
    if (option.name === 'with') {
      const policyName = (0, _helpers.isSourceItem)(command.args[0]) ? command.args[0].name : undefined;
      if (policyName) {
        const policyMetadata = await getPolicyMetadata(policyName);
        const anyEnhancedVariables = (0, _variables.collectVariables)(commands, appendEnrichFields(fieldsMap, policyMetadata), innerText);
        if (isNewExpression || (0, _helpers.findPreviousWord)(innerText) === 'WITH') {
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyEnhancedVariables)));
        }

        // make sure to remove the marker arg from the assign fn
        const assignFn = (0, _helpers.isAssignment)(lastArg) ? (0, _context.removeMarkerArgFromArgsList)(lastArg) : undefined;
        if (policyMetadata) {
          if (isNewExpression || assignFn && !(0, _helpers.isAssignmentComplete)(assignFn)) {
            // ... | ENRICH ... WITH a =
            suggestions.push(...(0, _factories.buildFieldsDefinitions)(policyMetadata.enrichFields));
          }
        }
        if (assignFn && hasSameArgBothSides(assignFn) && !isNewExpression && !(0, _helpers.isIncompleteItem)(assignFn)) {
          // ... | ENRICH ... WITH a
          // effectively only assign will apper
          suggestions.push(...pushItUpInTheList((0, _complete_items.getBuiltinCompatibleFunctionDefinition)(command.name, undefined, 'any'), true));
        }
        if (assignFn && ((0, _helpers.isAssignmentComplete)(assignFn) || hasSameArgBothSides(assignFn)) && !isNewExpression) {
          suggestions.push(...getFinalSuggestions({
            comma: true
          }));
        }
      }
    }
  }
  if (command.name === 'rename') {
    if (option.args.length < 2) {
      suggestions.push(...(0, _factories.buildVariablesDefinitions)([findNewVariable(anyVariables)]));
    }
  }
  if (command.name === 'dissect') {
    if (option.args.filter(arg => !((0, _helpers.isSingleItem)(arg) && arg.type === 'unknown')).length < 1 && optionDef) {
      suggestions.push(_complete_items.colonCompleteItem, _complete_items.semiColonCompleteItem);
    }
  }
  if (option.name === 'metadata') {
    const existingFields = new Set(option.args.filter(_helpers.isColumnItem).map(({
      name
    }) => name));
    const filteredMetaFields = _constants.METADATA_FIELDS.filter(name => !existingFields.has(name));
    if (isNewExpression) {
      suggestions.push(...(0, _factories.buildFieldsDefinitions)(filteredMetaFields));
    } else if (existingFields.size > 0) {
      if (filteredMetaFields.length > 0) {
        suggestions.push(_complete_items.commaCompleteItem);
      }
      suggestions.push(_complete_items.pipeCompleteItem);
    }
  }
  if (command.name === 'stats') {
    suggestions.push(...(await getFieldsOrFunctionsSuggestions(['column'], command.name, option.name, getFieldsByType, {
      functions: false,
      fields: true
    })));
    const argDef = optionDef === null || optionDef === void 0 ? void 0 : optionDef.signature.params[argIndex];
    const nodeArgType = extractFinalTypeFromArg(nodeArg, references);
    // These cases can happen here, so need to identify each and provide the right suggestion
    // i.e. ... | STATS ... BY field + <suggest>
    // i.e. ... | STATS ... BY field >= <suggest>

    if (nodeArgType) {
      if ((0, _helpers.isFunctionItem)(nodeArg) && !isFunctionArgComplete(nodeArg, references).complete) {
        suggestions.push(...(await getBuiltinFunctionNextArgument(command, option, {
          type: (argDef === null || argDef === void 0 ? void 0 : argDef.type) || 'any'
        }, nodeArg, nodeArgType, references, getFieldsByType)));
      }
    }
  }
  if (optionDef) {
    if (!suggestions.length) {
      const argDefIndex = optionDef.signature.multipleParams ? 0 : Math.max(option.args.length - 1, 0);
      const types = [optionDef.signature.params[argDefIndex].type].filter(_helpers.nonNullable);
      // If it's a complete expression then proposed some final suggestions
      // A complete expression is either a function or a column: <COMMAND> <OPTION> field <here>
      // Or an assignment complete: <COMMAND> <OPTION> field = ... <here>
      if (option.args.length && !isNewExpression && !(0, _helpers.isAssignment)(lastArg) || (0, _helpers.isAssignment)(lastArg) && (0, _helpers.isAssignmentComplete)(lastArg)) {
        suggestions.push(...getFinalSuggestions({
          comma: optionDef.signature.multipleParams
        }));
      } else if (isNewExpression || (0, _helpers.isAssignment)(nodeArg) && !(0, _helpers.isAssignmentComplete)(nodeArg)) {
        // Otherwise try to complete the expression suggesting some columns
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(types[0] === 'column' ? ['any'] : types, command.name, option.name, getFieldsByType, {
          functions: option.name === 'by',
          fields: true
        })));
        if (command.name === 'stats' && isNewExpression) {
          suggestions.push((0, _factories.buildNewVarDefinition)(findNewVariable(anyVariables)));
        }
      }
    }
  }
  return suggestions;
}