"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatESQLColumns = formatESQLColumns;
exports.getESQLQueryColumns = getESQLQueryColumns;
exports.getESQLQueryColumnsRaw = getESQLQueryColumnsRaw;
exports.getESQLResults = getESQLResults;
var _i18n = require("@kbn/i18n");
var _fieldTypes = require("@kbn/field-types");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function formatESQLColumns(columns) {
  return columns.map(({
    name,
    type
  }) => {
    const kibanaType = (0, _fieldTypes.esFieldTypeToKibanaFieldType)(type);
    return {
      id: name,
      name,
      meta: {
        type: kibanaType,
        esType: type
      }
    };
  });
}

// Returns the columns exactly as being returned by the _query endpoint
// Based on the search api from the data plugin
async function getESQLQueryColumnsRaw({
  esqlQuery,
  search,
  signal
}) {
  try {
    var _columns;
    const response = await (0, _rxjs.lastValueFrom)(search({
      params: {
        query: `${esqlQuery} | limit 0`
      }
    }, {
      abortSignal: signal,
      strategy: 'esql_async'
    }));
    return (_columns = response.rawResponse.columns) !== null && _columns !== void 0 ? _columns : [];
  } catch (error) {
    throw new Error(_i18n.i18n.translate('esqlUtils.columnsErrorMsg', {
      defaultMessage: 'Unable to load columns. {errorMessage}',
      values: {
        errorMessage: error.message
      }
    }));
  }
}

// Returns the columns with the kibana format
// Based on the search api from the data plugin
async function getESQLQueryColumns({
  esqlQuery,
  search,
  signal
}) {
  try {
    var _formatESQLColumns;
    const rawColumns = await getESQLQueryColumnsRaw({
      esqlQuery,
      search,
      signal
    });
    const columns = (_formatESQLColumns = formatESQLColumns(rawColumns)) !== null && _formatESQLColumns !== void 0 ? _formatESQLColumns : [];
    return columns;
  } catch (error) {
    throw new Error(_i18n.i18n.translate('esqlUtils.columnsErrorMsg', {
      defaultMessage: 'Unable to load columns. {errorMessage}',
      values: {
        errorMessage: error.message
      }
    }));
  }
}

// Returns the table as being returned by the _query endpoint
// Based on the search api from the data plugin
async function getESQLResults({
  esqlQuery,
  search,
  signal,
  filter,
  dropNullColumns
}) {
  const result = await (0, _rxjs.lastValueFrom)(search({
    params: {
      ...(filter ? {
        filter
      } : {}),
      query: esqlQuery,
      ...(dropNullColumns ? {
        dropNullColumns: true
      } : {})
    }
  }, {
    abortSignal: signal,
    strategy: 'esql_async'
  }));
  return {
    response: result.rawResponse,
    params: result.requestParams
  };
}