"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndexPatternFromESQLQuery = getIndexPatternFromESQLQuery;
exports.getLimitFromESQLQuery = getLimitFromESQLQuery;
exports.hasTransformationalCommand = hasTransformationalCommand;
exports.removeDropCommandsFromESQLQuery = removeDropCommandsFromESQLQuery;
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const DEFAULT_ESQL_LIMIT = 500;

// retrieves the index pattern from the aggregate query for ES|QL using ast parsing
function getIndexPatternFromESQLQuery(esql) {
  var _sourceCommand$args;
  const {
    ast
  } = (0, _esqlAst.getAstAndSyntaxErrors)(esql);
  const sourceCommand = ast.find(({
    name
  }) => ['from', 'metrics'].includes(name));
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  const indices = args.filter(arg => arg.sourceType === 'index');
  return indices === null || indices === void 0 ? void 0 : indices.map(index => index.text).join(',');
}

// For ES|QL we consider stats and keep transformational command
// The metrics command too but only if it aggregates
function hasTransformationalCommand(esql) {
  const transformationalCommands = ['stats', 'keep'];
  const {
    ast
  } = (0, _esqlAst.getAstAndSyntaxErrors)(esql);
  const hasAtLeastOneTransformationalCommand = transformationalCommands.some(command => ast.find(({
    name
  }) => name === command));
  if (hasAtLeastOneTransformationalCommand) {
    return true;
  }
  const metricsCommand = ast.find(({
    name
  }) => name === 'metrics');
  if (metricsCommand && 'aggregates' in metricsCommand) {
    return true;
  }
  return false;
}
function getLimitFromESQLQuery(esql) {
  const limitCommands = esql.match(new RegExp(/LIMIT\s[0-9]+/, 'ig'));
  if (!limitCommands) {
    return DEFAULT_ESQL_LIMIT;
  }
  const lastIndex = limitCommands.length - 1;
  const split = limitCommands[lastIndex].split(' ');
  return parseInt(split[1], 10);
}
function removeDropCommandsFromESQLQuery(esql) {
  const pipes = (esql || '').split('|');
  return pipes.filter(statement => !/DROP\s/i.test(statement)).join('|');
}