"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KQL_WILDCARD_SYMBOL = exports.KQL_NODE_TYPE_WILDCARD = void 0;
exports.buildNode = buildNode;
exports.hasLeadingWildcard = hasLeadingWildcard;
exports.isLoneWildcard = isLoneWildcard;
exports.isMatchAll = isMatchAll;
exports.isNode = isNode;
exports.test = test;
exports.toElasticsearchQuery = toElasticsearchQuery;
exports.toKqlExpression = toKqlExpression;
exports.toQueryStringQuery = toQueryStringQuery;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const KQL_WILDCARD_SYMBOL = exports.KQL_WILDCARD_SYMBOL = '@kuery-wildcard@';
const KQL_NODE_TYPE_WILDCARD = exports.KQL_NODE_TYPE_WILDCARD = 'wildcard';
// Copied from https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions
function escapeRegExp(str) {
  return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
}

// See https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#_reserved_characters
function escapeQueryString(str) {
  return str.replace(/[+\-=&|><!(){}[\]^"~*?:\\/]/g, '\\$&'); // $& means the whole matched string
}
function isNode(node) {
  return node.type === KQL_NODE_TYPE_WILDCARD;
}
function isMatchAll(node) {
  return node.value === KQL_WILDCARD_SYMBOL;
}
function buildNode(value) {
  // When called from the parser, all wildcard characters are replaced with a special flag (since escaped wildcards are
  // handled as normal strings). However, when invoking programmatically, callers shouldn't need to do this replacement.
  if (!value.includes(KQL_NODE_TYPE_WILDCARD) && value.includes('*')) {
    return buildNode(value.replaceAll('*', KQL_WILDCARD_SYMBOL));
  }
  return {
    type: KQL_NODE_TYPE_WILDCARD,
    value
  };
}
function test(node, str) {
  const {
    value
  } = node;
  const regex = value.split(KQL_WILDCARD_SYMBOL).map(escapeRegExp).join('[\\s\\S]*');
  const regexp = new RegExp(`^${regex}$`);
  return regexp.test(str);
}
function toElasticsearchQuery(node) {
  const {
    value
  } = node;
  return value.split(KQL_WILDCARD_SYMBOL).join('*');
}
function toQueryStringQuery(node) {
  const {
    value
  } = node;
  return value.split(KQL_WILDCARD_SYMBOL).map(escapeQueryString).join('*');
}
function isLoneWildcard({
  value
}) {
  return value.includes(KQL_WILDCARD_SYMBOL) && value.replace(KQL_WILDCARD_SYMBOL, '').length === 0;
}
function hasLeadingWildcard(node) {
  return !isLoneWildcard(node) && node.value.startsWith(KQL_WILDCARD_SYMBOL);
}
function toKqlExpression(node) {
  return toQueryStringQuery(node);
}