"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resetEntityDefinitionRoute = resetEntityDefinitionRoute;
var _configSchema = require("@kbn/config-schema");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _read_entity_definition = require("../../lib/entities/read_entity_definition");
var _stop_and_delete_transform = require("../../lib/entities/stop_and_delete_transform");
var _delete_ingest_pipeline = require("../../lib/entities/delete_ingest_pipeline");
var _delete_index = require("../../lib/entities/delete_index");
var _create_and_install_ingest_pipeline = require("../../lib/entities/create_and_install_ingest_pipeline");
var _create_and_install_transform = require("../../lib/entities/create_and_install_transform");
var _start_transform = require("../../lib/entities/start_transform");
var _entity_not_found = require("../../lib/entities/errors/entity_not_found");
var _constants_entities = require("../../../common/constants_entities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function resetEntityDefinitionRoute({
  router,
  logger,
  spaces
}) {
  router.post({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/definition/{id}/_reset`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, req, res) => {
    try {
      var _spaces$spacesService;
      const soClient = (await context.core).savedObjects.client;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const spaceId = (_spaces$spacesService = spaces === null || spaces === void 0 ? void 0 : spaces.spacesService.getSpaceId(req)) !== null && _spaces$spacesService !== void 0 ? _spaces$spacesService : 'default';
      const definition = await (0, _read_entity_definition.readEntityDefinition)(soClient, req.params.id, logger);

      // Delete the transform and ingest pipeline
      await (0, _stop_and_delete_transform.stopAndDeleteHistoryTransform)(esClient, definition, logger);
      await (0, _stop_and_delete_transform.stopAndDeleteLatestTransform)(esClient, definition, logger);
      await (0, _delete_ingest_pipeline.deleteHistoryIngestPipeline)(esClient, definition, logger);
      await (0, _delete_ingest_pipeline.deleteLatestIngestPipeline)(esClient, definition, logger);
      await (0, _delete_index.deleteIndices)(esClient, definition, logger);

      // Recreate everything
      await (0, _create_and_install_ingest_pipeline.createAndInstallHistoryIngestPipeline)(esClient, definition, logger, spaceId);
      await (0, _create_and_install_ingest_pipeline.createAndInstallLatestIngestPipeline)(esClient, definition, logger, spaceId);
      await (0, _create_and_install_transform.createAndInstallHistoryTransform)(esClient, definition, logger);
      await (0, _create_and_install_transform.createAndInstallLatestTransform)(esClient, definition, logger);
      await (0, _start_transform.startTransform)(esClient, definition, logger);
      return res.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (e) {
      if (e instanceof _entity_not_found.EntityDefinitionNotFound) {
        return res.notFound({
          body: e
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return res.customError({
          body: e,
          statusCode: 400
        });
      }
      return res.customError({
        body: e,
        statusCode: 500
      });
    }
  });
}