"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteEntityDefinitionRoute = deleteEntityDefinitionRoute;
var _configSchema = require("@kbn/config-schema");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _read_entity_definition = require("../../lib/entities/read_entity_definition");
var _entity_not_found = require("../../lib/entities/errors/entity_not_found");
var _constants_entities = require("../../../common/constants_entities");
var _uninstall_entity_definition = require("../../lib/entities/uninstall_entity_definition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function deleteEntityDefinitionRoute({
  router,
  server
}) {
  router.delete({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/definition/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, req, res) => {
    try {
      const {
        logger
      } = server;
      const soClient = (await context.core).savedObjects.client;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const definition = await (0, _read_entity_definition.readEntityDefinition)(soClient, req.params.id, logger);
      await (0, _uninstall_entity_definition.uninstallEntityDefinition)({
        definition,
        soClient,
        esClient,
        logger
      });
      return res.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (e) {
      if (e instanceof _entity_not_found.EntityDefinitionNotFound) {
        return res.notFound({
          body: e
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return res.customError({
          body: e,
          statusCode: 400
        });
      }
      return res.customError({
        body: e,
        statusCode: 500
      });
    }
  });
}