"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEntityDefinitionRoute = createEntityDefinitionRoute;
var _entitiesSchema = require("@kbn/entities-schema");
var _zodHelpers = require("@kbn/zod-helpers");
var _entity_id_conflict_error = require("../../lib/entities/errors/entity_id_conflict_error");
var _entity_security_exception = require("../../lib/entities/errors/entity_security_exception");
var _invalid_transform_error = require("../../lib/entities/errors/invalid_transform_error");
var _start_transform = require("../../lib/entities/start_transform");
var _constants_entities = require("../../../common/constants_entities");
var _install_entity_definition = require("../../lib/entities/install_entity_definition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createEntityDefinitionRoute({
  router,
  server,
  spaces
}) {
  router.post({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/definition`,
    validate: {
      body: (body, res) => {
        try {
          return res.ok(_entitiesSchema.entityDefinitionSchema.parse(body));
        } catch (e) {
          return res.badRequest((0, _zodHelpers.stringifyZodError)(e));
        }
      }
    }
  }, async (context, req, res) => {
    var _spaces$spacesService;
    const {
      logger
    } = server;
    const core = await context.core;
    const soClient = core.savedObjects.client;
    const esClient = core.elasticsearch.client.asCurrentUser;
    const spaceId = (_spaces$spacesService = spaces === null || spaces === void 0 ? void 0 : spaces.spacesService.getSpaceId(req)) !== null && _spaces$spacesService !== void 0 ? _spaces$spacesService : 'default';
    try {
      const definition = await (0, _install_entity_definition.installEntityDefinition)({
        soClient,
        esClient,
        spaceId,
        logger,
        definition: req.body
      });
      await (0, _start_transform.startTransform)(esClient, definition, logger);
      return res.ok({
        body: definition
      });
    } catch (e) {
      if (e instanceof _entity_id_conflict_error.EntityIdConflict) {
        return res.conflict({
          body: e
        });
      }
      if (e instanceof _entity_security_exception.EntitySecurityException || e instanceof _invalid_transform_error.InvalidTransformError) {
        return res.customError({
          body: e,
          statusCode: 400
        });
      }
      return res.customError({
        body: e,
        statusCode: 500
      });
    }
  });
}