"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enableEntityDiscoveryRoute = enableEntityDiscoveryRoute;
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _constants_entities = require("../../../common/constants_entities");
var _auth = require("../../lib/auth");
var _built_in = require("../../lib/entities/built_in");
var _install_entity_definition = require("../../lib/entities/install_entity_definition");
var _saved_objects = require("../../saved_objects");
var _errors = require("../../../common/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function enableEntityDiscoveryRoute({
  router,
  server,
  logger
}) {
  router.put({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/managed/enablement`,
    validate: false
  }, async (context, req, res) => {
    try {
      const apiKeysEnabled = await (0, _auth.checkIfAPIKeysAreEnabled)(server);
      if (!apiKeysEnabled) {
        return res.ok({
          body: {
            success: false,
            reason: _errors.ERROR_API_KEY_SERVICE_DISABLED,
            message: 'API key service is not enabled; try configuring `xpack.security.authc.api_key.enabled` in your elasticsearch config'
          }
        });
      }
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const canEnable = await (0, _auth.canEnableEntityDiscovery)(esClient);
      if (!canEnable) {
        return res.ok({
          body: {
            success: false,
            reason: _errors.ERROR_USER_NOT_AUTHORIZED,
            message: 'Current Kibana user does not have the required permissions to enable entity discovery'
          }
        });
      }
      const soClient = (await context.core).savedObjects.getClient({
        includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
      });
      const existingApiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      if (existingApiKey !== undefined) {
        const isValid = await (0, _auth.checkIfEntityDiscoveryAPIKeyIsValid)(server, existingApiKey);
        if (!isValid) {
          await (0, _auth.deleteEntityDiscoveryAPIKey)(soClient);
          await server.security.authc.apiKeys.invalidateAsInternalUser({
            ids: [existingApiKey.id]
          });
        }
      }
      const apiKey = await (0, _auth.generateEntityDiscoveryAPIKey)(server, req);
      if (apiKey === undefined) {
        throw new Error('could not generate entity discovery API key');
      }
      await (0, _auth.saveEntityDiscoveryAPIKey)(soClient, apiKey);
      const fakeRequest = (0, _fake_kibana_request.getFakeKibanaRequest)({
        id: apiKey.id,
        api_key: apiKey.apiKey
      });
      const scopedSoClient = server.core.savedObjects.getScopedClient(fakeRequest);
      const scopedEsClient = server.core.elasticsearch.client.asScoped(fakeRequest).asCurrentUser;
      await (0, _install_entity_definition.installBuiltInEntityDefinitions)({
        logger,
        builtInDefinitions: _built_in.builtInDefinitions,
        spaceId: 'default',
        esClient: scopedEsClient,
        soClient: scopedSoClient
      });
      return res.ok({
        body: {
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      return res.customError({
        statusCode: 500,
        body: err
      });
    }
  });
}