"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.disableEntityDiscoveryRoute = disableEntityDiscoveryRoute;
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _constants_entities = require("../../../common/constants_entities");
var _auth = require("../../lib/auth");
var _errors = require("../../../common/errors");
var _uninstall_entity_definition = require("../../lib/entities/uninstall_entity_definition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function disableEntityDiscoveryRoute({
  router,
  server,
  logger
}) {
  router.delete({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/managed/enablement`,
    validate: false
  }, async (context, req, res) => {
    try {
      server.logger.debug('reading entity discovery API key from saved object');
      const apiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      if (apiKey === undefined) {
        return res.ok({
          body: {
            success: false,
            reason: _errors.ERROR_API_KEY_NOT_FOUND
          }
        });
      }
      server.logger.debug('validating existing entity discovery API key');
      const isValid = await (0, _auth.checkIfEntityDiscoveryAPIKeyIsValid)(server, apiKey);
      if (!isValid) {
        return res.ok({
          body: {
            success: false,
            reason: _errors.ERROR_API_KEY_NOT_VALID
          }
        });
      }
      const fakeRequest = (0, _fake_kibana_request.getFakeKibanaRequest)({
        id: apiKey.id,
        api_key: apiKey.apiKey
      });
      const soClient = server.core.savedObjects.getScopedClient(fakeRequest);
      const esClient = server.core.elasticsearch.client.asScoped(fakeRequest).asCurrentUser;
      await (0, _uninstall_entity_definition.uninstallBuiltInEntityDefinitions)({
        soClient,
        esClient,
        logger
      });
      await (0, _auth.deleteEntityDiscoveryAPIKey)((await context.core).savedObjects.client);
      await server.security.authc.apiKeys.invalidateAsInternalUser({
        ids: [apiKey.id]
      });
      return res.ok({
        body: {
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      return res.customError({
        statusCode: 500,
        body: err
      });
    }
  });
}