"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkEntityDiscoveryEnabledRoute = checkEntityDiscoveryEnabledRoute;
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _constants_entities = require("../../../common/constants_entities");
var _auth = require("../../lib/auth");
var _errors = require("../../../common/errors");
var _find_entity_definition = require("../../lib/entities/find_entity_definition");
var _built_in = require("../../lib/entities/built_in");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function checkEntityDiscoveryEnabledRoute({
  router,
  server,
  logger
}) {
  router.get({
    path: `${_constants_entities.ENTITY_INTERNAL_API_PREFIX}/managed/enablement`,
    validate: false
  }, async (context, req, res) => {
    try {
      logger.debug('reading entity discovery API key from saved object');
      const apiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      if (apiKey === undefined) {
        return res.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_API_KEY_NOT_FOUND
          }
        });
      }
      logger.debug('validating existing entity discovery API key');
      const isValid = await (0, _auth.checkIfEntityDiscoveryAPIKeyIsValid)(server, apiKey);
      if (!isValid) {
        return res.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_API_KEY_NOT_VALID
          }
        });
      }
      const fakeRequest = (0, _fake_kibana_request.getFakeKibanaRequest)({
        id: apiKey.id,
        api_key: apiKey.apiKey
      });
      const soClient = server.core.savedObjects.getScopedClient(fakeRequest);
      const esClient = server.core.elasticsearch.client.asScoped(fakeRequest).asCurrentUser;
      const entityDiscoveryState = await Promise.all(_built_in.builtInDefinitions.map(async builtInDefinition => {
        const definitions = await (0, _find_entity_definition.findEntityDefinitions)({
          esClient,
          soClient,
          id: builtInDefinition.id
        });
        return definitions[0];
      })).then(results => results.reduce((state, definition) => {
        return {
          installed: Boolean(state.installed && (definition === null || definition === void 0 ? void 0 : definition.state.installed)),
          running: Boolean(state.running && (definition === null || definition === void 0 ? void 0 : definition.state.running))
        };
      }, {
        installed: true,
        running: true
      }));
      if (!entityDiscoveryState.installed) {
        return res.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_PARTIAL_BUILTIN_INSTALLATION
          }
        });
      }
      if (!entityDiscoveryState.running) {
        return res.ok({
          body: {
            enabled: false,
            reason: _errors.ERROR_DEFINITION_STOPPED
          }
        });
      }
      return res.ok({
        body: {
          enabled: true
        }
      });
    } catch (err) {
      logger.error(err);
      return res.customError({
        statusCode: 500,
        body: err
      });
    }
  });
}