"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installBuiltInEntityDefinitions = installBuiltInEntityDefinitions;
exports.installEntityDefinition = installEntityDefinition;
var _create_and_install_ingest_pipeline = require("./create_and_install_ingest_pipeline");
var _create_and_install_transform = require("./create_and_install_transform");
var _validate_transform_ids = require("./transform/validate_transform_ids");
var _delete_entity_definition = require("./delete_entity_definition");
var _delete_ingest_pipeline = require("./delete_ingest_pipeline");
var _find_entity_definition = require("./find_entity_definition");
var _save_entity_definition = require("./save_entity_definition");
var _start_transform = require("./start_transform");
var _stop_and_delete_transform = require("./stop_and_delete_transform");
var _uninstall_entity_definition = require("./uninstall_entity_definition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function installEntityDefinition({
  esClient,
  soClient,
  definition,
  logger,
  spaceId
}) {
  const installState = {
    ingestPipelines: {
      history: false,
      latest: false
    },
    transforms: {
      history: false,
      latest: false
    },
    definition: false
  };
  try {
    logger.debug(`Installing definition ${JSON.stringify(definition)}`);
    (0, _validate_transform_ids.validateDefinitionCanCreateValidTransformIds)(definition);
    const entityDefinition = await (0, _save_entity_definition.saveEntityDefinition)(soClient, definition);
    installState.definition = true;

    // install ingest pipelines
    logger.debug(`Installing ingest pipelines for definition ${definition.id}`);
    await (0, _create_and_install_ingest_pipeline.createAndInstallHistoryIngestPipeline)(esClient, entityDefinition, logger, spaceId);
    installState.ingestPipelines.history = true;
    await (0, _create_and_install_ingest_pipeline.createAndInstallLatestIngestPipeline)(esClient, entityDefinition, logger, spaceId);
    installState.ingestPipelines.latest = true;

    // install transforms
    logger.debug(`Installing transforms for definition ${definition.id}`);
    await (0, _create_and_install_transform.createAndInstallHistoryTransform)(esClient, entityDefinition, logger);
    installState.transforms.history = true;
    await (0, _create_and_install_transform.createAndInstallLatestTransform)(esClient, entityDefinition, logger);
    installState.transforms.latest = true;
    return entityDefinition;
  } catch (e) {
    logger.error(`Failed to install entity definition ${definition.id}`, e);
    // Clean up anything that was successful.
    if (installState.definition) {
      await (0, _delete_entity_definition.deleteEntityDefinition)(soClient, definition, logger);
    }
    if (installState.ingestPipelines.history) {
      await (0, _delete_ingest_pipeline.deleteHistoryIngestPipeline)(esClient, definition, logger);
    }
    if (installState.ingestPipelines.latest) {
      await (0, _delete_ingest_pipeline.deleteLatestIngestPipeline)(esClient, definition, logger);
    }
    if (installState.transforms.history) {
      await (0, _stop_and_delete_transform.stopAndDeleteHistoryTransform)(esClient, definition, logger);
    }
    if (installState.transforms.latest) {
      await (0, _stop_and_delete_transform.stopAndDeleteLatestTransform)(esClient, definition, logger);
    }
    throw e;
  }
}
async function installBuiltInEntityDefinitions({
  esClient,
  soClient,
  logger,
  builtInDefinitions,
  spaceId
}) {
  if (builtInDefinitions.length === 0) return [];
  logger.debug(`Starting installation of ${builtInDefinitions.length} built-in definitions`);
  const installPromises = builtInDefinitions.map(async builtInDefinition => {
    const definitions = await (0, _find_entity_definition.findEntityDefinitions)({
      esClient,
      soClient,
      id: builtInDefinition.id
    });
    if (definitions.length === 0) {
      return await installAndStartDefinition({
        definition: builtInDefinition,
        esClient,
        soClient,
        logger,
        spaceId
      });
    }
    const definition = definitions[0];
    // verify current installation
    if (!definition.state.installed) {
      logger.debug(`Detected partial installation of definition [${definition.id}], reinstalling`);
      await (0, _uninstall_entity_definition.uninstallEntityDefinition)({
        esClient,
        soClient,
        logger,
        definition
      });
      return await installAndStartDefinition({
        definition: builtInDefinition,
        esClient,
        soClient,
        logger,
        spaceId
      });
    }
    if (!definition.state.running) {
      logger.debug(`Starting transforms for definition [${definition.id}]`);
      await (0, _start_transform.startTransform)(esClient, definition, logger);
    }
    return definition;
  });
  return await Promise.all(installPromises);
}
async function installAndStartDefinition(params) {
  const definition = await installEntityDefinition(params);
  await (0, _start_transform.startTransform)(params.esClient, definition, params.logger);
  return definition;
}