"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateEntityDiscoveryAPIKey = exports.checkIfEntityDiscoveryAPIKeyIsValid = exports.checkIfAPIKeysAreEnabled = void 0;
var _fake_kibana_request = require("@kbn/security-plugin/server/authentication/api_keys/fake_kibana_request");
var _privileges = require("../privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const checkIfAPIKeysAreEnabled = async server => {
  return await server.security.authc.apiKeys.areAPIKeysEnabled();
};
exports.checkIfAPIKeysAreEnabled = checkIfAPIKeysAreEnabled;
const checkIfEntityDiscoveryAPIKeyIsValid = async (server, apiKey) => {
  server.logger.debug('validating API key against authentication service');
  const isValid = await server.security.authc.apiKeys.validate({
    id: apiKey.id,
    api_key: apiKey.apiKey
  });
  if (!isValid) return false;

  // this fake kibana request is how you get an API key-scoped client...
  const esClient = server.core.elasticsearch.client.asScoped((0, _fake_kibana_request.getFakeKibanaRequest)({
    id: apiKey.id,
    api_key: apiKey.apiKey
  })).asCurrentUser;
  server.logger.debug('validating API key has runtime privileges for entity discovery');
  return (0, _privileges.canRunEntityDiscovery)(esClient);
};
exports.checkIfEntityDiscoveryAPIKeyIsValid = checkIfEntityDiscoveryAPIKeyIsValid;
const generateEntityDiscoveryAPIKey = async (server, req) => {
  const apiKey = await server.security.authc.apiKeys.grantAsInternalUser(req, {
    name: 'Entity discovery API key',
    role_descriptors: {
      entity_discovery_admin: _privileges.requiredRunTimePrivileges
    },
    metadata: {
      description: 'API key used to manage the transforms and ingest pipelines created by the entity discovery framework'
    }
  });
  if (apiKey !== null) {
    return {
      id: apiKey.id,
      name: apiKey.name,
      apiKey: apiKey.api_key
    };
  }
};
exports.generateEntityDiscoveryAPIKey = generateEntityDiscoveryAPIKey;