"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SchemaLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _engine = require("../engine");
var _constants = require("./constants");
var _schema_base_logic = require("./schema_base_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SchemaLogic = exports.SchemaLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'schema_logic'],
  connect: {
    values: [_schema_base_logic.SchemaBaseLogic, ['dataLoading', 'schema']],
    actions: [_schema_base_logic.SchemaBaseLogic, ['loadSchema', 'onSchemaLoad', 'setSchema']]
  },
  actions: {
    addSchemaField: (fieldName, fieldType) => ({
      fieldName,
      fieldType
    }),
    updateSchemaFieldType: (fieldName, fieldType) => ({
      fieldName,
      fieldType
    }),
    updateSchema: successMessage => successMessage,
    onSchemaUpdateError: true,
    openModal: true,
    closeModal: true
  },
  reducers: {
    isUpdating: [false, {
      updateSchema: () => true,
      onSchemaLoad: () => false,
      onSchemaUpdateError: () => false
    }],
    cachedSchema: [{}, {
      onSchemaLoad: (_, {
        schema
      }) => schema
    }],
    mostRecentIndexJob: [{}, {
      onSchemaLoad: (_, {
        mostRecentIndexJob
      }) => mostRecentIndexJob
    }],
    unconfirmedFields: [[], {
      onSchemaLoad: (_, {
        unconfirmedFields
      }) => unconfirmedFields
    }],
    incompleteFields: [[], {
      onSchemaLoad: (_, {
        incompleteFields
      }) => incompleteFields
    }],
    hasNewUnsearchedFields: [false, {
      onSchemaLoad: (_, {
        unsearchedUnconfirmedFields
      }) => unsearchedUnconfirmedFields
    }],
    isModalOpen: [false, {
      openModal: () => true,
      closeModal: () => false,
      onSchemaLoad: () => false,
      onSchemaUpdateError: () => false
    }]
  },
  selectors: {
    hasSchema: [selectors => [selectors.cachedSchema], cachedSchema => Object.keys(cachedSchema).length > 0],
    hasSchemaChanged: [selectors => [selectors.schema, selectors.cachedSchema], (schema, cachedSchema) => !(0, _lodash.isEqual)(schema, cachedSchema)],
    hasUnconfirmedFields: [selectors => [selectors.unconfirmedFields], unconfirmedFields => unconfirmedFields.length > 0],
    hasIncompleteFields: [selectors => [selectors.incompleteFields], incompleteFields => incompleteFields.length > 0]
  },
  listeners: ({
    actions,
    values
  }) => ({
    addSchemaField: ({
      fieldName,
      fieldType
    }) => {
      if (values.schema.hasOwnProperty(fieldName)) {
        (0, _flash_messages.setErrorMessage)((0, _constants.ADD_SCHEMA_ERROR)(fieldName));
        actions.closeModal();
      } else {
        const updatedSchema = {
          ...values.schema
        };
        updatedSchema[fieldName] = fieldType;
        actions.setSchema(updatedSchema);
        actions.updateSchema((0, _constants.ADD_SCHEMA_SUCCESS)(fieldName));
      }
    },
    updateSchemaFieldType: ({
      fieldName,
      fieldType
    }) => {
      const updatedSchema = {
        ...values.schema
      };
      updatedSchema[fieldName] = fieldType;
      actions.setSchema(updatedSchema);
    },
    updateSchema: async successMessage => {
      const {
        schema
      } = values;
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      try {
        const response = await http.post(`/internal/app_search/engines/${engineName}/schema`, {
          body: JSON.stringify(schema)
        });
        actions.onSchemaLoad(response);
        (0, _flash_messages.flashSuccessToast)(successMessage || _constants.UPDATE_SCHEMA_SUCCESS);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
        actions.onSchemaUpdateError();
        // Restore updated schema back to server/cached schema, so we don't keep
        // erroneous or bad fields in-state
        actions.setSchema(values.cachedSchema);
      } finally {
        // Re-fetch engine data so that other views also dynamically update
        // (e.g. Documents results, nav icons for invalid boosts or unconfirmed flags)
        _engine.EngineLogic.actions.initializeEngine();
      }
    }
  })
});