"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrawlerDomainsLogic = void 0;
var _kea = require("kea");
var _constants = require("../../../shared/constants");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _table_pagination = require("../../../shared/table_pagination");
var _engine = require("../engine");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CrawlerDomainsLogic = exports.CrawlerDomainsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'crawler', 'crawler_domains_logic'],
  actions: {
    deleteDomain: domain => ({
      domain
    }),
    fetchCrawlerDomainsData: true,
    onReceiveData: (domains, meta) => ({
      domains,
      meta
    }),
    onPaginate: newPageIndex => ({
      newPageIndex
    }),
    crawlerDomainDeleted: data => ({
      data
    })
  },
  reducers: {
    dataLoading: [true, {
      onReceiveData: () => false,
      onPaginate: () => true
    }],
    domains: [[], {
      onReceiveData: (_, {
        domains
      }) => domains
    }],
    meta: [_constants.DEFAULT_META, {
      onReceiveData: (_, {
        meta
      }) => meta,
      onPaginate: (state, {
        newPageIndex
      }) => (0, _table_pagination.updateMetaPageIndex)(state, newPageIndex)
    }]
  },
  listeners: ({
    actions,
    values
  }) => ({
    fetchCrawlerDomainsData: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const {
        meta
      } = values;
      const query = {
        'page[current]': meta.page.current,
        'page[size]': meta.page.size
      };
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/crawler/domains`, {
          query
        });
        const domains = response.results.map(_utils.crawlerDomainServerToClient);
        actions.onReceiveData(domains, response.meta);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteDomain: async ({
      domain
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.delete(`/internal/app_search/engines/${engineName}/crawler/domains/${domain.id}`, {
          query: {
            respond_with: 'crawler_details'
          }
        });
        const crawlerData = (0, _utils.crawlerDataServerToClient)(response);
        // Publish for other logic files to listen for
        actions.crawlerDomainDeleted(crawlerData);
        actions.fetchCrawlerDomainsData();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});