"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createConversationRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _utils = require("../utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createConversationRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_CONVERSATIONS_URL,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ConversationCreateProps)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      // Perform license and authenticated user checks
      const checkResponse = (0, _helpers.performChecks)({
        authenticatedUser: true,
        context: ctx,
        license: true,
        request,
        response
      });
      if (checkResponse) {
        return checkResponse;
      }
      const dataClient = await ctx.elasticAssistant.getAIAssistantConversationsDataClient();
      const currentUser = ctx.elasticAssistant.getCurrentUser();
      const userFilter = currentUser !== null && currentUser !== void 0 && currentUser.username ? `name: "${currentUser === null || currentUser === void 0 ? void 0 : currentUser.username}"` : `id: "${currentUser === null || currentUser === void 0 ? void 0 : currentUser.profile_uid}"`;
      const result = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findDocuments({
        perPage: 100,
        page: 1,
        filter: `users:{ ${userFilter} } AND title:${request.body.title}`,
        fields: ['title']
      }));
      if ((result === null || result === void 0 ? void 0 : result.data) != null && result.total > 0) {
        return assistantResponse.error({
          statusCode: 409,
          body: `conversation title: "${request.body.title}" already exists`
        });
      }
      const createdConversation = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createConversation({
        conversation: request.body
      }));
      if (createdConversation == null) {
        return assistantResponse.error({
          body: `conversation with title: "${request.body.title}" was not created`,
          statusCode: 400
        });
      }
      return response.ok({
        body: _elasticAssistantCommon.ConversationResponse.parse(createdConversation)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createConversationRoute = createConversationRoute;