"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postActionsConnectorExecuteRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _server = require("@kbn/data-plugin/server");
var _configSchema = require("@kbn/config-schema");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _event_based_telemetry = require("../lib/telemetry/event_based_telemetry");
var _constants = require("../../common/constants");
var _build_response = require("../lib/build_response");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postActionsConnectorExecuteRoute = (router, getElser) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.POST_ACTIONS_CONNECTOR_EXECUTE,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ExecuteConnectorRequestBody),
        params: _configSchema.schema.object({
          connectorId: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, response) => {
    const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const assistantContext = ctx.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    let onLlmResponse;
    try {
      var _ref;
      const authenticatedUser = assistantContext.getCurrentUser();
      if (authenticatedUser == null) {
        return response.unauthorized({
          body: `Authenticated user not found`
        });
      }
      let latestReplacements = request.body.replacements;
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      let messages;
      let newMessage;
      const conversationId = request.body.conversationId;
      const actionTypeId = request.body.actionTypeId;
      const connectorId = decodeURIComponent(request.params.connectorId);

      // if message is undefined, it means the user is regenerating a message from the stored conversation
      if (request.body.message) {
        newMessage = {
          content: request.body.message,
          role: 'user'
        };
      }

      // get the actions plugin start contract from the request context:
      const actions = ctx.elasticAssistant.actions;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const conversationsDataClient = await assistantContext.getAIAssistantConversationsDataClient();
      onLlmResponse = async (content, traceData = {}, isError = false) => {
        if (conversationsDataClient && conversationId) {
          await (0, _helpers.appendAssistantMessageToConversation)({
            conversationId,
            conversationsDataClient,
            messageContent: content,
            replacements: latestReplacements,
            isError,
            traceData
          });
        }
      };
      return await (0, _helpers.langChainExecute)({
        abortSignal,
        isStream: request.body.subAction !== 'invokeAI',
        actionsClient,
        actionTypeId,
        connectorId,
        conversationId,
        context: ctx,
        getElser,
        logger,
        messages: (_ref = newMessage ? [newMessage] : messages) !== null && _ref !== void 0 ? _ref : [],
        onLlmResponse,
        onNewReplacements,
        replacements: latestReplacements,
        request,
        response,
        telemetry
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (onLlmResponse) {
        await onLlmResponse(error.message, {}, true);
      }
      telemetry.reportEvent(_event_based_telemetry.INVOKE_ASSISTANT_ERROR_EVENT.eventType, {
        actionTypeId: request.body.actionTypeId,
        model: request.body.model,
        errorMessage: error.message,
        assistantStreamingEnabled: request.body.subAction !== 'invokeAI'
      });
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.postActionsConnectorExecuteRoute = postActionsConnectorExecuteRoute;