"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKnowledgeBaseStatusRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _get_kb_resource = require("./get_kb_resource");
var _build_response = require("../../lib/build_response");
var _elasticsearch_store = require("../../lib/langchain/elasticsearch_store/elasticsearch_store");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the status of the Knowledge Base index, pipeline, and resources (collection of documents)
 *
 * @param router IRouter for registering routes
 * @param getElser Function to get the default Elser ID
 */
const getKnowledgeBaseStatusRoute = (router, getElser) => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_URL,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_INTERNAL_API_VERSION,
    validate: {
      request: {
        params: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ReadKnowledgeBaseRequestParams)
      }
    }
  }, async (context, request, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    try {
      // Use asInternalUser
      const esClient = (await context.core).elasticsearch.client.asInternalUser;
      const elserId = await getElser();
      const kbResource = (0, _get_kb_resource.getKbResource)(request);
      const kbDataClient = await assistantContext.getAIAssistantKnowledgeBaseDataClient();
      if (!kbDataClient) {
        return response.custom({
          body: {
            success: false
          },
          statusCode: 500
        });
      }

      // Use old status checks by overriding esStore to use kbDataClient
      const esStore = new _elasticsearch_store.ElasticsearchStore(esClient, kbDataClient.indexTemplateAndPattern.alias, logger, telemetry, elserId, kbResource, kbDataClient);
      const indexExists = await esStore.indexExists();
      const pipelineExists = await esStore.pipelineExists();
      const modelExists = await esStore.isModelInstalled(elserId);
      const setupAvailable = await kbDataClient.isSetupAvailable();
      const body = {
        elser_exists: modelExists,
        index_exists: indexExists,
        is_setup_in_progress: kbDataClient.isSetupInProgress,
        is_setup_available: setupAvailable,
        pipeline_exists: pipelineExists
      };
      if (kbResource === _constants.ESQL_RESOURCE) {
        const esqlExists = indexExists && (await esStore.similaritySearch(_constants.ESQL_DOCS_LOADED_QUERY)).length > 0;
        return response.ok({
          body: {
            ...body,
            esql_exists: esqlExists
          }
        });
      }
      return response.ok({
        body
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getKnowledgeBaseStatusRoute = getKnowledgeBaseStatusRoute;