"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findKnowledgeBaseEntriesRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _utils = require("../../utils");
var _helpers = require("../../helpers");
var _transforms = require("../../../ai_assistant_data_clients/knowledge_base/transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findKnowledgeBaseEntriesRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_ENTRIES_URL_FIND,
    options: {
      tags: ['access:elasticAssistant']
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        query: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.FindKnowledgeBaseEntriesRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    try {
      const {
        query
      } = request;
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);

      // Perform license, authenticated user and FF checks
      const checkResponse = (0, _helpers.performChecks)({
        authenticatedUser: true,
        capability: 'assistantKnowledgeBaseByDefault',
        context: ctx,
        license: true,
        request,
        response
      });
      if (checkResponse) {
        return checkResponse;
      }
      const kbDataClient = await ctx.elasticAssistant.getAIAssistantKnowledgeBaseDataClient();
      const currentUser = ctx.elasticAssistant.getCurrentUser();
      const additionalFilter = query.filter ? ` AND ${query.filter}` : '';
      const result = await (kbDataClient === null || kbDataClient === void 0 ? void 0 : kbDataClient.findDocuments({
        perPage: query.per_page,
        page: query.page,
        sortField: query.sort_field,
        sortOrder: query.sort_order,
        filter: `users:{ id: "${currentUser === null || currentUser === void 0 ? void 0 : currentUser.profile_uid}" }${additionalFilter}`,
        // TODO: Update filter to include non-user system entries
        fields: query.fields
      }));
      if (result) {
        return response.ok({
          body: {
            perPage: result.perPage,
            page: result.page,
            total: result.total,
            data: (0, _transforms.transformESSearchToKnowledgeBaseEntry)(result.data)
          }
        });
      }
      return response.ok({
        body: {
          perPage: query.per_page,
          page: query.page,
          data: [],
          total: 0
        }
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.findKnowledgeBaseEntriesRoute = findKnowledgeBaseEntriesRoute;